/******************************************************************************
*
*    Initialize the partition
*    Copyright (C) 1993 A. Bode, S. Lamberts, T. Ludwig, G. Stellner
*
*    This file is part of NXLIB (Paragon(TM) message passing on workstations)
*
*    NXLIB is free software; you can redistribute it and/or
*    modify it under the terms of the GNU Library General Public
*    License as published by the Free Software Foundation; either
*    version 2 of the License, or (at your option) any later version.
*
*    NXLIB is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*    Library General Public License for more details.
*
*    You should have received a copy of the GNU Library General Public
*    License along with this library; if not, write to the Free
*    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*    Contact to the authors:
*
*    electronic mail: nxlib@informatik.tu-muenchen.de
*
*    paper mail:      Prof. Dr. A. Bode
*                     Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
*                     Institut f"ur Informatik
*                     Technische Universit"at M"unchen
*                     80290 M"unchen
*                     Germany
*
*    Paragon(TM) is a trademark of Intel Corporation.
*
******************************************************************************/
/******************************************************************************
  
  nxinit.c,v
  1995/02/06 16:10:35
  1.16
  Exp
  lamberts
  
  Authors: Stefan Lamberts, Susanna Schink
  
  Description:
  This file contains the nx_initve() call and procedures used by nx_initve().
  Additionally procedures are provides which are called by the SIGIO handler.
  Theese procedures need to excange information with nx_initve(), i.e.
  Information about the reception of messages.

  Available functions from this module:

  long nx_initve(partiton,size,account,argc,argv)
  char *partition;
  long size;
  char *account;
  long *argc;
  char *argv[];

  The procedure which receives DC_ADDNXD messages.
  void _dc_addnxd(msg)
  msg_desc *msg;
  
******************************************************************************/
#ifndef lint
static char rcs_id[] = "nxinit.c,v 1.16 1995/02/06 16:10:35 lamberts Exp";
#endif

#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef SUN_OS4
#include <memory.h>             /* Should be included in string.h */
#endif

#ifdef HP_UX
#include <malloc.h>             /* For mallopt() */
#endif

#include "../include/config.h"
#include "../include/sys/nxerrno.h"
#include "../include/sys/nxlib.h"
#include "../include/nxmalloc.h"

extern long      __mynode();
extern long      __myptype();
extern long      __numnodes();
extern void      __setnumnodes();
extern void      __setnode();
extern long      __setptype();
extern void      _dbgmsg();
extern p_listel *_find_plistel();
extern int       _init_c_buf();
extern int       _init_i_buf();
extern int       _init_m_buf();
extern int       _ins_plistel();
extern int       _read_part();
extern int       _rel_c_setup();
extern void      _rel_disable();
extern void      _rel_enable();
extern int       _rel_send();
extern void      _rel_wait_msg();
extern int       _set_addr_plistel();
extern void      _sig_tstp();
extern char     *ltoa();
extern void      nx_perror();

#ifdef XPARAGON
extern int _xp_setup();
#endif

extern int __tcpbufsize;

static long expected_msgs = 0;
static int nx_init_error = 0;

/*****************************************************************************
 * Wait for a message as long as this condition is true
 *****************************************************************************/
static int all_msgs
#ifdef ANSI
(void)
#else
()
#endif
/*****************************************************************************/
{
  return (expected_msgs > 0);
}


/**************************************************************************
 * handler for SIGCHLD signals
 *************************************************************************/
static void sigchild_handler
#ifdef ANSI
(void)
#else
()
#endif
{
  int stat;
  pid_t pid;
  int loc_errno = errno;
  
  while ((pid = waitpid(-1,&stat,WNOHANG|WUNTRACED)) != 0)
  {
    if ((pid < (pid_t)0) && (errno == ECHILD))
      break;

    if (pid < (pid_t)0)
    {
      nx_perror("sigchild_handler()");
      exit(-1);
    }
    
    errno = 0;
    _dbgmsg("nxdaemon died before initialisation");
    expected_msgs--;
    nx_init_error++;
  }

  errno = loc_errno;
  return;
}




/**************************************************************************
 * Store address information about daemon addresses
 * This is executed only by the controlling process
 *************************************************************************/
int _dc_addnxd
#ifdef ANSI
(msg_desc *msgd)
#else
(msgd)
msg_desc *msgd;
#endif
/**************************************************************************/
{
  msg_dc_addnxd  *msg;
  
  msg = (msg_dc_addnxd *)msgd->msg_ptr;
  
  if (_set_addr_plistel(msg->node,
                        (long)DAEMON_PTYPE,
                        LOC_UNDEF,
                        &(msg->radr),
                        (loc_addr *)0,
                        (char *)0,
                        (char *)0,
                        (char *)0,
                        UNDEF_PID) < 0)
  {
    nx_init_error++;
    expected_msgs--;
    return (-1);
  }
  expected_msgs--;
  return (0);
}



/**************************************************************************
 * Remove initialisation arguments
 *************************************************************************/
static void
#ifdef ANSI
rm_arg(int *argc,char **argv, int argi, int num)
#else
rm_arg(argc, argv, argi, num)
long *argc;
char **argv;
int argi;
int num;
#endif
/*************************************************************************/
/* num MUST be greater than *argc:  (num > *argc) */
{
  int i;
  
  /* cont the number of arguments before arg */
  for (i = 0; i < *argc; i++) {
    if (i >= (argi + num)) {
      argv[i-num] = argv[i];
    }
  }
  *argc -= num;
}



/*************************************************************************/
static int
#ifdef ANSI
nxinit_args(long *argc, char **argv, char **partition, long *size, int *tcpbuf)
#else
nxinit_args(argc, argv, partition, size, tcpbuf)
long *argc;
char **argv;
char **partition;
long *size;
int  *tcpbuf;
#endif /* ANSI */
/*************************************************************************/
{
  int argi;      /* argument index */
  
  for (argi = *argc - 1; argi > 0; argi--) {
    /* parse the command line in reverse order */
    
    if (!strcmp(argv[argi],"-pn")) { /* partition */
      if (argi == (*argc - 1)) {
        /* the required argument is missing */
        errno = EPINVALPART;
        return -1;
      }
      *partition = argv[argi+1]; /* there is no need to copy the string */
      rm_arg(argc,argv,argi,2);
    }
    
    else if (!strcmp(argv[argi],"-sz")) { /* size */
      if (argi == (*argc - 1)) {
        /* the required argument is missing */
        errno = EQESIZE;
        return (-1);
      }
      if ((*size = atol(argv[argi+1])) <= 0) {
        /* bogous size specification */
        errno = EQESIZE;
        return (-1);
      }
      rm_arg(argc,argv,argi,2);
    }
    
    if (!strcmp(argv[argi],"-tcpbuf"))
    {
      /* TCP buffer size */
      if (argi == (*argc - 1)) {
        /* the required argument is missing */
        errno = ENXLTCPBUF;
        return -1;
      }
      if ((*tcpbuf = atoi(argv[argi+1])) <= 0) 
      {
        /* bogous TCP buffer size specification */
        errno = ENXLTCPBUF;
        return(-1);
      }
      rm_arg(argc,argv,argi,2);
    }
    
    /* unsupported options which require one argument */
    else if ((!strcmp(argv[argi],"-gth")) || /* give_threshold */
             (!strcmp(argv[argi],"-mbf")) || /* memory_buffer */
             (!strcmp(argv[argi],"-mea")) || /* memory_each */
             (!strcmp(argv[argi],"-mex")) || /* memory_export */
             (!strcmp(argv[argi],"-pkt")) || /* packet_size */
             (!strcmp(argv[argi],"-pri")) || /* priority */
             (!strcmp(argv[argi],"-sct")) || /* send_count */
             (!strcmp(argv[argi],"-sth"))) {  /* send_threshold */
      if (argi == (*argc - 1)) {
        /* the option would require an argument
         * since the option is not supported
         * the error will be discarded */
        rm_arg(argc,argv,argi,1);
      }
      else {
        /* discard this option */
        rm_arg(argc,argv,argi,2);
      }
    }
    
    /* unsupported option without argument */
    else if (!strcmp(argv[argi],"-plk")) {
      rm_arg(argc,argv,argi,1);
    }
  }
  return 0;
}


/*************************************************************************
 ************************************************************************/
static int start_daemons
#ifdef ANSI
(long size,
 vpn_spec *vpns)
#else
(size, vpns)
long size;
vpn_spec *vpns;
#endif
/************************************************************************/
{
  char *port_str;
  char *size_str;
  char *node_str;
  char *cppid_str;
  char *nxld;
  char *nxdpath;
  char *tcpbuf_str;
  
  long i;
  pid_t pid;
  
  vpn_spec *ptr;
  p_listel *pe;
  

  if ((pe = _find_plistel(__mynode(),__myptype())) == (p_listel *)0)
    return (-1);
  
  if ((port_str = ltoa((long)pe->radr.sad.sin_port)) == (char *)0)
    return (-1);
  if ((size_str = ltoa(size)) == (char *)0)
    return (-1);
  if ((cppid_str = ltoa((long)getpid())) == (char *)0)
    return (-1);
  if ((tcpbuf_str = ltoa((long)__tcpbufsize)) == (char *)0)
    return (-1);

  if ((nxld = getenv(NXLIBPATH)) != (char *)0)
  {
    /* Environment variable NXLIBPATH is set; use the nxdaemon
       binary in that "bin" subdirectory */
    if ((nxdpath = malloc((size_t)(strlen(nxld)+
                                   strlen("/bin/")+
                                   strlen(NXDAEMONNAME)+
                                   1))) == (char *)0)
      return (-1);
    
    strcpy(nxdpath,nxld);
    strcat(nxdpath,"/bin/");
    strcat(nxdpath,NXDAEMONNAME);
  }
  else
  {
    /* Use the nxdaemon binary in the users PATH */
    nxdpath = NXDAEMONNAME;
  }
      
  for (ptr = vpns, i = 0;
       (ptr != NULL) && (i < size);
       ptr = ptr->next, i++)
  {

    if ((pid = fork()) == -1) 
      return (-1);
    else if (pid == (pid_t)0) 
    {                            /* Child process */
      if ((node_str = ltoa(ptr->node)) == (char *)0) 
      {
        nx_perror("nx_initve(): child");
        _dbgmsg("Could not start nxdaemon on %s",pe->hname);
        exit(-1);
      }
  
      if (strcmp(ptr->hname,pe->hname) == 0) 
      {
        /* VPN is on local machine; Start nxdaemon with exec..() */
        execlp(nxdpath,
               NXDAEMONNAME,
               node_str,pe->hname,port_str,size_str,cppid_str,ptr->path,
               tcpbuf_str,
               (char *)0);
        nx_perror("nx_initve(): child");
        _dbgmsg("Could not start nxdaemon on %s",pe->hname);
        exit(-1);
      }
      if (ptr->login[0] == '\0')
      {
        /* Use the same login on the remote machine */
        execl(RSHPATH,
              RSHNAME,
              ptr->hname,
              RSHOPTION,
              nxdpath,
              node_str,pe->hname,port_str,size_str,cppid_str,ptr->path,
               tcpbuf_str,
              (char *)0);
        nx_perror("nx_initve(): child");
        _dbgmsg("Could not start nxdaemon on %s",ptr->hname);
        exit(-1);
      }

      /* Use different login on the remote machine */
      execl(RSHPATH,
            RSHNAME,
            ptr->hname,
            "-l",ptr->login,
            RSHOPTION,
            nxdpath,
            node_str,pe->hname,port_str,size_str,cppid_str,ptr->path,
            tcpbuf_str,
            (char *)0);
      nx_perror("nx_initve(): child");
      _dbgmsg("Could not start nxdaemon on %s",ptr->hname);
      exit(-1);
    }
    else 
    {                            /* Parent process */
      /**
       ** Enter all daemons in the address database
       **/
      if (_ins_plistel(ptr->node,
                       (long)DAEMON_PTYPE,
                       LOC_REMOTE,
                       (rem_addr *)0,
                       (loc_addr *)0,
                       ptr->hname,
                       ptr->login,
                       ptr->arch,
                       UNDEF_PID,
                       (long)UNDEF_NODE,
                       (long)UNDEF_PTYPE) < 0)
        return (-1);
      /* LAMBO: Probably some kind of message */
    }
    
  } /* end for */
  
  free((void *)size_str);
  free((void *)port_str);
  free((void *)cppid_str);
  free((void *)node_str);
  free((void *)tcpbuf_str);
  if (nxld != (char *)0)
    free((void *)nxdpath);
  
  return (0);
}




  



/***************************************************************************
 * forward the list with informations about all daemon processes to all
 * daemon processes of the parallel application
 **************************************************************************/
static int
#ifdef ANSI
init_daemons(long size)
#else
init_daemons(size)
long size;
#endif
/**************************************************************************/
{
  p_listel  *p;
  long i;
  
  msg_cd_initd *msg;

  _rel_disable();

  /**
   ** Compose message
   **/
  if ((msg = (msg_cd_initd *)malloc((size_t)(sizeof(msg_cd_initd)*size))) ==
      (msg_cd_initd *)0)
  {
    _rel_enable();
    return (-1);
  }

  for (i = 0; i < size; i++)
  {
    if ((p = _find_plistel(i,(long)DAEMON_PTYPE)) == (p_listel *)0)
    {
      _rel_enable();
      return (-1);
    }

    msg[i].node = i;
    strcpy(msg[i].hname,p->hname);
    strcpy(msg[i].login,p->login);
    strcpy(msg[i].arch,p->arch);
    msg[i].radr.sockfd = UNDEF_SOCK;
    msg[i].radr.stat = p->radr.stat;
    memcpy((void *)&msg[i].radr.sad,
           (void *)&p->radr.sad,
           sizeof(struct sockaddr_in));
    msg[i].ladr = 0;
  }
  
  /**
   ** Send the message to all daemons
   **/

  for (i = 0; i < size; i++)
    if (_rel_send((long)CD_INITD,
                  (char *)msg,
                  (long)(sizeof(msg_cd_initd) * size),
                  i,
                  (long)DAEMON_PTYPE) < 0)
    {
      _rel_enable();
      return (-1);
    }

  free((void *)msg);
  
  _rel_enable();

  return (0);
}

/****************************************************************************/
long
#ifdef ANSI
nx_initve(char *partition, long size, char *account, long *argc, char **argv)
#else
nx_initve(partition, size, account, argc, argv)
char *partition;
long size;
char *account;
long *argc;
char **argv;
#endif
/****************************************************************************/
{
  static int allready_called = 0; 
  long thesize = 0;
  char *thepart = (char *)0;
  long partsize;
  char *siz;
  vpn_spec *vpnspec;
  

#ifdef BSD_SIG
  struct sigvec vec, ovec;
#else
  struct sigaction sigact, osigact;
  sigset_t chld_sigset;
#endif

  /**
   ** Check wether this application did allready a nx_initve() call
   **/
  if (allready_called) {
    errno = EAEXIST;
    return (-1);
  }
  else {
    allready_called = 1;
  }
  
  /**
   ** If the ptype was already set nx_initve() was already called
   **/
  if (__myptype() != -1)
  {
    errno = EAEXIST;
    return (-1);
  }
  
  /**
   ** Become process group leader
   **/
  if (setpgid((pid_t)0,(pid_t)0) < 0)
    return (-1);
    

  /**
   ** If size if less than 0 return error
   **/
  if (size < 0) {
    errno = EQESIZE;
    return (-1);
  }
  
  /**
   ** Read command line switches
   **/
  if (argc != 0) {    /* LAMBO: denk an die Fortran Semantik!! */
    if (nxinit_args(argc, argv, &thepart, &thesize, &__tcpbufsize) < 0) {
      return (-1);
    }
  }
  
  /**
   ** Set the partition variable
   **/
  if (thepart == (char *)0)
  {
    if ((partition == (char *)0) ||
        (partition[0] == '\0') ||
        (strcmp(partition," ") == 0))
    {
      if ((thepart = getenv(NX_DFLT_PART)) == (char *)0)
      {
        thepart = DEFAULT_PART_NAME;
      }
    }
    else
    {
      thepart = partition;
    }
  }
  
  /**
   ** Read the partition file
   **/
  if ((partsize = _read_part(thepart,&vpnspec)) <= 0) {
    return (-1);
  }
  
  
  /**
   ** Set the size variable
   **/
  if (thesize == 0)
  {                              /* -sz switch not set */
    if (size == 0)
    {                            /* input paramenter size == 0 */
      if ((siz = getenv(NX_DFLT_SIZE)) == (char *)0)
        thesize = partsize;
      else
      {
        thesize = atol(siz);
        if (thesize <= 0)
        {
          errno = EQESIZE;
          return (-1);
        }
      }
    }
    else
      thesize = size;
  }
  if (thesize > partsize)
  {
    errno = EPXRS;
    return (-1);
  }
  
  /***
   *** now we can start with the work since all input parameters are checked 
   ***/

  /* Block Signals during memmory allocating calls */
#ifdef HP_UX
  if (mallopt(M_BLOCK,1) != 0)
    return (-1);
#endif

  /**
   ** Set the internal ptype varible to the one of the controlling
   ** process.
   ** Actually, a process becomes controlling process by calling
   ** nx_initve()
   **/
  if (__setptype((long)CP_PTYPE) < 0)
    return (-1);
  
  /**
   ** Set the number of nodes
   **/
  __setnumnodes(thesize);

  /**
   ** Set the node number of this node
   **/
  __setnode(thesize);

  /* Initialize buffer layer */
  if (_init_m_buf() < 0)
    return (-1);

  if (_init_c_buf() < 0)
    return (-1);

  if (_init_i_buf() < 0)
    return (-1);

  /**
   ** Initialize and setup the connection to XPARAGON
   **/
#ifdef XPARAGON
  if (_xp_setup() < 0)
    _dbgmsg("Warning: Could not connect to XPARAGON");
#endif    
  
  /**
   ** Initialize the communication facilities and add the information about the
   ** daemons to the address database
   ** daemon processes*/
  if (_rel_c_setup() < 0)
    return (-1);

  /* Block SIGCHLD now done in _re_disable */
  
  _rel_disable();

  /**
   ** Setup SIGCHLD handler
   ** If one of the forked child processes terminates before the
   ** address information is received form all daemons there was an
   ** error.
   **/
#ifdef BSD_SIG
  vec.sv_handler = sigchild_handler;
  vec.sv_mask = sigmask(SIGIO) |
    sigmask(SIGCHLD) |
      sigmask(SIGTSTP) |
        sigmask(SIGCONT);
  vec.sv_flags = 0;
  
  if (sigvec(SIGCHLD, &vec, &ovec) != 0)
    return (-1);
#else
  if ((sigemptyset(&sigact.sa_mask) != 0) ||
      (sigaddset(&sigact.sa_mask,SIGIO) != 0) || /* Block incomming messages,
                                                    see _rel_disable() */
      (sigaddset(&sigact.sa_mask,SIGCHLD) != 0) ||
      (sigaddset(&sigact.sa_mask,SIGTSTP) != 0) ||
      (sigaddset(&sigact.sa_mask,SIGCONT) != 0))
    return (-1);
  
  sigact.sa_handler = sigchild_handler;
  sigact.sa_flags = SA_NOCLDSTOP;

  if(sigaction(SIGCHLD, &sigact, &osigact) < 0)
    return(-1);
#endif

  /* Ignore SIGTTIN wich is delivered by the rsh call */
  /*  signal(SIGTTIN,SIG_IGN); */
  /* Using the -n option for rsh this should be no longer necessary */
  
  
  /**
   ** Start the nxdaemons
   **/
  if (start_daemons(thesize,vpnspec) < 0)
    return (-1);

  expected_msgs = thesize;

  _rel_enable();

  /* Allow SIGCHLD signals now done in _rel_enable */
  
  /**
   ** Pause until the address information is receivde from all daemons or
   ** the fork()ed process terminated.
   **/
  _rel_wait_msg(all_msgs);

  /***
   *** Reset SIGCHLD handler
   ***/
#ifdef BSD_SIG
  if (sigvec(SIGCHLD,&ovec, (struct sigvec *)0) != 0)
    return (-1);
#else
  if (sigaction(SIGCHLD, &osigact, (struct sigaction *)0) != 0)
    return(-1);
#endif

  if (nx_init_error > 0)
  {
    errno = EPALLOCERR;
    return (-1);
  }
  
  
  /**
   ** Send address information to the daemons
   **/
  if (init_daemons(thesize) == -1) {
    return (-1);
  }

  /**
   ** Forward the SIGCONT and SIGTSTP
   **/

  /* If a SIGTSTP arrives it will be forwarded to the
     application processes the handler for SIGCONT is added
     and SIGTSTP is delivered to the process itself.
     A following SIGCONT is handled the same way. */

#ifdef BSD_SIG
  vec.sv_handler = _sig_tstp;
  vec.sv_mask = sigmask(SIGCONT) |
    sigmask(SIGTSTP) |
      sigmask(SIGIO) |
        sigmask(SIGCHLD);
  vec.sv_flags = 0;

  if (sigvec(SIGTSTP, &vec, (struct sigvec *)0) != 0)
    return (-1);
#else
  if ((sigemptyset(&sigact.sa_mask) != 0) ||
      (sigaddset(&sigact.sa_mask,SIGCONT) != 0) || /* Block SIGCONT */
      (sigaddset(&sigact.sa_mask,SIGTSTP) != 0) ||
      (sigaddset(&sigact.sa_mask,SIGIO) != 0) ||
      (sigaddset(&sigact.sa_mask,SIGCHLD) != 0))
    return (-1);

  sigact.sa_handler = _sig_tstp;
  sigact.sa_flags = 0;

  if(sigaction(SIGTSTP, &sigact, (struct sigaction *)0) < 0)
    return(-1);
#endif

  /* LAMBO: vpnspec mu"s freigegeben werden */
  
  return (thesize);
}

