/******************************************************************************
*
*    extended integer calls
*    Copyright (C) 1993 A. Bode, S. Lamberts, T. Ludwig, G. Stellner
*
*    This file is part of NXLIB (Paragon(TM) message passing on workstations)
*
*    NXLIB is free software; you can redistribute it and/or
*    modify it under the terms of the GNU Library General Public
*    License as published by the Free Software Foundation; either
*    version 2 of the License, or (at your option) any later version.
*
*    NXLIB is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*    Library General Public License for more details.
*
*    You should have received a copy of the GNU Library General Public
*    License along with this library; if not, write to the Free
*    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*    Contact to the authors:
*
*    electronic mail: nxlib@informatik.tu-muenchen.de
*
*    paper mail:      Prof. Dr. A. Bode
*                     Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
*                     Institut f"ur Informatik
*                     Technische Universit"at M"unchen
*                     80290 M"unchen
*                     Germany
*
*    Paragon(TM) is a trademark of Intel Corporation.
*
******************************************************************************/
/******************************************************************************

  extint.c,v
  1994/12/30 16:33:39
  1.3
  Exp
  lamberts
 
  Authors: Norman Thomson, Stefan Lamberts

  Description: extended integer calls

  Available functions from this module: ....

******************************************************************************/
#ifndef lint
static char rcs_id[] = "extint.c,v 1.3 1994/12/30 16:33:39 lamberts Exp";
#endif


#include <errno.h>

#include "../include/sys/nxerrno.h"
#include "../include/sys/nxesize.h"

#define N (8*sizeof(long))
  
extern void nx_perror();


static int two[62][20] = {    
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,6},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,2},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,4},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,2,8},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,5,6},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,1,2},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,2,4},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,4,8},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,9,6},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,1,9,2},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,6,3,8,4},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,2,7,6,8},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,5,5,3,6},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,1,0,7,2},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,6,2,1,4,4},
{0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,2,4,2,8,8},
{0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,4,8,5,7,6},
{0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,9,7,1,5,2},
{0,0,0,0,0,0,0,0,0,0,0,0,0,4,1,9,4,3,0,4},
{0,0,0,0,0,0,0,0,0,0,0,0,0,8,3,8,8,6,0,8},
{0,0,0,0,0,0,0,0,0,0,0,0,1,6,7,7,7,2,1,6},
{0,0,0,0,0,0,0,0,0,0,0,0,3,3,5,5,4,4,3,2},
{0,0,0,0,0,0,0,0,0,0,0,0,6,7,1,0,8,8,6,4},
{0,0,0,0,0,0,0,0,0,0,0,1,3,4,2,1,7,7,2,8},
{0,0,0,0,0,0,0,0,0,0,0,2,6,8,4,3,5,4,5,6},
{0,0,0,0,0,0,0,0,0,0,0,5,3,6,8,7,0,9,1,2},
{0,0,0,0,0,0,0,0,0,0,1,0,7,3,7,4,1,8,2,4},
{0,0,0,0,0,0,0,0,0,0,2,1,4,7,4,8,3,6,4,8},
{0,0,0,0,0,0,0,0,0,0,4,2,9,4,9,6,7,2,9,6},
{0,0,0,0,0,0,0,0,0,0,8,5,8,9,9,3,4,5,9,2},
{0,0,0,0,0,0,0,0,0,1,7,1,7,9,8,6,9,1,8,4},
{0,0,0,0,0,0,0,0,0,3,4,3,5,9,7,3,8,3,6,8},
{0,0,0,0,0,0,0,0,0,6,8,7,1,9,4,7,6,7,3,6},
{0,0,0,0,0,0,0,0,1,3,7,4,3,8,9,5,3,4,7,2},
{0,0,0,0,0,0,0,0,2,7,4,8,7,7,9,0,6,9,4,4},
{0,0,0,0,0,0,0,0,5,4,9,7,5,5,8,1,3,8,8,8},
{0,0,0,0,0,0,0,1,0,9,9,5,1,1,6,2,7,7,7,6},                 
{0,0,0,0,0,0,0,2,1,9,9,0,2,3,2,5,5,5,5,2},
{0,0,0,0,0,0,0,4,3,9,8,0,4,6,5,1,1,1,0,4},
{0,0,0,0,0,0,0,8,7,9,6,0,9,3,0,2,2,2,0,8},
{0,0,0,0,0,0,1,7,5,9,2,1,8,6,0,4,4,4,1,6},
{0,0,0,0,0,0,3,5,1,8,4,3,7,2,0,8,8,8,3,2},
{0,0,0,0,0,0,7,0,3,6,8,7,4,4,1,7,7,6,6,4},
{0,0,0,0,0,1,4,0,7,3,7,4,8,8,3,5,5,3,2,8},
{0,0,0,0,0,2,8,1,4,7,4,9,7,6,7,1,0,6,5,6},
{0,0,0,0,0,5,6,2,9,4,9,9,5,3,4,2,1,3,1,2},
{0,0,0,0,1,1,2,5,8,9,9,9,0,6,8,4,2,6,2,4},
{0,0,0,0,2,2,5,1,7,9,9,8,1,3,6,8,5,2,4,8},
{0,0,0,0,4,5,0,3,5,9,9,6,2,7,3,7,0,4,9,6},
{0,0,0,0,9,0,0,7,1,9,9,2,5,4,7,4,0,9,9,2},
{0,0,0,1,8,0,1,4,3,9,8,5,0,9,4,8,1,9,8,4},
{0,0,0,3,6,0,2,8,7,9,7,0,1,8,9,6,3,9,6,8},
{0,0,0,7,2,0,5,7,5,9,4,0,3,7,9,2,7,9,3,6},
{0,0,1,4,4,1,1,5,1,8,8,0,7,5,8,5,5,8,7,2},
{0,0,2,8,8,2,3,0,3,7,6,1,5,1,7,1,1,7,4,4},
{0,0,5,7,6,4,6,0,7,5,2,3,0,3,4,2,3,4,8,8},
{0,1,1,5,2,9,2,1,5,0,4,6,0,6,8,4,6,9,7,6},
{0,2,3,0,5,8,4,3,0,0,9,3,1,3,6,9,3,9,5,2},
};




/*****************************************************************************/
static esize_t enegate
#ifdef ANSI
(esize_t e)
#else
(e)
esize_t e;
#endif
/*****************************************************************************/
{
  esize_t res;

  if ((e.shigh == 0) && (e.slow < 0) && (e.slow == -e.slow))
  {
    res.shigh = 1;
    res.slow  = 0;
  }
  else if ((e.shigh > 0) && (e.slow == 0))
  {
    res.shigh = 1 - e.shigh;
    res.slow  = (long)1 << (N - 1); /* Highest negative number in slow */
  }
  else 
  {
    res.shigh = -e.shigh;
    res.slow  = -e.slow;
  }

  return (res);
}



/*****************************************************************************/
static esize_t eshift
#ifdef ANSI
(esize_t e, int s, int *ofl)
#else
(e, s, ofl)
esize_t e;
int s;
int *ofl;
#endif
/*****************************************************************************/
{
  esize_t erg;
  
  *ofl = 0;                     /* No overflow occured */
  
  if (s == 0)
  {
    return (e);
  }

  if (s < 0)
  {
    /* Shift to the left */

    if (
        /* Shift all shigh out of shigh --> ofl if e.shigh != 0 */
        (((-s) > (N - 1)) && (e.shigh != 0)) ||
        /* Shift all slow out of shigh --> ofl if e.slow !=0 */
        (((-s) > (2*(N - 1))) && (e.slow != 0)) ||
        /* Shift part of slow out of shigh --> ofl if bits are lost */
        (((-s) > (N - 1)) &&
         (e.slow & (((1 << (-s - (N - 1))) - 1) << (2*N - 3 + s)))) ||
        /* Shift part of shigh out of shigh --> ofl if bits are lost */
        (e.shigh & (((1 << (-s)) - 1) << (N - 2 + s))))
      *ofl = 1;
    

    erg.shigh = e.shigh << (-s);   /* Shift shigh to the left */
    /* Extract the highest s bit from slow and add them to shigh.
       Shift slow to the right as far as necessary and add this to
       the result */
    
    erg.shigh += (e.slow >> (N - 1 + s));
    
    /* Shift slow to the left */
    erg.slow = ((unsigned long)(e.slow << (1-s)) >> 1);
                
    return (erg);
  }

  /* Shift to the right */
  erg.slow = e.slow >> s;    /* Shift slow to the right */
  /* Extract the lowest s bits form shigh and add them to slow
     
     Shift shigh one more than necessary to the left, then once
     to the right (in order to eliminate the sign bit) than add
     this to the result. */
     
  erg.slow += ((unsigned long)(e.shigh << (N - s)) >> 1);

  erg.shigh = e.shigh >> s;

  return (erg);
}


/*****************************************************************************/
esize_t _eadd
#ifdef ANSI
(esize_t p1, esize_t p2)
#else
(p1, p2)
esize_t p1;
esize_t p2;
#endif
/*****************************************************************************/
{
  esize_t sum;
  esize_t a,b;
  long carry = 0;
  int nflg = 0;
  
  /* Set a to the positive parameter b to the other */
  if ((p1.shigh >= 0) && (p1.slow >= 0))
  {
    a = p1;
    b = p2;
  }
  else if ((p2.shigh >= 0) && (p2.slow >= 0))
  {
    a = p2;
    b = p1;
  }
  else 
  {
    /* Both are negative. Negate them, add them and negate the sum later */
    a = enegate(p1);
    b = enegate(p2);
    nflg = 1;
  }
  

  if ((b.shigh >= 0) && (b.slow >= 0))
  {
    if ((sum.slow = a.slow + b.slow) < 0)
    {
      /* Remove first bit */
      sum.slow  = (((unsigned long)sum.slow << 1) >> 1);
      carry = 1;
    }
    if ((sum.shigh = a.shigh + b.shigh + carry) < 0)
    {
      sum.slow  = -1; sum.shigh = -1;
      errno = EQESIZE;
      return (sum);
    }
  }

  else if ((b.shigh == 0) && (b.slow < 0))
  {
    if (((sum.slow = a.slow + b.slow) < 0) && (a.shigh > 0))
    {
      /* Remove first bit */
      sum.slow  = (((unsigned long)sum.slow << 1) >> 1);
      sum.shigh = a.shigh - 1;
    }
    else
      sum.shigh = a.shigh;
  }

  else if ((b.shigh < 0) && (b.slow < 0))
  {
    sum.shigh = a.shigh + b.shigh;
    sum.slow  = a.slow  + a.slow;

    if ((sum.shigh > 0) && (sum.slow < 0))
    {
      /* Remove first bit */
      sum.slow  = (((unsigned long)sum.slow << 1) >> 1);
      sum.shigh--;
    }
    else if ((sum.shigh < 0) && (sum.slow > 0))
    {
      /* Set first bit */
      sum.slow = sum.slow | ((long)1 << (N - 1));
      sum.shigh++;
    }
  }

  else 
  {
    sum.shigh = -1; sum.slow  = -1;
    errno     = EQESIZE;
    return (sum);
  }

  if (nflg)
    sum = enegate(sum);

  return (sum);
}

    

/*****************************************************************************/
esize_t eadd
#ifdef ANSI
(esize_t p1, esize_t p2)
#else
(p1, p2)
esize_t p1;
esize_t p2;
#endif
/*****************************************************************************/
{
  esize_t res;
  int tmperr = errno;
  
  errno = 0;
  
  res = _eadd(p1,p2);
  if ((res.shigh == -1) && (res.slow == -1) && (errno != 0))
  {
    nx_perror("eadd()");
    exit(1);
  }
  
  errno = tmperr;
  
  return (res);
}


/*****************************************************************************/
esize_t _esub
#ifdef ANSI
(esize_t p1, esize_t p2)
#else
(p1, p2)
esize_t p1;
esize_t p2;
#endif
/*****************************************************************************/
{
  return (_eadd(p1,enegate(p2)));
}


/*****************************************************************************/
esize_t esub
#ifdef ANSI
(esize_t p1, esize_t p2)
#else
(p1, p2)
esize_t p1;
esize_t p2;
#endif
/*****************************************************************************/
{
  esize_t res;
  int tmperr = errno;
  
  errno = 0;
  
  res = _esub(p1,p2);
  if ((res.shigh == -1) && (res.slow == -1) && (errno != 0))
  {
    nx_perror("esub()");
    exit(1);
  }
  
  errno = tmperr;
  
  return (res);
}



/*****************************************************************************/
long _ecmp
#ifdef ANSI
(esize_t p1, esize_t p2)
#else
(p1,p2)
esize_t p1;
esize_t p2;
#endif
/*****************************************************************************/
{
  if ((p1.shigh < p2.shigh) ||
      ((p1.shigh == p2.shigh) && (p1.slow < p2.slow)))
  {
    return (-1);
  }
  
  if ((p1.shigh == p2.shigh) && (p1.slow == p2.slow))
  {
    return (0);
  }
  
  return (1);
}


/*****************************************************************************/
long ecmp
#ifdef ANSI
(esize_t e1, esize_t e2)
#else
(e1, e2)
esize_t e1;
esize_t e2;
#endif
/*****************************************************************************/
{
  return (_ecmp(e1,e2));
}



/*****************************************************************************/
long _ediv
#ifdef ANSI
(esize_t e, long l)
#else
(e, l)
esize_t e;
long l;
#endif
/*****************************************************************************/
{
  long n = l;
  esize_t p;
  int i,j,k,nflg;
  long erg;
  unsigned long bit;
  int ofl;
  
  nflg = 0;
  
  if (n==0)                     /* division by 0 */
  {
    errno = EQESIZE;
    return (-1);
  }
  
  
  if (e.shigh == 0)
    return (e.slow / n);
  
  if (n < 0)
  {
    n = -n;
    nflg = !nflg;
  }
  
  if ((e.shigh <= 0) && (e.slow < 0))
  {
    p = enegate(e);
    nflg = !nflg;
  }
  else
    p = e;
  
  
  /***
   *** First divide by the highest possible power of 2 by shifting
   *** the divisor and the dividend
   ***/

  /* Set i to the number of the lowest bit set in n */
  bit = 1;
  for (i=0; i < 32; i++)
  {
    if ((bit & n) != 0)
    {
      break;
    }
    
    bit <<= 1;
  }

  /* Shift both p and n by i to the right */
  n >>= i;                      /* Shift n */
  p = eshift(p,i,&ofl);        /* Shift p to the right */

  /***
   *** Check wether there is still something to do
   ***/
  if (n == 1)
  {
    if (p.shigh != 0)
    {
      errno = EQESIZE;
      return (-1);
    }
    return(nflg ? -p.slow : p.slow);
  }
  
  /* Try to devide only the lower bits */
  if (p.shigh == 0)
    return(nflg ? (-(p.slow/n)) : (p.slow/n));
  
  /***
   *** Unfortulately it is necessary to devide esize_t
   ***/

  /* Set i to the number of the highest bit set in n */
  bit = 1 << (N - 2);
  for (i=(N-2); i > 0; i--)
  {
    if ((bit & n) != 0)
      break;
    bit >>= 1;
  }
  
  /* Set j to the number of the highest bit set in p.shigh */
  bit = 1 << (N - 2);
  for (j=(N-2); j > 0; j--)
  {
    if ((bit & p.shigh) != 0)
      break;
    bit >>= 1;
  }

  /* Align n and p.shigh */
  if (i < j)
  {
    errno = EQESIZE;
    return (-1);
    /* The following case leads allways to an error
     * Prove:
     * n > 1:
     *        2^n > 1
     * ==>    2^n + 2^x > 1 [x = N-1]
     * ==>    0 > 1 - 2^n - 2^x
     * ==>    2^{n+x} > 1 - 2^n - 2^x + 2^{n+x}
     * ==>    2^{n+x} > (2^n - 1)(2^x - 1)
     * ==>    2^{n+x}/(2^n - 1) > 2^x - 1
     * That means that the result of the division of a number with
     * p.shigh > n is greater than the greatest long
     */
    /* Shift n to the left */
    /* n <<= (j-i);
       k = N-1+(j-i); */
  }
  else if (i > j)
  {
    p = eshift(p,(j-i),&ofl);         /* Shift p to the left */
    if (ofl)
    {
      errno = EQESIZE;
      return (-1);
    }
    
    k = N-1-(i-j);
  }
  else
    k = N-1;

  erg = 0;

  /* Divide */
  for (;k >= 0; k--)
  {
    erg <<= 1;
    if (erg < 0)
    {
      errno = EQESIZE;
      return(-1);
    }
    if (p.shigh >= n)
    {
      p.shigh -= n;
      erg++;
    }
    p = eshift(p,-1,&ofl);
    if (ofl)
    {
      errno = EQESIZE;
      return (-1);
    }
  }

  return (nflg ? -erg : erg);
}




/*****************************************************************************/
long ediv
#ifdef ANSI
(esize_t p, long n)
#else
(p, n)
esize_t p;
long n;
#endif
/*****************************************************************************/
{
  long e;
  int tmperr = errno;
  
  errno = 0;
  
  e = _ediv(p,n);
  
  if ((e == -1) && (errno != 0))
  {
    nx_perror("ediv");
    exit(1);
  }
  
  errno = tmperr;
  return e;
}


/*****************************************************************************/
long _emod
#ifdef ANSI
(esize_t e, long l)
#else
(e, l)
esize_t e;
long l;
#endif
/*****************************************************************************/
{
  long n = l;
  esize_t p,q;
  int i,j,k,nflg;
  long erg;
  unsigned long bit;
  int p2;
  long mod2;
  int ofl;
  
  nflg = 0;
  erg = 0;
  
  if (n==0)                     /* division by 0 */
  {
    errno = EQESIZE;
    return (-1);
  }
  
  
  if (e.shigh == 0)
    return (e.slow % n);
  
  if (n < 0)
    n = -n;
  
  if ((e.shigh <= 0) && (e.slow < 0))
  {
    p = enegate(e);
    nflg = !nflg;
  }
  else
    p = e;
  
  
  /***
   *** First divide by the highest possible power of 2 by shifting
   *** the divisor and the dividend
   ***/

  /* Set i to the number of the lowest bit set in n */
  bit = 1;
  for (p2=0; p2 < 32; p2++)
  {
    if ((bit & n) != 0)
    {
      break;
    }
    
    bit <<= 1;
  }

  /* Shift both p and n by p2 to the right */
  n >>= p2;                     /* Shift n */
  p = eshift(p,p2,&ofl);       /* Shift p to the right */
  q = eshift(p,-p2,&ofl);

  if (e.slow >= 0)
    mod2 = e.slow - q.slow;
  else
    mod2 = -e.slow - q.slow;

  /***
   *** Check wether there is still something to do
   ***/
  if (n == 1)
  {
    return(nflg ? -mod2 : mod2);
  }
  
  /* Try to mod only the lower bits */
  if (p.shigh == 0)
  {
    erg = p.slow % n;
    erg <<= (p2 - 1);                 /* LAMBO: Vorsicht "Uberlauf */
    erg += mod2;
    return(nflg ? -erg : erg);
  }
  
  /***
   *** Unfortulately it is necessary to devide esize_t to get the rest
   ***/

  /* Set i to the number of the highest bit set in n */
  bit = 1 << (N - 2);
  for (i=(N-2); i > 0; i--)
  {
    if ((bit & n) != 0)
      break;
    bit >>= 1;
  }
  
  /* Set j to the number of the highest bit set in p.shigh */
  bit = 1 << (N - 2);
  for (j=(N-2); j > 0; j--)
  {
    if ((bit & p.shigh) != 0)
      break;
    bit >>= 1;
  }

  /* Align n and p.shigh */
  if (i < j)
  {
    n <<= (j-i);                /* Shift n to the left */
    k = N-1+(j-i);
  }
  else if (i > j)
  {
    p = eshift(p,(j-i),&ofl);  /* Shift p to the left */
    if (ofl)
    {
      errno = EQESIZE;
      return (-1);
    }
    k = N-1-(i-j);
  }
  else
    k = N-1;

  /* Divide */
  for (;k >= 0; k--)
  {
    if (p.shigh >= n)
    {
      p.shigh -= n;
    }
    p = eshift(p,-1,&ofl);
    if (ofl)
    {
      errno = EQESIZE;
      return (-1);
    }
  }

  erg = p.shigh << (p2 - 1);          /* LAMBO: Vorsicht "Uberlauf */
  erg += mod2;
  
  return (nflg ? -erg : erg);
}

/*****************************************************************************/
long emod
#ifdef ANSI
(esize_t e, long n)
#else
(e, n)
esize_t e;
long n;
#endif
/*****************************************************************************/
{
  long res;
  int tmperr = errno;
  
  errno = 0;
  
  res = _emod(e,n);
  
  if ((res == -1) && (errno != 0))  {
    nx_perror("emod");
    exit(1);
  }
  
  errno = tmperr;
  return res;
}


/*****************************************************************************/
esize_t _emul
#ifdef ANSI
(esize_t e, long y)
#else
(e, y)
esize_t e;
long y;
#endif
/*****************************************************************************/
{
  int i;
  unsigned long b;
  esize_t erg;
  esize_t tmp;
  esize_t x;
  int nflg=0;
  int ofl;
  int tmperr = errno;
  
  erg.slow = 0;
  erg.shigh = 0;
  
  if ((y == 0) || ((e.shigh == 0) && (e.slow == 0)))
  {
    return erg;
  }

  errno = 0;

  if (y < 0)
  {
    nflg = !nflg;
    y = -y;                     /* Hier kann's zum "uberlauf kommen */
  }
  if ((e.shigh <= 0) && (e.slow < 0))
  {
    x = enegate(e);
    nflg = !nflg;
  }
  else
    x = e;
  
  b = 1;
  if (b & y)
    erg = x;

  
  for (i = 1, b = 2; i < (N - 2); i++, b <<= 1)
  {
    if (b & y)
    {
      tmp = eshift(x,-i,&ofl);
      if (ofl)
      {
        erg.slow = -1; erg.shigh = -1;
        errno = EQESIZE;
        return (erg);
      }
      
      erg = _eadd(tmp,erg);
      if ((erg.slow == -1) && (erg.shigh == -1) && (errno != 0))
        return (erg);
    }
  }
  
  if (nflg)
    erg = enegate(erg);
  
  errno = tmperr;

  return (erg);
}




/*****************************************************************************/
esize_t emul
#ifdef ANSI
(esize_t p, long n)
#else
(p, n)
esize_t p;
long n;
#endif
/*****************************************************************************/
{
  esize_t e;
  int tmperr = errno;
  
  errno = 0;
  
  e = _emul(p,n);
  if ((e.shigh == -1) && (e.slow == -1) && (errno != 0))
  {
    nx_perror("emul()");
    exit(1);
  }
  
  errno = tmperr;
  
  return e;
}


/*****************************************************************************/
long _etos
#ifdef ANSI
(esize_t e, char *s)
#else
(e, s)
esize_t e;
char *s;
#endif
/*****************************************************************************/
{
  int tmp[20];
  
  int nflg=0;
  int i,j;
  unsigned long  c;
  char *sp;
  
  for (j=0; j<20;j++)
    tmp[j] = 0;
  
  if ((e.shigh <= 0) && (e.slow < 0))
  {
    e = enegate(e);
    nflg = 1;
  }
  
  for (i = 0, c = 1; i < 31; c <<= 1, i++)
  {
    if (e.slow & c)
      for (j = 0; j < 20; j++)
        tmp[j] += two[i][j];
  }
  
  for (i = 31, c = 1; i < 62; c <<= 1, i++)
  {
    if (e.shigh & c)
      for (j = 0; j <= 19; j++)
        tmp[j] += two[i][j];
  }
  
  for (j=19; j>0; j--)
  {
    tmp[j-1] += tmp[j]/10;
    tmp[j] %= 10;
  }
  
  sp = s;
  if (nflg)
    *sp++ = '-';
  
  for (j = 0; j < 19; j++)      /* Skip leading 0s */
    if (tmp[j]) break;
  
  for (; j < 19; j++)
    *sp++ = '0'+(char)tmp[j];
  
  *sp++ = '0'+(char)tmp[19];
  
  *sp = '\0';
  
  return 0;
}


/*****************************************************************************/
void etos
#ifdef ANSI
(esize_t e, char *s)
#else
(e, s)
esize_t e;
char *s;
#endif
/*****************************************************************************/
{
  if (_etos(e,s) < 0)
  {
    nx_perror("etos");
    exit(1);
  }
  
  return;
}


/*****************************************************************************/
esize_t _stoe
#ifdef ANSI
(char *s)
#else
(s)
char *s;
#endif
/*****************************************************************************/
{
  char *sp = s;
  esize_t e;
  esize_t res;
  int nflg = 0;
  int tmperr = errno;
  errno = 0;
  
  if (*sp == '-')
  {
    nflg = 1;
    sp++;
  }
  else if (*sp == '+')
    sp++;
  
  res.slow = 0;
  res.shigh = 0;
  e.shigh = 0;
  
  for (;*sp;sp++)
  {
    if ((*sp > '9') || (*sp < '0'))
    {
      res.slow = -1; res.shigh = -1;
      errno = EQESIZE;
      return (res);
    }
    e.slow = (long)(*sp - '0');
    res = _emul(res,(long)10);
    if ((res.slow == -1) && (res.shigh == -1) && (errno != 0))
      return (res);
    res = _eadd(res,e);
    if ((res.slow == -1) && (res.shigh == -1) && (errno != 0))
      return (res);
  }
  
  errno = tmperr;

  return (nflg ? enegate(res) : res);
}



/*****************************************************************************/
esize_t stoe
#ifdef ANSI
(char *s)
#else
(s)
char *s;
#endif
/*****************************************************************************/
{
  esize_t res;
  int tmperr = errno;
  
  errno = 0;
  
  res = _stoe(s);
  if ((res.shigh == -1) && (res.slow == -1) && (errno != 0))
  {
    nx_perror("stoe()");
    exit(1);
  } 
  
  errno = tmperr;
  
  return (res);
}
