/******************************************************************************
*
*    Address and Process database 
*    Copyright (C) 1993 A. Bode, S. Lamberts, T. Ludwig, G. Stellner
*
*    This file is part of NXLIB (Paragon(TM) message passing on workstations)
*
*    NXLIB is free software; you can redistribute it and/or
*    modify it under the terms of the GNU Library General Public
*    License as published by the Free Software Foundation; either
*    version 2 of the License, or (at your option) any later version.
*
*    NXLIB is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*    Library General Public License for more details.
*
*    You should have received a copy of the GNU Library General Public
*    License along with this library; if not, write to the Free
*    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*    Contact to the authors:
*
*    electronic mail: nxlib@informatik.tu-muenchen.de
*
*    paper mail:      Prof. Dr. A. Bode
*                     Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
*                     Institut f"ur Informatik
*                     Technische Universit"at M"unchen
*                     80290 M"unchen
*                     Germany
*
*    Paragon(TM) is a trademark of Intel Corporation.
*
******************************************************************************/
/******************************************************************************

  addr_db.c,v
  1995/02/09 10:49:24
  1.10
  Exp
  lamberts
 
  Authors: Stefan Lamberts

  Description:

  This file contains procedures to manipulate the database of address ond
  other information about the processes of an application.

  Available functions from this module:

  int _init_plist()

  int _rem_set_fdset(fdset)
  fd_set **fdset;

  p_listel *_find_plistel(node, ptype)
  long node;
  long ptype;

  p_listel *_find_plistel_radr(radr)
  rem_addr *radr;

  p_listel *_find_plistel_radr(node, pid)
  long node;
  int pid;

  int _add_ptype_plistel(node, ptype, pid)
  long node;
  long ptype;
  int pid;

  int _set_addr_plistel(node, ptype, loc, radr, ladr, hname, login, arch, pid)
  long node;
  long ptype;
  int loc;
  rem_addr *radr;
  loc_addr *ladr;
  char *hname;
  char *login;
  char *arch;
  int pid;

  int _update_plistel(node, ptype, loc, radr, ladr, hname, login, arch, pid)
  long node;
  long ptype;
  int loc;
  rem_addr *radr;
  loc_addr *ladr;
  char *hname;
  char *login;
  char *arch;
  int pid;

  int _ins_plistel(node, ptype, loc, radr, ladr, hname, login, arch, pid, l_node, l_ptype)
  long node;
  long ptype;
  int loc;
  rem_addr *radr;
  loc_addr *ladr;
  char *hname;
  char *login;
  char *arch;
  int pid;
  long l_node;
  long l_ptype;


  _del_plistel_radr(radr)
  rem_addr *radr;


  ....... some more.
  


******************************************************************************/
#ifndef lint
static char rcs_id[] = "addr_db.c,v 1.10 1995/02/09 10:49:24 lamberts Exp";
#endif

#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <errno.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef SUN_OS4
#include <memory.h>
#endif

#ifdef RS6K_AIX
#include <sys/select.h>
#endif

#include "../include/sys/nxerrno.h"
#include "../include/sys/nxlib.h"
#include "../include/nxmalloc.h"


extern long __numnodes();
extern long __mynode();
extern long __myptype();
extern int  _answer_requests();
extern int  _rel_send();

static p_listel **nnrarr = (p_listel **)0; /* Array of pointers to address
                                              entries indexed by node number */
static long nnrsiz = 0;         /* Size of the array */

static p_listel **rcdarr = (p_listel **)0; /* Array of pointers to address
                                              entries indexed by the remote
                                              communication descriptors, which
                                              are socket file descriptors */
static int rcdsiz = 0;         /* Size of the array */

static fd_set fdesc_set;

static max_fdesc = 0;


/*****************************************************************************
  Initialize the database

  Returns  0 on success
          -1 on failure (errno will be set)
*****************************************************************************/
int _init_plist
#ifdef ANSI
(void)
#else
()
#endif
/*****************************************************************************/
{
  static int wascalled = 0;
  
  long i;
  
  struct rlimit rlim;

  if (wascalled)
  {
    errno = EALREADY;
    return (-1);
  }
  wascalled = 1;
  
  /***
   *** Get the number of nodes, allocate nnrarr, and initalize it
   ***/
  if ((nnrsiz = __numnodes()) < (long)0)
    return (-1);
  nnrsiz++;                     /* One more for the controlling process */

  if ((nnrarr = (p_listel **)malloc((size_t)(sizeof(p_listel **)*nnrsiz))) ==
      (p_listel **)0)
    return (-1);
  
  for (i=0; i < nnrsiz; i++)
    nnrarr[i] = (p_listel *)0;

  /***
   *** Get the maximal number of file descriptors, allocate rcdarr, and
   *** initalize it
   ***/
#ifdef RS6K_AIX
  rcdsiz = getdtablesize();
#else
#ifdef LINUX
  rcdsiz = 256;
#else
  if (getrlimit(RLIMIT_NOFILE,&rlim) != 0)
    return (-1);
  
  rcdsiz = rlim.rlim_max;
#endif
#endif

  if ((rcdarr = (p_listel **)malloc((size_t)(sizeof(p_listel **)*rcdsiz)))
      == (p_listel **)0)
    return (-1);

  for (i=0; i < rcdsiz; i++)
    rcdarr[i] = (p_listel *)0;


  FD_ZERO(&fdesc_set);
  
  max_fdesc = -1;
  
  return 0;
}


/*****************************************************************************
  Set the filedescriptor set of all remote connections

  Returns the highest descriptor on success
          -1 on failure

*****************************************************************************/
int _rem_set_fdset
#ifdef ANSI
(fd_set **fdset)
#else
(fdset)
fd_set **fdset;
#endif
/*****************************************************************************/
{
  *fdset = &fdesc_set;
  return (max_fdesc);
}

  

        

/*****************************************************************************
  Find a database entry by nodenumber and ptype

  Returns a pointer to the entry on success
          a NULL pointer on failure (errno will be set)
                         if errno is 0 there is no entry

*****************************************************************************/
p_listel *_find_plistel
#ifdef ANSI
(long node, long ptype)
#else
(node, ptype)
long node;
long ptype;
#endif
/*****************************************************************************/
{
  p_listel *tmpadr;
  ptype_el *tmppt;

  /* Check input parameter */
  if (((node < 0) || (node >= nnrsiz)) ||

      ((ptype != DAEMON_PTYPE) && (ptype != CP_PTYPE) && (ptype < 0)))
  {
    errno = EINVAL;
    return ((p_listel *)0);
  }  
  
  errno = 0;

  /***
   *** If node is the number of the controlling process,
   *** don't care about the ptype because there is only one process
   *** on this node.
   ***/
  if (node == (nnrsiz-1))
    return (nnrarr[node]);
  
  /***
   *** Look at every entry with this node number
   ***/
  for (tmpadr = nnrarr[node]; tmpadr != (p_listel *)0; tmpadr = tmpadr->next)
  {
    
    /***
     *** Check the list of ptypes that this process had
     ***/
    for (tmppt = &tmpadr->ptypes; tmppt != (ptype_el *)0; tmppt = tmppt->next)
      if (tmppt->ptype == ptype)
        return (tmpadr);
  }
  
  return ((p_listel *)0);
}





/*****************************************************************************
  Find an entry by the remote address
  
  Returns a pointer to the entry on success
          a NULL pointer on failure (errno will be set)
                         if errno is 0 there is no entry

*****************************************************************************/
p_listel *_find_plistel_radr
#ifdef ANSI
(rem_addr *radr)
#else
(radr)
rem_addr *radr;
#endif
/*****************************************************************************/
{

  /* Check input parameter */
  if ((radr == (rem_addr *)0) ||
      (radr->sockfd < 0) || 
      (radr->sockfd >= rcdsiz))
  {
    errno = EINVAL;
    return ((p_listel *)0);
  }  

  /***
   *** rcdarr cointains the pointer to the entry or NULL
   ***/
  errno = 0;
  return (rcdarr[radr->sockfd]);
}



/*****************************************************************************
  Find an entry by the nodenumber and process id
  
  Returns a pointer to the entry on success
          a NULL pointer on failure (errno will be set)
                         if errno is 0 there is no entry

*****************************************************************************/
p_listel *_find_plistel_pid
#ifdef ANSI
(long node, pid_t pid)
#else
(node, pid)
long node;
pid_t pid;
#endif
/*****************************************************************************/
{
  p_listel *tmpadr;

  /* Check input parameter */
  if ((node < 0) || (node >= nnrsiz))
  {
    errno = EINVAL;
    return ((p_listel *)0);
  }  
  
  errno = 0;

  /***
   *** Look at every entry with this node number
   ***/
  for (tmpadr = nnrarr[node]; tmpadr != (p_listel *)0; tmpadr = tmpadr->next)
    if (tmpadr->pid == pid)
      return (tmpadr);
  
  return (p_listel *)0;
}



/*****************************************************************************
  Create a database entry

  Returns a pointer to the entry on success
          a NULL pointer on failure (errno will be set)
                         if errno is 0 there is no entry

*****************************************************************************/
static p_listel *create_plistel
#ifdef ANSI
(long node, long ptype, int loc, rem_addr *radr, loc_addr *ladr, char *hname,
 char *login, char *arch, pid_t pid, long l_node, long l_ptype)
#else
(node, ptype, loc, radr, ladr, hname, login, arch, pid, l_node, l_ptype)
long node;
long ptype;
int loc;
rem_addr *radr;
loc_addr *ladr;
char *hname;
char *login;
char *arch;
pid_t pid;
long l_node;
long l_ptype;
#endif
/*****************************************************************************/
{
  p_listel *entry;
  
  if ((entry = (p_listel *)malloc(sizeof(p_listel))) == (p_listel *)0)
    return ((p_listel *)0);

  /* Initialize structure elements */

  entry->node = node;

  if (hname != (char *)0)
  {
    (void)strncpy(entry->hname,hname,MAXHNAMELEN-1);
    entry->hname[MAXHNAMELEN-1] = '\0';
  }
  else
    entry->hname[0] = '\0';

  if (login != (char *)0)
  {
    (void)strncpy(entry->login,login,MAXLOGINLEN-1);
    entry->login[MAXLOGINLEN-1] = '\0';
  }
  else
    entry->login[0] = '\0';

  if (arch != (char *)0)
  {
    (void)strncpy(entry->arch,arch,MAXARCHLEN-1);
    entry->arch[MAXARCHLEN-1] = '\0';
  }
  else
    entry->arch[0] = '\0';
  
  entry->loc = loc;

  if (radr == (rem_addr *)0)
  {
    entry->radr.sockfd = UNDEF_SOCK;
    entry->radr.stat = ADDR_UNKNOWN;
    memset((void *)&entry->radr.sad,(int)0,sizeof(struct sockaddr_in));
  }
  else
  {
    entry->radr.sockfd = radr->sockfd;
    entry->radr.stat = radr->stat;
    memcpy((void *)&entry->radr.sad, (void *)&radr->sad,
           sizeof(struct sockaddr_in));
  }

  entry->pid = pid;
  entry->stat = PROC_RUNN;
  
  entry->l_node = l_node;
  entry->l_ptype = l_ptype;

  entry->ptypes.ptype = ptype;
  entry->ptypes.next = (ptype_el *)0;

  return(entry);
}    




/*****************************************************************************
 * Adds a ptype to an entry
 * returns
 *         PT_OK if the entry exists (ptype is entered)
 *         PT_EX if the entry exists but is a different process
 *         PT_NE if the entry is not kwown
 *        -1 on failure
 *****************************************************************************/
int _add_ptype_plistel
#ifdef ANSI
(long node, long ptype, pid_t pid)
#else
(node, ptype, pid)
long node;
long ptype;
pid_t pid;
#endif
/*****************************************************************************/
{
  p_listel *entry;
  ptype_el *ptent;

  if (((node < 0) || (node >= nnrsiz)) ||

      ((ptype != DAEMON_PTYPE) && (ptype != CP_PTYPE) && (ptype < 0)))
  {
    errno = EINVAL;
    return (-1);
  }
  
  /* controlling process may have any ptype */
  if (node == (nnrsiz - 1))
  {
    if (_find_plistel(node,ptype) == (p_listel *)0)
      return (PT_NE);           /* Controlling process not yet known */
    
    return (PT_OK);             /* Controlling process is known */
  }
  
  
  /* Check weather this ptype existes */
  if ((entry = _find_plistel(node,ptype)) != (p_listel *)0)
  {
    if ((entry->pid != (pid_t)UNDEF_PID) && (entry->pid != pid))
    {
      return (PT_EX);           /* other process */
    }
    return (PT_OK);             /* same process */
  }
      
  /* try to find this process */
  for (entry = nnrarr[node];
       (entry != (p_listel *)0) && (entry->pid != pid);
       entry = entry->next);

  if (entry == (p_listel *)0) 
    return (PT_NE);                 /* Entry does not exist */
  
  /* add ptype */
  if ((ptent = (ptype_el *)malloc(sizeof(ptype_el)))
      == (ptype_el *)0)
    return (-1);
  
  ptent->ptype = entry->ptypes.ptype;
  ptent->next = entry->ptypes.next;
  entry->ptypes.ptype = ptype;
  entry->ptypes.next = ptent;

  /* Send an answer to pending address requests */
  if (__myptype() == DAEMON_PTYPE)
    if (_answer_requests() < 0)
      return (-1);
      
  return (PT_OK);               /* same process */
}




/*****************************************************************************
 * Insert address infromation in an existing entry
 * returns
 *         0 on success
 *        -1 on failure
 *****************************************************************************/
int _set_addr_plistel
#ifdef ANSI
(long node, long ptype, int loc, rem_addr *radr, loc_addr *ladr, char *hname,
 char *login, char *arch, (pid_t) pid)
#else
(node, ptype, loc, radr, ladr, hname, login, arch, pid)
long node;
long ptype;
int loc;
rem_addr *radr;
loc_addr *ladr;
char *hname;
char *login;
char *arch;
pid_t pid;
#endif
/*****************************************************************************/
{
  p_listel *entry  = (p_listel *)0;
  int i;
  
  
  /***
   *** Check input parameter
   ***/
  
  if (ladr != (loc_addr *)0)
  {
    errno = EINVAL;
    return (-1);
  }
  
  if (((node < 0) || (node >= nnrsiz)) ||

      ((ptype != DAEMON_PTYPE) && (ptype != CP_PTYPE) && (ptype < 0)) ||

      ((radr != (rem_addr *)0) &&
       (radr->sockfd != UNDEF_SOCK) &&
       ((radr->sockfd < 0) || (radr->sockfd >= rcdsiz))))

  {
    errno = EINVAL;
    return (-1);
  }  
    
  /* find entry by node number and ptype */
  if ((entry = _find_plistel(node,ptype)) == (p_listel *)0)
  {
    errno = ENXLENEX;
    return (-1);                /* entry does not exist */
  }
  
  /* update remote address information */
  if (radr != (rem_addr *)0)
  {
    /* update socket descriptor */
    if ((radr->sockfd != UNDEF_SOCK) && (entry->stat == PROC_RUNN))
    {
      if (entry->radr.sockfd == UNDEF_SOCK)
      {
        entry->radr.sockfd = radr->sockfd;
        if (rcdarr[entry->radr.sockfd] != (p_listel *)0)
        {
          /* This socket is used by another process */
          errno = ENXLADR;
          return (-1);
        }
        rcdarr[entry->radr.sockfd] = entry;
        FD_SET(entry->radr.sockfd,&fdesc_set);
        if (entry->radr.sockfd > max_fdesc)
          max_fdesc = entry->radr.sockfd;
      }
      else if ((entry->radr.sockfd != radr->sockfd) ||
               (entry != rcdarr[radr->sockfd]))
      {
        /* There is already a socket connection to this process.
         * That happens if both processes try to send to eachother
         * simultaneously.
         * The sender process with the higher nodenumber or
         * the same nodenumber and higher pid closes the socket.
         * The other process keeps the socket open until a message is sent
         * that allows to close the socket. */
        if (pid == (pid_t)UNDEF_PID)
        {
          errno = EINVAL;
          return (-1);
        }
        
        if ((node < __mynode()) ||
            ((node == __mynode()) && (pid < getpid())))
        {
          /* I am the process with higher nodenumber or the same nodenumber
           * and higher pid */

          /* Send a message to the receiving process that this socket can
           * be closed */
          if (_rel_send((long)DOUBLE_CONN,
                        (char *)0,
                        (long)0,
                        node,
                        ptype) < 0)
            return (-1);

          /* Close the socket */
          close(entry->radr.sockfd);
          rcdarr[entry->radr.sockfd] = (p_listel *)0;
          FD_CLR(entry->radr.sockfd,&fdesc_set);
          for (i = (rcdsiz-1); (i >= 0) && (rcdarr[i] == (p_listel *)0); i--);
          max_fdesc = i;
        
          /* Enter the other socket descriptor into the process entry */
          entry->radr.sockfd = radr->sockfd;
          if (rcdarr[entry->radr.sockfd] != (p_listel *)0)
          {
            /* This socket is used by another process */
            errno = ENXLADR;
            return (-1);
          }
          rcdarr[entry->radr.sockfd] = entry;
          FD_SET(entry->radr.sockfd,&fdesc_set);
          if (entry->radr.sockfd > max_fdesc)
            max_fdesc = entry->radr.sockfd;
        }
      }
    }

    /* update address information */
    switch (entry->radr.stat)
    {
    case ADDR_SET:
      break;
    case ADDR_KNOWN:
      if (radr->stat == ADDR_SET)
      {
        entry->radr.stat = ADDR_SET;
        memcpy((void *)&entry->radr.sad, (void *)&radr->sad,
               sizeof(struct sockaddr_in));
      }
      break;
    case ADDR_UNKNOWN:
      if ((radr->stat == ADDR_KNOWN) || (radr->stat == ADDR_SET))
      {
        entry->radr.stat = radr->stat;
        memcpy((void *)&entry->radr.sad, (void *)&radr->sad,
               sizeof(struct sockaddr_in));
      }
      break;
    default:
      errno = ENXLADR;
      return (-1);
    }
  }

  /* update other information */
  if ((hname != (char *)0) && (entry->hname[0] == '\0'))
  {
    (void)strncpy(entry->hname,hname,MAXHNAMELEN-1);
    entry->hname[MAXHNAMELEN-1] = '\0';
  }
  
  if ((login != (char *)0) && (entry->login[0] == '\0'))
  {
    (void)strncpy(entry->login,login,MAXLOGINLEN-1);
    entry->login[MAXLOGINLEN-1] = '\0';
  }

  if ((arch != (char *)0) && (entry->arch[0] == '\0'))
  {
    (void)strncpy(entry->arch,arch,MAXARCHLEN-1);
    entry->arch[MAXARCHLEN-1] = '\0';
  }

  if ((loc != LOC_UNDEF) && (entry->loc != LOC_UNDEF) && (loc != entry->loc))
  {
    errno = ENXLADR;
    return (-1);
  }
  if (entry->loc == LOC_UNDEF)
    entry->loc = loc;

  if (entry->pid == (pid_t)UNDEF_PID)
    entry->pid = pid;

  /* Send an answer to pending address requests */
  if (__myptype() == DAEMON_PTYPE)
    if (_answer_requests() < 0)
      return (-1);
  
  return (0);
}


/*****************************************************************************
 * Insert address information in an existing entry or create a new entry
 * return
 *        0 on sucess
 *       -1 on failure
 *****************************************************************************/
int _update_plistel
#ifdef ANSI
(long node, long ptype, int loc, rem_addr *radr, loc_addr *ladr, char *hname,
 char *login, char *arch, pid_t pid, long l_node, long l_ptype)
#else
(node, ptype, loc, radr, ladr, hname, login, arch, pid)
long node;
long ptype;
int loc;
rem_addr *radr;
loc_addr *ladr;
char *hname;
char *login;
char *arch;
pid_t pid;
#endif
/*****************************************************************************/
{
  p_listel *entry;

  /***
   *** Check input parameter
   ***/

  if (ladr != (loc_addr *)0)
  {
    errno = EINVAL;
    return (-1);
  }
  
  
  if (((node < 0) || (node >= nnrsiz)) ||

      ((ptype != DAEMON_PTYPE) && (ptype != CP_PTYPE) && (ptype < 0)) ||

      ((loc != LOC_REMOTE) && (loc != LOC_MYSELF)) ||

      ((radr != (rem_addr *)0) &&
       (radr->sockfd != UNDEF_SOCK) &&
       ((radr->sockfd < 0) || (radr->sockfd >= rcdsiz))) ||

      (((ptype > 0) || (ptype == CP_PTYPE)) && (pid < (pid_t)0)) ||
      ((ptype == DAEMON_PTYPE) && (pid != (pid_t)UNDEF_PID) &&
       (pid < (pid_t)0)))

  {
    errno = EINVAL;
    return (-1);
  }  


  /* Depending on the existence of this process */
  switch (_add_ptype_plistel(node,ptype,pid))
  {
  case PT_NE:
    /* create a new entry */
    if ((entry = create_plistel(node, ptype, loc, radr, ladr,
                                hname, login, arch,
                                pid,
                                (long)UNDEF_NODE,
                                (long)UNDEF_PTYPE))
        == (p_listel *)0)
      return (-1);

    /* add in nnrarr to the list of ptypes with the same node number */
    entry->next = nnrarr[node];   /* LAMBO: Es ist evtl. besser am Ende
                                     einzuf"ugen; wegen den Adressen
                                     der Daemone */
    nnrarr[node] = entry;
    
    /* enter in rcdarr */
    if (entry->radr.sockfd != UNDEF_SOCK)
    {
      if (rcdarr[entry->radr.sockfd] != (p_listel *)0)
      {
        errno = ENXLADR;
        return (-1);
      }
      rcdarr[entry->radr.sockfd] = entry;
      FD_SET(entry->radr.sockfd,&fdesc_set);
      if (entry->radr.sockfd > max_fdesc)
        max_fdesc = entry->radr.sockfd;
    }

    /* Send an answer to pending address requests */
    if (__myptype() == DAEMON_PTYPE)
      if (_answer_requests() < 0)
        return (-1);
      
    break;

  case PT_OK:
    /* update the existing entry */
    if (_set_addr_plistel(node,ptype,loc,radr,ladr,hname,login,arch,pid)
        < 0)
      return (-1);
    break;
  case PT_EX:
    /* There is a different process with on the same node with the same ptype */
  default:
    errno = ENXLADR;
    return (-1);
  }
  return (0);
}
    


/*****************************************************************************
  Create and insert a new entry into the database
  Returns
           0 on success
          -1 on failure (errno is set)
*****************************************************************************/
int _ins_plistel
#ifdef ANSI
(long node, long ptype, int loc, rem_addr *radr, loc_addr *ladr, char *hname,
 char *login, char *arch, pid_t pid, long l_node, long l_ptype)
#else
(node, ptype, loc, radr, ladr, hname, login, arch, pid, l_node, l_ptype)
long node;
long ptype;
int loc;
rem_addr *radr;
loc_addr *ladr;
char *hname;
char *login;
char *arch;
pid_t pid;
long l_node;
long l_ptype;
#endif
/*****************************************************************************/
{
  p_listel *entry  = (p_listel *)0;
  
  
  /***
   *** Check input parameter
   ***/
  
  if (ladr != (loc_addr *)0)
  {
    errno = EINVAL;
    return (-1);
  }
  
  if (((node < 0) || (node >= nnrsiz)) ||

      ((ptype != DAEMON_PTYPE) && (ptype != CP_PTYPE) && (ptype < 0)) ||

      ((loc != LOC_REMOTE) && (loc != LOC_MYSELF)) ||

      ((radr != (rem_addr *)0) && (radr->sockfd != UNDEF_SOCK) &&
       ((radr->sockfd < 0) || (radr->sockfd >= rcdsiz))) ||

      ((ptype == DAEMON_PTYPE) && (pid != (pid_t)UNDEF_PID) &&
       (pid < (pid_t)0)) ||
      ((ptype != DAEMON_PTYPE) && (pid < (pid_t)0)) ||

      ((l_node != UNDEF_NODE) && ((l_node < 0) || (l_node >= nnrsiz))) ||

      ((l_ptype != UNDEF_PTYPE) &&
       (l_ptype != CP_PTYPE) &&
       (l_ptype < 0)))
  {
    errno = EINVAL;
    return (-1);
  }  
    
  if (_find_plistel(node,ptype) != (p_listel *)0)
  {
    errno = ENXLEEX;
    return (-1);
  }

  if ((entry = create_plistel(node, ptype, loc, radr, ladr, hname,
                                  login, arch, pid, l_node, l_ptype))
      == (p_listel *)0)
      return (-1);

  /* add in nnrarr to the list of ptypes with the same node number */
  entry->next = nnrarr[node];   /* LAMBO: Es ist evtl. besser am Ende
                                   einzuf"ugen; wegen den Adressen
                                   der Daemone */
  nnrarr[node] = entry;
    
  /* enter in rcdarr */
  if (entry->radr.sockfd != UNDEF_SOCK)
  {
    if (rcdarr[entry->radr.sockfd] != (p_listel *)0)
    {
      errno = ENXLADR;
      return (-1);
    }
    rcdarr[entry->radr.sockfd] = entry;
    FD_SET(entry->radr.sockfd,&fdesc_set);
    if (entry->radr.sockfd > max_fdesc)
      max_fdesc = entry->radr.sockfd;
  }

  return (0);
}

  

/*****************************************************************************
  Free the memory used by an entry
*****************************************************************************/
static void free_plistel
#ifdef ANSI
(p_listel *elem)
#else
(elem)
p_listel *elem;
#endif
/*****************************************************************************/
{
  ptype_el *ptel;
  ptype_el *nptel;

  if (elem == (p_listel *)0)
    return;

  for (ptel=elem->ptypes.next; ptel != (ptype_el *)0; ptel = nptel)
  {
    nptel = ptel->next;
    free((void *)ptel);
  }
  
  free((void *)elem);
  return;
}


      
 
/*****************************************************************************
  Delete an entry
  Returns
          0 on success
         -1 on failure
*****************************************************************************/
static int del_plistel
#ifdef ANSI
(p_listel *entry)
#else
(entry)
p_listel *entry;
#endif
/*****************************************************************************/
{
  p_listel *act;
  p_listel *prev;
  int i;

  /* Delete it from rcdarr */
  if (entry->radr.sockfd != UNDEF_SOCK)
  {
    rcdarr[entry->radr.sockfd] = (p_listel *)0;
    FD_CLR(entry->radr.sockfd,&fdesc_set);
    for (i = (rcdsiz-1); (i >= 0) && (rcdarr[i] == (p_listel *)0); i--);
    max_fdesc = i;
  }
  

  /* Delete it from nnrarr. */
  for (prev = (p_listel *)0, act = nnrarr[entry->node];
       (act != (p_listel *)0) && (act != entry);
       prev = act, act = act->next);
  if (act == (p_listel *)0) 
  {
    errno = ENXLADR;
    return (-1);
  }
  if (prev == (p_listel *)0) 
    nnrarr[entry->node] = entry->next;
  else
    prev->next = entry->next;

  free_plistel(entry);

  return (0);
}




/*****************************************************************************
  Delete an entry with a spsecified remote address

  Returns
          0 on success
         -1 on failure
*****************************************************************************/
int _del_plistel_radr
#ifdef ANSI
(rem_addr *radr)
#else
(radr)
rem_addr *radr;
#endif
/*****************************************************************************/
{
  p_listel *entry;
  
  /***
   *** Check input parameter
   ***/
  
  if ((radr == (rem_addr *)0) ||
      (radr->sockfd < 0) ||
      (radr->sockfd >= rcdsiz))
  {
    errno = EINVAL;
    return (-1);
  }  
  

  /* find entry by communication descriptor */
  if ((entry = rcdarr[radr->sockfd]) == (p_listel *)0)
    return (0);

  return (del_plistel(entry));
}


 

/*****************************************************************************
  Delete an entry with a specified node and process id

  Returns
          0 on success.
         -1 on failure.
*****************************************************************************/
int _del_plistel_pid
#ifdef ANSI
(long node, pid_t pid)
#else
(node, pid)
long node;
pid_t pid;
#endif
/*****************************************************************************/
{
  p_listel *entry;
  
  /***
   *** Check input parameter
   ***/
  
  if ((node < 0) ||
      (node >= nnrsiz))
  {
    errno = EINVAL;
    return (-1);
  }  
  
  /* find entry nodenumber and pid */
  for (entry = nnrarr[node];
       (entry != (p_listel *)0) && (entry->pid != pid);
       entry = entry->next);
    
  if (entry  == (p_listel *)0)
    return (0);

  return (del_plistel(entry));
}



/*****************************************************************************
  Invalidate the socketdescriptor of an entry

  Returns
          0 on success.
         -1 on failure.
*****************************************************************************/
int _inval_plistel_radr
#ifdef ANSI
(rem_addr *radr)
#else
(radr)
rem_addr *radr;
#endif
/*****************************************************************************/
{
  p_listel *entry;
  int i;
  
  /***
   *** Check input parameter
   ***/
  
  if ((radr == (rem_addr *)0) ||
      (radr->sockfd < 0) ||
      (radr->sockfd >= rcdsiz))
  {
    errno = EINVAL;
    return (-1);
  }  
  
  /* find entry by communication descriptor */
  if ((entry = rcdarr[radr->sockfd]) == (p_listel *)0)
    return (0);
  
  rcdarr[radr->sockfd] = (p_listel *)0;
  entry->radr.sockfd = UNDEF_SOCK;
  FD_CLR(radr->sockfd,&fdesc_set);
  for (i = (rcdsiz-1); (i >= 0) && (rcdarr[i] == (p_listel *)0); i--);
  max_fdesc = i;

  return (0);
}




/*****************************************************************************/
int _inval_plistel_pid
#ifdef ANSI
(long node, pid_t pid)
#else
(node, pid)
long node;
pid_t pid;
#endif
/*****************************************************************************/
{
  p_listel *entry;
  
  /***
   *** Check input parameter
   ***/
  
  if ((node < 0) ||
      (node >= nnrsiz))
  {
    errno = EINVAL;
    return (-1);
  }  
  
  /* find entry nodenumber and pid */
  for (entry = nnrarr[node];
       (entry != (p_listel *)0) && (entry->pid != pid);
       entry = entry->next);
    
  if (entry  == (p_listel *)0)
    return (0);
  
  entry->stat = PROC_DEAD;

  return (0);
}







/*****************************************************************************/
int _kill_aps
#ifdef ANSI
(int sig)
#else
(sig)
int sig;
#endif
/*****************************************************************************/
{
  p_listel *entry;
  
  for (entry = nnrarr[__mynode()]; entry != (p_listel *)0; entry = entry->next)
    if (entry->ptypes.ptype != DAEMON_PTYPE)
    {
      if (kill(entry->pid, sig) != 0)
        return (-1);
    }
  
  return (0);
}

      

/*****************************************************************************
 * Forward a FLUSHMSG message to all processes of this node
 ****************************************************************************/ 
int _flush_aps
#ifdef ANSI
(msg_desc *msgd)
#else
(msgd)
msg_desc *msgd;
#endif
/*****************************************************************************/
{
  p_listel *entry;
  msg_flushmsg *msg;

  msg = (msg_flushmsg *)msgd->msg_ptr;
  
  for (entry = nnrarr[__mynode()]; entry != (p_listel *)0; entry = entry->next)
    /* Don't send a FLUSHMSG message to the process that delivered the call */
    if ((entry->ptypes.ptype != DAEMON_PTYPE) &&
        ((msg->f_node != __mynode()) || (msg->f_pid != entry->pid)))
    {
      if (_rel_send((long)FLUSHMSG,
                    msgd->msg_ptr,
                    msgd->msg_len,
                    __mynode(),
                    entry->ptypes.ptype) < 0)
        return (-1);
    }
  
  return (0);
}

      
