/******************************************************************************
*
*    Set and get ptype
*    Copyright (C) 1993 A. Bode, S. Lamberts, T. Ludwig, G. Stellner
*
*    This file is part of NXLIB (Paragon(TM) message passing on workstations)
*
*    NXLIB is free software; you can redistribute it and/or
*    modify it under the terms of the GNU Library General Public
*    License as published by the Free Software Foundation; either
*    version 2 of the License, or (at your option) any later version.
*
*    NXLIB is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*    Library General Public License for more details.
*
*    You should have received a copy of the GNU Library General Public
*    License along with this library; if not, write to the Free
*    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*    Contact to the authors:
*
*    electronic mail: nxlib@informatik.tu-muenchen.de
*
*    paper mail:      Prof. Dr. A. Bode
*                     Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
*                     Institut f"ur Informatik
*                     Technische Universit"at M"unchen
*                     80290 M"unchen
*                     Germany
*
*    Paragon(TM) is a trademark of Intel Corporation.
*
******************************************************************************/
/******************************************************************************

  ptype.c,v
  1995/02/06 16:11:53
  1.7
  Exp
  lamberts
 
  Authors: Stefan Lamberts, Susanna Schink

  Description: Set and get ptype

  Available functions from this module: ....

******************************************************************************/
#ifndef lint
static char rcs_id[] = "ptype.c,v 1.7 1995/02/06 16:11:53 lamberts Exp";
#endif

#include <sys/types.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>

#include "../include/sys/nxerrno.h"
#include "../include/sys/nxlib.h"
#include "../include/nxmalloc.h"

extern long __myhost();
extern long __mynode();
extern int  _confirm_messages();
extern int  _handle_hcalls();
extern void _rel_disable();
extern void _rel_enable();
extern int  _rel_send();
extern void _rel_wait_msg();
extern void nx_perror();

static long ptype = -1;
static long *ptypes = (long *)0;
static long ptypeanz = 0;
static int setptype_answer;
static int setptype_ok;


/*************************************************************************
 * Returns the present ptype without any error checking
 ************************************************************************/
long
#ifdef ANSI
__myptype(void)
#else
__myptype()
#endif
/************************************************************************/
{
  return(ptype);
}



/*************************************************************************
 * Gets the process type of the calling process
 * - upon successful completion, returns the ptype of the caller
 * - otherwise, returns -1 and indicate the error
 ************************************************************************/
long
#ifdef ANSI
_myptype(void)
#else
_myptype()
#endif
/************************************************************************/
{
  if ((ptype < 0)  || (ptype > MAXPTYPE))
  {
    errno = EQNOSET;
    return (-1);
  }
  return (ptype);
}


/*************************************************************************
 * Gets the process type of the calling process
 * - upon successful completion, returns the ptype of the caller, and  
 *    returns control to the caller 
 * - otherwise, displays an error message to stderr. and causes the calling
 *   process to terminate
 ************************************************************************/
long
#ifdef ANSI
myptype(void)
#else
myptype()
#endif
/************************************************************************/
{
  long res;

  if ((res = _myptype()) < 0)
  {
    nx_perror("myptype()");
    exit(-1);
  }
  return (res);
  
}



/*************************************************************************
 * Sets the process ptype without any error checking.
 * The ptype is added to a list of ptypes this process allready had.
 ************************************************************************/
long
#ifdef ANSI
__setptype(long val)
#else
__setptype(val)
long val;
#endif
/************************************************************************/
{
  
  if (ptypeanz == 0) {
    if ((ptypes = (long *)malloc((size_t)sizeof(long))) == (long *)0)
      return (-1);
  }
  else 
  {
    if ((ptypes = (long *)realloc((void *)ptypes,
                                  (size_t)(sizeof(long)*(ptypeanz+1)))) ==
        (long *)0)
      return (-1);
  }
  ptypes[ptypeanz++] = ptype = val;
  return (0);
}


/*****************************************************************************/
int _had_ptype
#ifdef ANSI
(long val)
#else
(val)
long val;
#endif
/*****************************************************************************/
{
  int i;

  for (i = 0; i < ptypeanz; i++) 
    if (ptypes[i] == val)
      return (1);
    
  return (0);
}


  
/*****************************************************************************
 * Wait for a message as long as this condition is true
 *****************************************************************************/
static int setpt_answer
#ifdef ANSI
(void)
#else
()
#endif
/************************************************************************/
{
  return(setptype_answer == 0);
}




/*************************************************************************
 * Sets the process type of the message
 * - upon successful completion returns 0
 * - otherwise returns -1 and indicate the error
 ************************************************************************/
long
#ifdef ANSI
_setptype(long val)
#else
_setptype(val)
long val;
#endif
/************************************************************************/
{
  long  node;
  
  msg_ad_setptype msg;
  
  /* Invalid ptype */
  if (val < 0 || val > MAXPTYPE) {
    errno = EQPID;
    return (-1);
  }

  /* ptype allready set */
  if (val == ptype) 
    return (0);

  _rel_disable();

  node = __mynode();

  if (node == __myhost())        /* this is the controlling process */
  {
    if (_had_ptype(val) == 1)
      ptype = val;
    else   if (__setptype(val) < 0)
    {
      _rel_enable();
      return (-1);
    }
  }
  else 
  {
    /* this is an application process */
    if (_had_ptype(val) == 1)
      ptype = val;
    else 
    {
      /* Ask the daemon for permission to set this ptype */
      setptype_answer = 0;
  
      msg.pid = getpid();
      msg.node = __mynode();
      msg.ptype = val;
  
      if (_rel_send((long)AD_SETPTYPE,
                    (char *)&msg,
                    (long)sizeof(msg_ad_setptype),
                    node,
                    (long)DAEMON_PTYPE) < 0)
      {
        _rel_enable();
        return (-1);
      }

      _rel_wait_msg(setpt_answer);
   
  
      switch (setptype_ok) 
      {
      case PT_OK:
        break;
      case PT_EX:
        errno = EQUSEPID;
        return (-1);
      case PT_NE:
      default:
        errno = ENXLADR;
        _rel_enable();
        return (-1);
      }  

      if (__setptype(val) < 0)
      {
        _rel_enable();
        return (-1);
      }
    }
  }
  
  if ((_confirm_messages() < 0) || (_handle_hcalls() < 0))
  {
    _rel_enable();
    return (-1);
  }
   
  /* Enable incomming messages */
  _rel_enable();  
 
  return (0);
}


/*************************************************************************
 * Set the process type of the process
 * - upon successful completion returns control to the calling process
 * - otherwise displays an error msg. to stderr. and causes the calling
 *   process to terminate 
 ************************************************************************/
void
#ifdef ANSI
setptype(long val)
#else
setptype(val)
long val;
#endif
/************************************************************************/
{
  if (_setptype(val) < 0) 
  {
    nx_perror("setptype()");
    exit(-1);
  }
  
  return;  
}


/*************************************************************************
 * Receive the answer from the daemon process about the requested ptype
 * If the daemon answer is positive set the new ptype to the requested
 * value and return 0
 * Else return -1 and set errno to indicate the error 
 ************************************************************************/
int _da_ackptype
#ifdef ANSI
(msgd *msg_desc)
#else
(msgd)
msg_desc *msgd;
#endif
/************************************************************************/
{
  msg_da_ackptype *msg;

  msg = (msg_da_ackptype *)msgd->msg_ptr;
  
  setptype_ok = msg->setptype_ok;

  setptype_answer = 1;
  
  return (0);
}
