/******************************************************************************
*
*    Load a program on nodes
*    Copyright (C) 1993 A. Bode, S. Lamberts, T. Ludwig, G. Stellner
*
*    This file is part of NXLIB (Paragon(TM) message passing on workstations)
*
*    NXLIB is free software; you can redistribute it and/or
*    modify it under the terms of the GNU Library General Public
*    License as published by the Free Software Foundation; either
*    version 2 of the License, or (at your option) any later version.
*
*    NXLIB is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*    Library General Public License for more details.
*
*    You should have received a copy of the GNU Library General Public
*    License along with this library; if not, write to the Free
*    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*    Contact to the authors:
*
*    electronic mail: nxlib@informatik.tu-muenchen.de
*
*    paper mail:      Prof. Dr. A. Bode
*                     Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
*                     Institut f"ur Informatik
*                     Technische Universit"at M"unchen
*                     80290 M"unchen
*                     Germany
*
*    Paragon(TM) is a trademark of Intel Corporation.
*
******************************************************************************/
/******************************************************************************

  nxload.c,v
  1995/02/06 16:11:50
  1.9
  Exp
  lamberts
 
  Authors: Stefan Lamberts, Susanna Schink

  Description:
  This file contains the implementation of the NX calls that are used for
  process creation and waiting for created processes.
  Additionally procedures are provieded which are used to receive related
  messages (DC_PIDNR, DC_APEXIT).

  Available functions from this module:
  
  long nx_loadve(node_list,numnodes, ptype, pid_list, pathname, argv, envp)
  long node_list[];
  long numnodes;
  long ptype;
  long pid_list[];
  char *pathname;
  char *argv[];
  char *envp[];

  long nx_load(node_list,numnodes, ptype, pid_list, pathname)
  long node_list[];
  long numnodes;
  long ptype;
  long pid_list[];
  char *pathname;

  long nx_waitall();
  
  void _dc_pidnr(msg)
  msg_desc *msg;

  void _dc_apexit(msg)
  msg_desc *msg;

******************************************************************************/
#ifndef lint
static char rcs_id[] = "nxload.c,v 1.9 1995/02/06 16:11:50 lamberts Exp";
#endif


#include <sys/wait.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "../include/sys/nxerrno.h"
#include "../include/sys/nxlib.h"
#include "../include/nxmalloc.h"

extern long __mynode();
extern long __myptype();
extern long __numnodes();
extern void _dbgmsg();
extern int  _inval_plistel_pid();
extern void _rel_disable();
extern void _rel_enable();
extern int  _rel_send();
extern void _rel_wait_msg();

extern char **environ;

static long no_loaded_procs = 0; /* Number of loaded processes */
static long no_exited_procs = 0; /* Number of terminated processes */
static long proc_returned_error = 0; /* If a processes retruned an
                                      * error this variable will be increased
                                      */
static long rec_pid_msgs;       /* number of pid messages received */
static long exp_pid_msgs;
static long *pids;





/*************************************************************************/
static int all_pid_msgs
#ifdef ANSI
(void)
#else
()
#endif
/*************************************************************************/
{
  return (rec_pid_msgs < exp_pid_msgs);
}



/**************************************************************************
 * Receive address information from the application process and update own
 * tables with it
 *************************************************************************/
int
#ifdef ANSI
_da_pidnr(msg_desc *msgd)
#else
_da_pidnr(msgd)
msg_desc *msgd;
#endif
/*************************************************************************/
{
  msg_da_pidnr *msg;
    
  msg = (msg_da_pidnr *)msgd->msg_ptr;
  
  if (__myptype() == DAEMON_PTYPE)
  {
    /* Forward the message to the application process */
    if (_rel_send((long)DA_PIDNR,
                  msgd->msg_ptr,
                  msgd->msg_len,
                  msg->l_node,
                  msg->l_ptype) < 0)
      return (-1);
    return (0);
  }

  pids[msg->naidx] = (long)msg->pid;

  rec_pid_msgs++;

  return (0);
}



/**************************************************************************/
long
#ifdef ANSI
nx_loadve(long *node_list,long num_nodes, long ptype, long *pid_list, char *path, char **argv, char **envp)
#else
nx_loadve(node_list, num_nodes, ptype, pid_list, path, argv, envp)
long *node_list;
long num_nodes;
long ptype;
long *pid_list;
char *path;
char **argv;
char **envp;
#endif
/**************************************************************************/
{

  long nn;                      /* Number of nodes */
  long nss;                      /* Number of successfully started programs */
  long *na;                      /* Array of nodenumbers */
  long i;
  long  len;                    /* Length of message */
  int  argc;                    /* Number of arguments */
  int  envc;                    /* Number of environment values */
  
  char **ptr;
  
  char *msg_buff;
  
  msg_ad_loadappl  *msg_part;
  
  char *msg_rest;

  /**
   ** Check input variables
   **/
  if ((num_nodes < -1) ||
      (num_nodes > __numnodes()))
  {
    errno = EPBADNODE;
    return (-1);
  }
  
  if (num_nodes == 0)
    return (0);

  if (ptype < 0) 
  {
    errno = EQPID;
    return (-1);
  }
  
  
  /**
   ** Set the number of nodes and the list of nodes,
   ** on which the program shall be loaded.
   **/

  if (num_nodes == -1)
  {                              /* if no nodes are specified,
                                 * use all nodes in the application
                                 */
    nn = __numnodes();
    if ((na = (long *)malloc((size_t)(sizeof(long)*nn))) == (long *)NULL)
    {
      errno = EPALLOCERR;
      return (-1);
    }
    for (i = 0; i < nn; i++)
      na[i] = i;
  }
  else
  {                              /* use the specified list */
    nn = num_nodes;
    na = node_list;
  } 
    

  /**
   ** Calculate the length of the message, which is sent to the daemon
   ** processes, and allocate memory.
   **/
  len = sizeof(msg_ad_loadappl);

  len += (strlen(path) + 1);

  argc = 0;
  for (ptr = argv; *ptr != (char *)0; ptr++) 
  {
    len += (strlen(*ptr) + 1) ;
    argc++;
  }

  envc = 0;
  for (ptr = envp; *ptr != NULL; ptr++)
  {
    len += (strlen(*ptr) + 1);
    envc++;
  }

  if ((msg_buff = (char *)malloc((size_t)len)) == (char *)NULL) {
    errno = EPALLOCERR;
    return (-1);
  }  

  /**
   ** Compose the message
   **/
  msg_part = (msg_ad_loadappl *)msg_buff;
  msg_part->l_node = __mynode();
  msg_part->l_ptype = __myptype();
  msg_part->ptype = ptype;
  msg_part->argc = argc;
  msg_part->envc = envc;

  /* set msg_rest to the beginning of the rest of the message */
  msg_rest = msg_buff + sizeof(msg_ad_loadappl);

  /* copy path */
  strcpy(msg_rest,path);
  msg_rest += (strlen(path) + 1);
  
  /* copy arguments */
  for (i = 0; i < argc; i++) 
  {
    strcpy(msg_rest,argv[i]);
    msg_rest += (strlen(argv[i]) + 1) ;
  }

  /* copy environment */
  for (i = 0; i < envc; i++) 
  {
    strcpy(msg_rest,envp[i]);
    msg_rest += (strlen(envp[i]) + 1) ;
  }

  /**
   ** Block receiving of messages.
   **/
  _rel_disable();
  

  /**
   ** Send the message to the daemon processes
   **/
  nss = 0;

  for (i = 0; i < nn; i++) {
    if ((na[i] < 0) || (na[i] >= __numnodes())) 
    {                            /* skip invalid node specifications  */
      pid_list[i] = 0;
      continue;
    }

    msg_part->naidx  = i;
    msg_part->node = na[i];
    
    if (_rel_send((long)AD_LOADAPPL,
                  msg_buff,
                  len,
                  /* The controlling process send to the daemons directly;
                     application processes send to their daemon wich forwards
                     the message */
                  (__mynode() == __numnodes()) ? na[i] : __mynode(),
                  (long)DAEMON_PTYPE) < 0)
    {
      _rel_enable();
      return (-1);
    }
    
    
    nss++;                      /* increase number of so far
                                 * successfully started processes
                                 */
  }
  
  

  if (num_nodes == -1) {
    free((void *)na);
  }

  pids = pid_list;                /* set list of received pids */
  exp_pid_msgs = nss;
  rec_pid_msgs = 0;

  /**
   ** allow messages again
   **/
  _rel_enable();

  /**
   ** Wait for messages containig the pids of the application processes
   **/
  
  _rel_wait_msg(all_pid_msgs);
  
  nss = 0;
  for (i = 0; i < nn; i++)
    if (pid_list[i] != 0) 
      nss++;

  _rel_disable();

  no_loaded_procs += nss;        /* increase the number of loaded processes
                                 * by the number of successfully started
                                 * processes
                                 */
  _rel_enable();
  
  return(nss);
}





/**************************************************************************/
long
#ifdef ANSI
nx_load(long *node_list, long num_nodes, long ptype, long *pid_list, char *path)
#else
nx_load(node_list, num_nodes, ptype, pid_list, path)
long *node_list;
long num_nodes;
long ptype;
long *pid_list;
char *path;
#endif
/**************************************************************************/
{
  char *argv[2];

  if ((argv[0] = strrchr(path,'/')) == (char *)0) /* find last occurence of '/'
                                                   */
    argv[0] = path;
  else
    argv[0]++;                  /* name of exececutable
                                   starts right after last '/' */
  argv[1] = (char *)0;
 
  return (nx_loadve(node_list,
                    num_nodes,
                    ptype,
                    pid_list,
                    path,
                    argv,
                    environ));
}





/*************************************************************************/
static int all_exit_msgs
#ifdef ANSI
(void)
#else
()
#endif
/*************************************************************************/
{
  return (no_loaded_procs > no_exited_procs);
}


/***************************************************************************
 * Receive exit code of a process started by nxloadve()
 **************************************************************************/
int
#ifdef ANSI
_da_apexit(msg_desc *msgd)
#else
_da_apexit(msgd)
msg_desc *msgd;
#endif
/*************************************************************************/
{
  msg_da_apexit *msg;
  
  msg = (msg_da_apexit *)msgd->msg_ptr;
  
  if (__myptype() == DAEMON_PTYPE)
  {
    /* Forward the message to the application process */
    if (_rel_send((long)DA_APEXIT,
                  msgd->msg_ptr,
                  msgd->msg_len,
                  msg->l_node,
                  msg->l_ptype) < 0)
      return (-1);
    return (0);
  }

  if (!WIFSTOPPED(msg->status)) 
  {
    if (WIFEXITED(msg->status))
    {
      fprintf(stderr,
             "Loaded Process (node: %ld, pid: %d) exited with status %d\n",
             msg->node, msg->pid, WEXITSTATUS(msg->status));
    }
    else if (WIFSIGNALED(msg->status)) 
    {
      fprintf(stderr,
             "Loaded Process (node: %ld, pid %d) terminated by signal %d\n",
             msg->node, msg->pid, WTERMSIG(msg->status));
      proc_returned_error++;
    }
    
    no_exited_procs++;
    
    if (_inval_plistel_pid(msg->node,msg->pid) < 0)
      return (-1);
  }
  
  return (0);
}



/**************************************************************************/
long
#ifdef ANSI
nx_waitall(void)
#else
nx_waitall()
#endif
/**************************************************************************/
{

  if (no_loaded_procs == 0)
  {
    errno = ECHILD;
    return (-1);
  }
  
  _rel_wait_msg(all_exit_msgs);

  no_loaded_procs = no_exited_procs = 0;
  
  errno = 0;
  if (proc_returned_error) 
  {
    proc_returned_error = 0;
    return (-1);
  }
  else
    return (0);
}


