/******************************************************************************
*
*    Buffer management for incomming and outgoing messages
*    Copyright (C) 1993 A. Bode, S. Lamberts, T. Ludwig, G. Stellner
*
*    This file is part of NXLIB (Paragon(TM) message passing on workstations)
*
*    NXLIB is free software; you can redistribute it and/or
*    modify it under the terms of the GNU Library General Public
*    License as published by the Free Software Foundation; either
*    version 2 of the License, or (at your option) any later version.
*
*    NXLIB is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*    Library General Public License for more details.
*
*    You should have received a copy of the GNU Library General Public
*    License along with this library; if not, write to the Free
*    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*    Contact to the authors:
*
*    electronic mail: nxlib@informatik.tu-muenchen.de
*
*    paper mail:      Prof. Dr. A. Bode
*                     Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
*                     Institut f"ur Informatik
*                     Technische Universit"at M"unchen
*                     80290 M"unchen
*                     Germany
*
*    Paragon(TM) is a trademark of Intel Corporation.
*
******************************************************************************/
/******************************************************************************

  msg_buffer.c,v
  1994/12/30 16:33:15
  1.9
  Exp
  lamberts
 
  Authors: Stefan Lamberts

  Description: Buffer management for incomming and outgoing messages

  Available functions from this module: ....

******************************************************************************/
#ifndef lint
static char rcs_id[] = "msg_buffer.c,v 1.9 1994/12/30 16:33:15 lamberts Exp";
#endif

#include <errno.h>
#include <stdlib.h>
#include <string.h>

#ifdef SUN_OS4
#include <memory.h>
#endif

#include "../include/sys/nxerrno.h"
#include "../include/sys/nxlib.h"
#include "../include/nxmalloc.h"

extern long __numnodes();
extern long __myptype();
extern int  _confirm_msg();

static pt_tab_m_el *ptt[PT_TAB_M_SIZE];

/*****************************************************************************/
int _init_m_buf
#ifdef ANSI
(void)
#else
()
#endif
/*****************************************************************************/
{
  int i;

  for (i=0; i < PT_TAB_M_SIZE; i++)
    ptt[i] = (pt_tab_m_el *)0;

  return (0);
}


/*****************************************************************************/
static pt_tab_m_el *find_m_pel
#ifdef ANSI
(long ptype)
#else
(ptype)
long ptype;
#endif
/*****************************************************************************/
{
  int ind;
  pt_tab_m_el *ptr;
  
  /* LAMBO: Check necessary */
  if ((ptype < 0) || (ptype > MAXPTYPE))
  {
    errno = EINVAL;
    return ((pt_tab_m_el *)0);
  }
  
  ind = (int)ptype%PT_TAB_M_SIZE;

  for (ptr = ptt[ind]; ptr != (pt_tab_m_el *)0; ptr = ptr->bucket)
    if (ptr->ptype == ptype)
      return (ptr);
  
  return (ptr);
}


/*****************************************************************************/
static int ins_m_pel
#ifdef ANSI
(pt_tab_m_el *pel)
#else
(pel)
pt_tab_m_el *pel;
#endif
/*****************************************************************************/
{
  int ind;
  pt_tab_m_el *ptr;
  
  ind = (int)pel->ptype%PT_TAB_M_SIZE;

  if (ptt[ind] == (pt_tab_m_el *)0)
  {
    ptt[ind] = pel;
    return (0);
  }
  
  /* Check wether there is already an element with that ptype */
  for (ptr = ptt[ind]; ptr->bucket != (pt_tab_m_el *)0; ptr = ptr->bucket)
    if (ptr->ptype == pel->ptype)
    {
      errno = ENXLBUF;              /* LAMBO: Internal error */
      return (-1);
    }

  if (ptr->ptype == pel->ptype)
  {
    errno = ENXLBUF;              /* LAMBO: Internal error */
    return (-1);
  }
  
  /* Append to the bucket list */
  ptr->bucket = pel;
  pel->bucket = (pt_tab_m_el *)0;
  
  return (0);
}



/*****************************************************************************/
static pt_tab_m_el *create_m_pel
#ifdef ANSI
(long ptype)
#else
(ptype)
long ptype;
#endif
/*****************************************************************************/
{
  pt_tab_m_el *ptr;
  int i;

  if ((ptr = (pt_tab_m_el *)malloc(sizeof(pt_tab_m_el))) == (pt_tab_m_el *)0)
    return (ptr);

  ptr->ptype = ptype;
  ptr->first = (buf_m_el *)0;
  ptr->last = (buf_m_el *)0;
  for (i = 0; i < MT_TAB_SIZE; i++)
    ptr->mt_tab[i] = (mt_tab_el *)0;

  ptr->bucket = (pt_tab_m_el *)0;

  return (ptr);
}



/*****************************************************************************/
static mt_tab_el *find_mtel
#ifdef ANSI
(long msg_type, pt_tab_m_el *pel)
#else
(msg_type, pel)
long msg_type;
pt_tab_m_el *pel;
#endif
/*****************************************************************************/
{
  int ind;
  mt_tab_el *ptr;
  
  ind = (int)msg_type%MT_TAB_SIZE;

  for (ptr = pel->mt_tab[ind]; ptr != (mt_tab_el *)0; ptr = ptr->bucket)
    if (ptr->msg_type == msg_type)
      return (ptr);
  
  return (ptr);
}


/*****************************************************************************/
static int ins_mtel
#ifdef ANSI
(mt_tab_el *mtel, pt_tab_m_el *pel)
#else
(mtel, pel)
mt_tab_el *mtel;
pt_tab_m_el *pel;
#endif
/*****************************************************************************/
{
  int ind;
  mt_tab_el *ptr;
  
  ind = (int)mtel->msg_type%MT_TAB_SIZE;

  if (pel->mt_tab[ind] == (mt_tab_el *)0)
  {
    pel->mt_tab[ind] = mtel;
    return (0);
  }
  
  /* Check wether there is already an element with that msg_type */
  for (ptr = pel->mt_tab[ind]; ptr->bucket!=(mt_tab_el *)0; ptr = ptr->bucket)
    if (ptr->msg_type == mtel->msg_type)
    {
      errno = ENXLBUF;              /* LAMBO: Internal error */
      return (1);
    }

  if (ptr->msg_type == mtel->msg_type)
  {
    errno = ENXLBUF;              /* LAMBO: Internal error */
    return (1);
  }
  
  ptr->bucket = mtel;
  mtel->bucket = (mt_tab_el *)0;
  
  return (0);
}


/*****************************************************************************/
static mt_tab_el *create_mtel
#ifdef ANSI
(long msg_type)
#else
(msg_type)
long msg_type;
#endif
/*****************************************************************************/
{
  mt_tab_el *ptr;

  if (((msg_type < 0) || 
       ((msg_type >= MAXUMT) && (msg_type < MAXSMT)) ||
       (msg_type >= MAXFMT)))
  {
    errno = EINVAL;
    return ((mt_tab_el *)0);
  }

  if ((ptr = (mt_tab_el *)malloc(sizeof(mt_tab_el))) == (mt_tab_el *)0)
    return (ptr);

  ptr->msg_type = msg_type;
  ptr->first = (buf_m_el *)0;
  ptr->last = (buf_m_el *)0;
  ptr->bucket = (mt_tab_el *)0;

  return (ptr);
}




/*****************************************************************************/
buf_m_el *_find_msg
#ifdef ANSI
(long msg_type, long s_node, long s_ptype)
#else
(msg_type, s_node, s_ptype)
long msg_type;
long s_node;
long s_ptype;
#endif
/*****************************************************************************/
{
  pt_tab_m_el *pel;
  mt_tab_el *mtel;
  buf_m_el *bme;
  
  if ((pel = find_m_pel(__myptype())) == (pt_tab_m_el *)0)
    return ((buf_m_el *)0);

  if (msg_type != -1)
  {
    /* Look in the list of messages with the same type */
    
    if ((mtel = find_mtel(msg_type,pel)) == (mt_tab_el *)0)
      return ((buf_m_el *)0);

    for (bme = mtel->first; bme != (buf_m_el *)0; bme = bme->mt_next)
      if (((s_node == -1) || (bme->s_node == s_node)) &&
          ((s_ptype == -1) || (bme->s_ptype == s_ptype)))
        return (bme);
  
    return (bme);
  }
  
  /* Look in the list of all messages */
  for (bme = pel->first; bme != (buf_m_el *)0; bme = bme->pt_next)
    if (((s_node == -1) || (bme->s_node == s_node)) &&
        ((s_ptype == -1) || (bme->s_ptype == s_ptype)))
      return (bme);
  
  return (bme);
}


/*****************************************************************************/
int _rm_msg
#ifdef ANSI
(buf_m_el *bme)
#else
(bme)
buf_m_el *bme;
#endif
/*****************************************************************************/
{
  pt_tab_m_el *pel;
  buf_m_el *prev;
  buf_m_el *next;
  mt_tab_el *mtel;
  
  if ((pel = find_m_pel(bme->r_ptype)) == (pt_tab_m_el *)0)
  {
    errno = ENXLBUF;              /* LAMBO Internal error */
    return (-1);
  }

  /* Remove form ptype list */
  if ((prev = bme->pt_prev) == (buf_m_el *)0)
    pel->first = bme->pt_next;
  else
    prev->pt_next = bme->pt_next;

  if ((next = bme->pt_next) == (buf_m_el *)0)
    pel->last = bme->pt_prev;
  else
    next->pt_prev = bme->pt_prev;

  /* Remove form message type list */
  if ((mtel = find_mtel(bme->msg_type,pel)) == (mt_tab_el *)0)
  {
    errno = ENXLBUF;                /* LAMBO: Internal error */
    return (-1);
  }
  
  if ((prev = bme->mt_prev) == (buf_m_el *)0)
    mtel->first = bme->mt_next;
  else
    prev->mt_next = bme->mt_next;

  if ((next = bme->mt_next) == (buf_m_el *)0)
    mtel->last = bme->mt_prev;
  else
    next->mt_prev = bme->mt_prev;

  return (0);
}


/*****************************************************************************/
int _ins_msg
#ifdef ANSI
(buf_m_el *bme)
#else
(bme)
buf_m_el *bme;
#endif
/*****************************************************************************/
{
  pt_tab_m_el *pel;
  mt_tab_el *mtel;
  buf_m_el *last;
  
  /* Find an entry in the ptype hashtable with the specified ptype */
  if ((pel = find_m_pel(bme->r_ptype)) == (pt_tab_m_el *)0)
  {
    /* Create and enter a new one if none was found */
    if ((pel = create_m_pel(bme->r_ptype)) == (pt_tab_m_el *)0)
      return (-1);

    if (ins_m_pel(pel) < 0)
      return (-1);
  }

  /* Insert the message in list with the same ptype */
  if ((last = pel->last) == (buf_m_el *)0)
  {
    pel->first = bme;
    bme->pt_prev = (buf_m_el *)0;
  }
  else
  {
    last->pt_next = bme;
    bme->pt_prev = last;
  }
  
  pel->last = bme;
  bme->pt_next = (buf_m_el *)0;

  /**
   ** Insert in message type list
   **/

  /* Find an entry in the msg_type hashtable with the specified msg_type */
  if ((mtel = find_mtel(bme->msg_type,pel)) == (mt_tab_el *)0)
  {
    /* Create and enter a new one if none was found */
    if ((mtel = create_mtel(bme->msg_type)) == (mt_tab_el *)0)
      return (-1);

    if (ins_mtel(mtel,pel) < 0)
      return (-1);
  }

  /* Insert the message in the list with the same msg_type */
  if ((last = mtel->last) == (buf_m_el *)0)
  {
    mtel->first = bme;
    bme->mt_prev = (buf_m_el *)0;
  }
  else
  {
    last->mt_next = bme;
    bme->mt_prev = last;
  }
  
  mtel->last = bme;
  bme->mt_next = (buf_m_el *)0;

  return (0);
}
  



/*****************************************************************************/
buf_m_el *_create_msg
#ifdef ANSI
(int loc, msg_desc *msgd)
#else
(loc, msgd)
int loc;
msg_desc *msgd;
#endif
/*****************************************************************************/
{
  buf_m_el *bmel;

  if ((msgd == (msg_desc *)0) ||
      ((msgd->s_node < 0) || (msgd->s_node > __numnodes())) ||
      ((msgd->r_node < 0) || (msgd->r_node > __numnodes())) ||
      ((msgd->s_ptype < 0) || (msgd->s_ptype > MAXPTYPE)) ||
      ((msgd->r_ptype < 0) || (msgd->r_ptype > MAXPTYPE)) ||
      ((msgd->msg_type < 0) || 
      ((msgd->msg_type >= MAXUMT) && (msgd->msg_type < MAXSMT)) ||
      (msgd->msg_type >= MAXFMT)))
  {
    errno = EINVAL;
    return ((buf_m_el *)0);
  }
      
  if ((bmel = (buf_m_el *)malloc(sizeof(buf_m_el))) == (buf_m_el *)0)
    return (bmel);

  bmel->location = loc;
  bmel->status = MSG_UNCONFIRMED;

  /* Copy message to a new buffer if it's a message form the same process.
     This is necessary in order to be able to reuse the buffer after the send
     call. */
  if ((loc == LOC_MYSELF) && (msgd->msg_len > 0))
  {
    if ((bmel->msg_ptr = malloc((size_t)msgd->msg_len)) == (char *)0)
      return ((buf_m_el *)0);
    memcpy((void *)bmel->msg_ptr,(void *)msgd->msg_ptr,(size_t)msgd->msg_len);
  }
  else
    bmel->msg_ptr = msgd->msg_ptr;

  bmel->msg_len = msgd->msg_len;
  bmel->msg_type = msgd->msg_type;
  bmel->s_node = msgd->s_node;
  bmel->s_ptype = msgd->s_ptype;
  bmel->r_node = msgd->r_node;
  bmel->r_ptype = msgd->r_ptype;
  bmel->pt_next = (buf_m_el *)0;
  bmel->pt_prev = (buf_m_el *)0;
  bmel->mt_next = (buf_m_el *)0;
  bmel->mt_prev = (buf_m_el *)0;

  return (bmel);
}


/*****************************************************************************/
int _del_msg
#ifdef ANSI
(buf_m_el *bme)
#else
(bme)
buf_m_el *bme;
#endif
/*****************************************************************************/
{
  /* free() message buffer only if the message is longer than 0 bytes */
  if (bme->msg_len > 0)
    free((void *)bme->msg_ptr);
  free((void *)bme);

  return (0);
}



/*****************************************************************************/
static int flush_msgs_pel
#ifdef ANSI
(long typesel, pt_tab_m_el *pel)
#else
(typesel, pel)
long typesel;
pt_tab_m_el *pel;
#endif
/*****************************************************************************/
{
  mt_tab_el *mtel;
  buf_m_el *bme;
  
  if (typesel != -1)
  {
    if ((mtel = find_mtel(typesel,pel)) == (mt_tab_el *)0)
      /* no message to this message type */
      return (0);
      
    /* Discard all messages of that type */
    for (bme = mtel->first; bme != (buf_m_el *)0; bme = mtel->first)
                                /* this is no infinite loop because
                                   mtel->first will be changed when the
                                   message is removed*/
    {
      if (_rm_msg(bme) < 0)
        return (-1);
      if (_del_msg(bme) < 0)
        return (-1);
    }

    return (0);
  }

  /* Discard all messages of that pel */
  for (bme = pel->first; bme != (buf_m_el *)0; bme = pel->first)
  {
    if (_rm_msg(bme) < 0)
      return (-1);
    if (_del_msg(bme) < 0)
      return (-1);
  }

  return (0);
}

/*****************************************************************************/
int _flush_msgs
#ifdef ANSI
(long typesel, long ptypesel)
#else
(typesel, ptypesel)
long typesel;
long ptypesel;
#endif
/*****************************************************************************/
{
  int i;
  pt_tab_m_el *pel;
  
  if (ptypesel != -1)
  {
    if ((pel = find_m_pel(ptypesel)) == (pt_tab_m_el *)0)
      /* No message to this ptype */
      return (0);

    if (flush_msgs_pel(typesel,pel) < 0)
      return (-1);

    return (0);
  }
  
  /* Check every entry in the hashtable for ptypes */
  for (i=0; i < PT_TAB_M_SIZE; i++)
    for (pel = ptt[i]; pel != (pt_tab_m_el *)0; pel = pel->bucket)
      if (flush_msgs_pel(typesel,pel) < 0)
        return (-1);

  return (0);
}

  




    
/*****************************************************************************/
int _conf_msgs
#ifdef ANSI
(void)
#else
()
#endif
/*****************************************************************************/
{
  pt_tab_m_el *pel;
  buf_m_el *bme;
  
  if ((pel = find_m_pel(__myptype())) == (pt_tab_m_el *)0)
    return (0);

  /* Look in the list of all messages */
  for (bme = pel->first; bme != (buf_m_el *)0; bme = bme->pt_next)
    if (bme->status == MSG_UNCONFIRMED)  {
      if (_confirm_msg(bme->s_node, bme->s_ptype) < 0)
        return (-1);

      bme->status = MSG_CONFIRMED;
    }    
  
  return (0);
}
