/******************************************************************************
*
*    Get addresses of processes
*    Copyright (C) 1993 A. Bode, S. Lamberts, T. Ludwig, G. Stellner
*
*    This file is part of NXLIB (Paragon(TM) message passing on workstations)
*
*    NXLIB is free software; you can redistribute it and/or
*    modify it under the terms of the GNU Library General Public
*    License as published by the Free Software Foundation; either
*    version 2 of the License, or (at your option) any later version.
*
*    NXLIB is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*    Library General Public License for more details.
*
*    You should have received a copy of the GNU Library General Public
*    License along with this library; if not, write to the Free
*    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*    Contact to the authors:
*
*    electronic mail: nxlib@informatik.tu-muenchen.de
*
*    paper mail:      Prof. Dr. A. Bode
*                     Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
*                     Institut f"ur Informatik
*                     Technische Universit"at M"unchen
*                     80290 M"unchen
*                     Germany
*
*    Paragon(TM) is a trademark of Intel Corporation.
*
******************************************************************************/
/******************************************************************************

  get_addr.c,v
  1994/12/30 16:33:07
  1.9
  Exp
  lamberts
 
  Authors: Stefan Lamberts

  Description:

  This file contains the code that is used for the address resolution.

  Available functions from this module:

  ....

******************************************************************************/
#ifndef lint
static char rcs_id[] = "get_addr.c,v 1.9 1994/12/30 16:33:07 lamberts Exp";
#endif

#include <errno.h>
#include <stdlib.h>
#include <string.h>

#ifdef SUN_OS4
#include <memory.h>
#endif

#include "../include/sys/nxerrno.h"
#include "../include/sys/nxlib.h"
#include "../include/nxmalloc.h"

extern long      __myhost();
extern long      __mynode();
extern long      __myptype();
extern p_listel *_find_plistel();
extern int       _rel_send();
extern void      _rel_wait_msg();
extern int       _update_plistel();
extern int       _buf_send();

static req_el *first_req = (req_el *)0;
static req_el *last_req = (req_el *)0;


/*****************************************************************************
 * Check wether the address information is available.
 * If not ask your daemon for the information.
 * A controlling process asks the appropriate daemon directly
 * returns
 *         1 if the address is known
 *         0 if the address is unknown
 *        -1 on error
 *****************************************************************************/
int _get_addr
#ifdef ANSI
(long node, long ptype)
#else
(node, ptype)
long node;
long ptype;
#endif
/*****************************************************************************/
{
  p_listel *plel;
  msg_addr_req msg;

  
  if ((plel = _find_plistel(node, ptype)) != (p_listel *)0)
  {
    switch (plel->loc)
    {
    case LOC_MYSELF:
      return (1);
    case LOC_REMOTE:
      if ((plel->radr.sockfd != UNDEF_SOCK) ||
          (plel->radr.stat == ADDR_SET))
        return (1);             /* address info exists */
      break;
    default:
      errno = ENOSYS;
      return (-1);
    }
  }
  
  /* Compose message */
  msg.rt_node = __mynode();
  msg.rt_ptype = __myptype();
  msg.f_node = __mynode();
  msg.f_ptype = __myptype();
  msg.r_node = node;
  msg.r_ptype = ptype;
  
  if(_rel_send((long)ADDR_REQ,
               (char *)&msg,
               (long)sizeof(msg_addr_req),
               (__mynode()==__myhost()) ? node : __mynode(), /* CP or AP */
               (long)DAEMON_PTYPE) < 0)
    return (-1);

  return (0);
}    
      

/*****************************************************************************
 * Handle a address reply from the daemon to an application process
 *****************************************************************************/
int _da_addr_repl
#ifdef ANSI
(msg_desc *msgd)
#else
(msgd)
msg_desc *msgd;
#endif
/*****************************************************************************/
{
  msg_addr_repl *msg;
  
  msg = (msg_addr_repl *)msgd->msg_ptr;
  
  if (_update_plistel(msg->r_node,
                      msg->r_ptype,
                      LOC_REMOTE,
                      &msg->radr,
                      (loc_addr *)0,
                      msg->hname,
                      msg->login,
                      msg->arch,
                      msg->pid) < 0)
    return (-1);

  /* send the message that is waiting for the address  */
  if (_buf_send(msg->f_ptype,msg->r_node,msg->r_ptype) < 0)
    return (-1);
  
  return (0);
}



/*****************************************************************************
 * Handle an address reply form a daemon to a daemon
 *****************************************************************************/
int _dd_addr_repl
#ifdef ANSI
(msg_desc *msgd)
#else
(msgd)
msg_desc *msgd;
#endif
/*****************************************************************************/
{
  msg_addr_repl *msg;
  
  msg = (msg_addr_repl *)msgd->msg_ptr;
  
  if (_update_plistel(msg->r_node,
                      msg->r_ptype,
                      LOC_REMOTE,
                      &msg->radr,
                      (loc_addr *)0,
                      msg->hname,
                      msg->login,
                      msg->arch,
                      msg->pid) < 0)
    return (-1);

  /* forward address info to the application process */
  if (_rel_send((long)ADDR_REPL,
                (char *)msg,
                (long)sizeof(msg_addr_repl),
                msg->f_node,
                msg->f_ptype) < 0)
    return (-1);

  return (0);
}



/*****************************************************************************/
static int ins_req
#ifdef ANSI
(long rt_node, long rt_ptype,
 long f_node, long f_ptype,
 long r_node, long r_ptype)
#else
(rt_node, rt_ptype, f_node, f_ptype, r_node, r_ptype)
long rt_node;
long rt_ptype;
long f_node;
long f_ptype;
long r_node;
long r_ptype;
#endif
/*****************************************************************************/
{
  req_el *re;
  
  if ((re = (req_el *)malloc(sizeof(req_el))) == (req_el *)0)
    return (-1);
  
  re->rt_node = rt_node;
  re->rt_ptype = rt_ptype;
  re->f_node = f_node;
  re->f_ptype = f_ptype;
  re->r_node = r_node;
  re->r_ptype = r_ptype;

  if (last_req == (req_el *)0)
  {
    first_req = re;
    last_req = re;
    re->next = (req_el *)0;
    re->prev = (req_el *)0;
    return (0);
  }

  last_req->next = re;
  re->prev = last_req;
  re->next = (req_el *)0;
  last_req = re;

  return (0);
}
  


/*****************************************************************************/
static int rm_req
#ifdef ANSI
(req_el *re)
#else
(re)
req_el *re;
#endif
/*****************************************************************************/
{
  req_el *prev;
  req_el *next;

  prev = re->prev;
  next = re->next;
  
  if (next == (req_el *)0)
    last_req = prev;
  else
    next->prev = prev;

  if (prev == (req_el *)0)
    first_req = next;
  else
    prev->next = next;

  free((void *)re);
  
  return (0);
}



/****************************************************************************/
static int snd_addr_repl
#ifdef ANSI
(long rt_node, long rt_ptype,
 long f_node, long f_ptype,
 long r_node, long r_ptype,
 p_listel *pe)
#else
(rt_node, rt_ptype, f_node, f_ptype, r_node, r_ptype, pe)
long rt_node;
long rt_ptype;
long f_node;
long f_ptype;
long r_node;
long r_ptype;
p_listel *pe;
#endif
/*****************************************************************************/
{
  msg_addr_repl repl;
  
  /* Compose message */
  repl.f_node = f_node;
  repl.f_ptype = f_ptype;
  repl.r_node = r_node;
  repl.r_ptype = r_ptype;
  repl.radr.stat = ADDR_SET;
  repl.radr.sockfd = UNDEF_SOCK;
  memcpy((void *)&repl.radr.sad,
         (void *)&pe->radr.sad,
         sizeof(struct sockaddr_in));
  repl.ladr = 0;
  strcpy(repl.hname,pe->hname);
  strcpy(repl.login,pe->login);
  strcpy(repl.arch,pe->arch);
  repl.pid = pe->pid;

  if (_rel_send((long)ADDR_REPL,
                (char *)&repl,
                (long)sizeof(msg_addr_repl),
                rt_node,
                rt_ptype) < 0)
    return (-1);

  return (0);
}





/*****************************************************************************/
int _answer_requests
#ifdef ANSI
(void)
#else
()
#endif
/*****************************************************************************/
{
  req_el *re;
  req_el *next;
  p_listel *pe;
  
  for (re = first_req; re != (req_el *)0; re = next)
  {
    next = re->next;

    if (((pe = _find_plistel(re->r_node,re->r_ptype)) != (p_listel *)0) &&
        (pe->radr.stat == ADDR_SET))
    {
      if (snd_addr_repl(re->rt_node, re->rt_ptype,
                        re->f_node, re->f_ptype,
                        re->r_node, re->r_ptype,
                        pe) < 0)
        return (-1);

      /* Remove and delete call */
      if (rm_req(re) < 0)
        return (-1);
    }
  }

  return (0);
}




/*****************************************************************************
 * Handle an address request
 *****************************************************************************/
int _addr_req
#ifdef ANSI
(msg_desc *msgd)
#else
(msgd)
msg_desc *msgd;
#endif
/*****************************************************************************/
{
  msg_addr_req *msg;
  p_listel *pe;
  
  msg = (msg_addr_req *)msgd->msg_ptr;
  
  if (((pe = _find_plistel(msg->r_node,msg->r_ptype)) != (p_listel *)0) &&
      (pe->radr.stat == ADDR_SET))
  {
    if (snd_addr_repl(msg->rt_node, msg->rt_ptype,
                      msg->f_node, msg->f_ptype,
                      msg->r_node, msg->r_ptype,
                      pe) < 0)
      return (-1);
  }
  else if (msg->r_node == __mynode())
  {
    /* Store request inorder to answer it as soon as the address is known */
    if (ins_req(msg->rt_node, msg->rt_ptype,
                 msg->f_node, msg->f_ptype,
                 msg->r_node, msg->r_ptype) < 0)
      return (-1);
  }
  else 
  {
    /* Forward the request to the appropriate daemon */
    msg->rt_ptype = DAEMON_PTYPE;
    if (_rel_send((long)ADDR_REQ,
                  (char *)msg,
                  (long)sizeof(msg_addr_req),
                  msg->r_node,
                  (long)DAEMON_PTYPE) < 0)
    return (-1);
  }
  
  return (0);
}


