/* Character classification and correspondance.
   Copyright (C) 1990 Free Software Foundation, Inc.
   Francois Pinard <pinard@iro.umontreal.ca>, 1988.
*/

/*

These defines replace what is usually <ctype.h>.  They take into account
local character set and possibly 8-bits characters.  There is also a few
defines for foreigh language processing.

These defines accept signed characters, unsigned characters or int's having
positive values.  Also, the `is...' defines accept and understand EOF.

PLEASE NOTE, however, that the signed character '\377' is quite dangerous,
because several of the following defines will interpret it as EOF.

*/

#define _U 0x01			/* upper case character */
#define _L 0x02			/* lower case character */
#define _N 0x04			/* numeric character */
#define _S 0x08			/* space character */
#define _P 0x10			/* punctuation character */
#define _C 0x20			/* control character */
#define _B 0x40			/* space character */
#define _X 0x80			/* hexadecimal character */

/* Table of set of attribute for each character.  */
extern char _ctype_set[];

#ifdef MSDOS
extern char _ctype_lower[];	/* lower case equivalents */
extern char _ctype_upper[];	/* upper case equivalents */
#endif
extern char _ctype_base[];	/* characters without diacritics */
extern char _ctype_diac[];	/* diacritic code for each character */

/* Returns non zero if c is alphabetic.  */
#define isalpha(c) \
  ((c) == EOF ? 0 : (_ctype_set[(unsigned char) c] & (_U | _L)))

/* Returns non zero if c is upper case.  */
#define isupper(c) \
  ((c) == EOF ? 0 : (_ctype_set[(unsigned char) c] & _U))

/* Returns non zero if c is lower case.  */
#define islower(c) \
  ((c) == EOF ? 0 : (_ctype_set[(unsigned char) c] & _L))

/* Returns non zero if c is decimal digit (0-9).  */
#define isdigit(c) \
  ((c) == EOF ? 0 : (_ctype_set[(unsigned char) c] & _N))

/* Returns non zero if c is hexadecimal digit (0-9, a-f, A-F).  */
#define isxdigit(c) \
  ((c) == EOF ? 0 : (_ctype_set[(unsigned char) c] & _X))

/* Returns non zero if c is is white space.  */
#define isspace(c) \
  ((c) == EOF ? 0 : (_ctype_set[(unsigned char) c] & _S))

/* Returns non zero if c is punctuation.  */
#define ispunct(c) \
  ((c) == EOF ? 0 : (_ctype_set[(unsigned char) c] & _P))

/* Returns non zero if c is alphanumeric.  */
#define isalnum(c) \
  ((c) == EOF ? 0 : (_ctype_set[(unsigned char) c] & (_U | _L | _N)))

/* Returns non zero if c is printable, (space included).  */
#define isprint(c) \
  ((c) == EOF ? 0 : (_ctype_set[(unsigned char) c] & (_P | _U | _L | _N | _B)))

/* Returns non zero if c is grafical, (space excluded).  */
#define isgraph(c) \
  ((c) == EOF ? 0 : (_ctype_set[(unsigned char) c] & (_P | _U | _L | _N)))

/* Returns non zero if c is control character.  */
#define iscntrl(c) \
  ((c) == EOF ? 0 : (_ctype_set[(unsigned char) c] & _C))

/* Returns non zero if c is ASCII character, that is, 7 bits.  */
#define isascii(c) \
  ((c) == EOF ? 0 : (!((c) & ~0177)))

/* Returns the upper case value of c, already known to be alphabetic.  */
#ifdef MSDOS
#define toupper(c)   (_ctype_upper[(unsigned char) c])
#else
#define toupper(c)   (islower(c) ? ((c) & ~040) : (c))
#endif

/* Returns the lower case value of c, already known to be alphabetic.  */
#ifdef MSDOS
#define tolower(c)   (_ctype_lower[(unsigned char) c])
#else
#define tolower(c)   (isupper(c) ? ((c) | 040) : (c))
#endif

/* Returns c without its diacritic.  c is known to be alphabetic.  */
#define tobase(c)     (_ctype_base[(unsigned char) c])

/* Returns a diacritic code for c.  c is known to be alphabetic.  */
#define todiac(c)   (_ctype_diac[(unsigned char) c])

