/*
*		file: speed.c
*
*   MPN Low Level Speed Test
*   no check is performed for correctness
*
*   Copyright (C) 1998 by Alex Stuebinger
*
*/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

#include "gmp.h"
#include "gmp-impl.h"
#include "longlong.h"

/* Configurable Section */
#ifndef CPU_CLOCK
#define CPU_CLOCK 20000000 	/* in HZ */
#endif

#ifndef LOOPS
#define LOOPS 100    	/* Iterations/10 */
#endif

const double iterations = LOOPS*10;

FILE * out = 0;


/* Reference Routines */

/* Cool population count of a mp_limb_t.
   You have to figure out how this works, I won't tell you!  */

static inline unsigned int
popc_limb (x)
     mp_limb_t x;
{
#if BITS_PER_MP_LIMB == 64
  /* We have to go into some trouble to define these constants.
     (For mp_limb_t being `long long'.)  */
  mp_limb_t cnst;
  cnst = 0x55555555L | ((mp_limb_t) 0x55555555L << BITS_PER_MP_LIMB/2);
  x = ((x & ~cnst) >> 1) + (x & cnst);
  cnst = 0x33333333L | ((mp_limb_t) 0x33333333L << BITS_PER_MP_LIMB/2);
  x = ((x & ~cnst) >> 2) + (x & cnst);
  cnst = 0x0f0f0f0fL | ((mp_limb_t) 0x0f0f0f0fL << BITS_PER_MP_LIMB/2);
  x = ((x >> 4) + x) & cnst;
  x = ((x >> 8) + x);
  x = ((x >> 16) + x);
  x = ((x >> 32) + x) & 0xff;
#endif
#if BITS_PER_MP_LIMB == 32
  x = ((x >> 1) & 0x55555555L) + (x & 0x55555555L);
  x = ((x >> 2) & 0x33333333L) + (x & 0x33333333L);
  x = ((x >> 4) + x) & 0x0f0f0f0fL;
  x = ((x >> 8) + x);
  x = ((x >> 16) + x) & 0xff;
#endif
  return x;
}

unsigned long int
#if __STDC__
refmpn_popcount (register mp_srcptr p, register mp_size_t size)
#else
mpn_popcount (p, size)
     register mp_srcptr p;
     register mp_size_t size;
#endif
{
  unsigned long int popcnt;
  mp_size_t i;

  popcnt = 0;
  for (i = 0; i < size; i++)
    popcnt += popc_limb (p[i]);

  return popcnt;
}

mp_limb_t
refmpn_lshift (wp, up, usize, cnt)
     register mp_ptr wp;
     register mp_srcptr up;
     mp_size_t usize;
     register unsigned int cnt;
{
  register mp_limb_t high_limb, low_limb;
  register unsigned sh_1, sh_2;
  register mp_size_t i;
  mp_limb_t retval;

#ifdef DEBUG
  if (usize == 0 || cnt == 0)
    abort ();
#endif

  sh_1 = cnt;
#if 0
  if (sh_1 == 0)
    {
      if (wp != up)
	{
	  /* Copy from high end to low end, to allow specified input/output
	     overlapping.  */
	  for (i = usize - 1; i >= 0; i--)
	    wp[i] = up[i];
	}
      return 0;
    }
#endif

  wp += 1;
  sh_2 = BITS_PER_MP_LIMB - sh_1;
  i = usize - 1;
  low_limb = up[i];
  retval = low_limb >> sh_2;
  high_limb = low_limb;
  while (--i >= 0)
    {
      low_limb = up[i];
      wp[i] = (high_limb << sh_1) | (low_limb >> sh_2);
      high_limb = low_limb;
    }
  wp[i] = high_limb << sh_1;

  return retval;
}

mp_limb_t
refmpn_rshift (wp, up, usize, cnt)
     register mp_ptr wp;
     register mp_srcptr up;
     mp_size_t usize;
     register unsigned int cnt;
{
  register mp_limb_t high_limb, low_limb;
  register unsigned sh_1, sh_2;
  register mp_size_t i;
  mp_limb_t retval;

#ifdef DEBUG
  if (usize == 0 || cnt == 0)
    abort ();
#endif

  sh_1 = cnt;
#if 0
  if (sh_1 == 0)
    {
      if (wp != up)
	{
	  /* Copy from low end to high end, to allow specified input/output
	     overlapping.  */
	  for (i = 0; i < usize; i++)
	    wp[i] = up[i];
	}
      return 0;
    }
#endif

  wp -= 1;
  sh_2 = BITS_PER_MP_LIMB - sh_1;
  high_limb = up[0];
  retval = high_limb << sh_2;
  low_limb = high_limb;

  for (i = 1; i < usize; i++)
    {
      high_limb = up[i];
      wp[i] = (low_limb >> sh_1) | (high_limb << sh_2);
      low_limb = high_limb;
    }
  low_limb >>= sh_1;
  wp[i] = low_limb;

  return retval;
}



mp_limb_t
#if __STDC__
refmpn_add_n (mp_ptr res_ptr,
	       mp_srcptr s1_ptr, mp_srcptr s2_ptr, mp_size_t size)
#else
refmpn_add_n (res_ptr, s1_ptr, s2_ptr, size)
     register mp_ptr res_ptr;
     register mp_srcptr s1_ptr;
     register mp_srcptr s2_ptr;
     mp_size_t size;
#endif
{
  register mp_limb_t x, y, cy;
  register mp_size_t j;

  /* The loop counter and index J goes from -SIZE to -1.  This way
     the loop becomes faster.  */
  j = -size;

  /* Offset the base pointers to compensate for the negative indices.  */
  s1_ptr -= j;
  s2_ptr -= j;
  res_ptr -= j;

  cy = 0;
  do
    {
      y = s2_ptr[j];
      x = s1_ptr[j];
      y += cy;			/* add previous carry to one addend */
      cy = (y < cy);		/* get out carry from that addition */
      y = x + y;		/* add other addend */
      cy = (y < x) + cy;	/* get out carry from that add, combine */
      res_ptr[j] = y;
    }
  while (++j != 0);

  return cy;
}


mp_limb_t
#if __STDC__
refmpn_sub_n (mp_ptr res_ptr,
	       mp_srcptr s1_ptr, mp_srcptr s2_ptr, mp_size_t size)
#else
refmpn_sub_n (res_ptr, s1_ptr, s2_ptr, size)
     register mp_ptr res_ptr;
     register mp_srcptr s1_ptr;
     register mp_srcptr s2_ptr;
     mp_size_t size;
#endif
{
  register mp_limb_t x, y, cy;
  register mp_size_t j;

  /* The loop counter and index J goes from -SIZE to -1.  This way
     the loop becomes faster.  */
  j = -size;

  /* Offset the base pointers to compensate for the negative indices.  */
  s1_ptr -= j;
  s2_ptr -= j;
  res_ptr -= j;

  cy = 0;
  do
    {
      y = s2_ptr[j];
      x = s1_ptr[j];
      y += cy;			/* add previous carry to subtrahend */
      cy = (y < cy);		/* get out carry from that addition */
      y = x - y;		/* main subtract */
      cy = (y > x) + cy;	/* get out carry from the subtract, combine */
      res_ptr[j] = y;
    }
  while (++j != 0);

  return cy;
}

mp_limb_t
refmpn_mul_1 (res_ptr, s1_ptr, s1_size, s2_limb)
     register mp_ptr res_ptr;
     register mp_srcptr s1_ptr;
     mp_size_t s1_size;
     register mp_limb_t s2_limb;
{
  register mp_limb_t cy_limb;
  register mp_size_t j;
  register mp_limb_t prod_high, prod_low;

  /* The loop counter and index J goes from -S1_SIZE to -1.  This way
     the loop becomes faster.  */
  j = -s1_size;

  /* Offset the base pointers to compensate for the negative indices.  */
  s1_ptr -= j;
  res_ptr -= j;

  cy_limb = 0;
  do
    {
      umul_ppmm (prod_high, prod_low, s1_ptr[j], s2_limb);

      prod_low += cy_limb;
      cy_limb = (prod_low < cy_limb) + prod_high;

      res_ptr[j] = prod_low;
    }
  while (++j != 0);

  return cy_limb;
}

mp_limb_t
refmpn_addmul_1 (res_ptr, s1_ptr, s1_size, s2_limb)
     register mp_ptr res_ptr;
     register mp_srcptr s1_ptr;
     mp_size_t s1_size;
     register mp_limb_t s2_limb;
{
  register mp_limb_t cy_limb;
  register mp_size_t j;
  register mp_limb_t prod_high, prod_low;
  register mp_limb_t x;

  /* The loop counter and index J goes from -SIZE to -1.  This way
     the loop becomes faster.  */
  j = -s1_size;

  /* Offset the base pointers to compensate for the negative indices.  */
  res_ptr -= j;
  s1_ptr -= j;

  cy_limb = 0;
  do
    {
      umul_ppmm (prod_high, prod_low, s1_ptr[j], s2_limb);

      prod_low += cy_limb;
      cy_limb = (prod_low < cy_limb) + prod_high;

      x = res_ptr[j];
      prod_low = x + prod_low;
      cy_limb += (prod_low < x);
      res_ptr[j] = prod_low;
    }
  while (++j != 0);

  return cy_limb;
}

mp_limb_t
refmpn_submul_1 (res_ptr, s1_ptr, s1_size, s2_limb)
     register mp_ptr res_ptr;
     register mp_srcptr s1_ptr;
     mp_size_t s1_size;
     register mp_limb_t s2_limb;
{
  register mp_limb_t cy_limb;
  register mp_size_t j;
  register mp_limb_t prod_high, prod_low;
  register mp_limb_t x;

  /* The loop counter and index J goes from -SIZE to -1.  This way
     the loop becomes faster.  */
  j = -s1_size;

  /* Offset the base pointers to compensate for the negative indices.  */
  res_ptr -= j;
  s1_ptr -= j;

  cy_limb = 0;
  do
    {
      umul_ppmm (prod_high, prod_low, s1_ptr[j], s2_limb);

      prod_low += cy_limb;
      cy_limb = (prod_low < cy_limb) + prod_high;

      x = res_ptr[j];
      prod_low = x - prod_low;
      cy_limb += (prod_low > x);
      res_ptr[j] = prod_low;
    }
  while (++j != 0);

  return cy_limb;
}

double gettimer(void)
{	return (double) clock()/ (double) (CLOCKS_PER_SEC);
}

void *xmalloc(size_t size)
{	void *p = malloc(size);
	if (p != 0 ) return p;
	else {
		fprintf(stderr, "no memory for size %d\n", size);
		exit(EXIT_FAILURE);
	}
}

int dummy(int x)
{ return x + 1;
}

void do_timings(int size)
{  mp_ptr a = xmalloc(sizeof(mp_limb_t) * size);
	 mp_ptr b = xmalloc(sizeof(mp_limb_t) * size);
	 mp_ptr c = xmalloc(sizeof(mp_limb_t) * size);
	 mp_limb_t limb;
	 double start, tyme, loop, cycles;
	 int i, y;

   if (((unsigned)a)&3 != 0 || ((unsigned)b)&3 != 0 || ((unsigned)c)&3 != 0 )
     fprintf(out, "misaligned\n");

	 fprintf(out, "=======================================================\n");
	 fprintf(out, "size = %d\n", size);
	 fprintf(out, "=======================================================\n");

	 start = gettimer();
   for (i = 1; i <= iterations; ++i);
	 loop = gettimer() - start;          /* loop overhead */
   loop = loop < 0 ? 0.0: loop;

   mpn_random2(a, size);

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
   {  refmpn_popcount(a, size);
      refmpn_popcount(a, size);
      refmpn_popcount(a, size);
      refmpn_popcount(a, size);
      refmpn_popcount(a, size);

      refmpn_popcount(a, size);
      refmpn_popcount(a, size);
      refmpn_popcount(a, size);
      refmpn_popcount(a, size);
      refmpn_popcount(a, size);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
   fprintf(out, "refmpn_popcount: \t%10.2f cycles/limb\n", cycles);

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
   {  mpn_popcount(a, size);
      mpn_popcount(a, size);
      mpn_popcount(a, size);
      mpn_popcount(a, size);
      mpn_popcount(a, size);

      mpn_popcount(a, size);
      mpn_popcount(a, size);
      mpn_popcount(a, size);
      mpn_popcount(a, size);
      mpn_popcount(a, size);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
   fprintf(out, "mpn_popcount: \t\t%10.2f cycles/limb\n", cycles);


   mpn_random2(a, size);

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	refmpn_lshift(c, a, size, 1);
			refmpn_lshift(c, a, size, 2);
			refmpn_lshift(c, a, size, 3);
			refmpn_lshift(c, a, size, 4);
			refmpn_lshift(c, a, size, 5);

			refmpn_lshift(c, a, size, 1);
			refmpn_lshift(c, a, size, 2);
			refmpn_lshift(c, a, size, 3);
			refmpn_lshift(c, a, size, 4);
			refmpn_lshift(c, a, size, 5);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "refmpn_lshift: \t\t%10.2f cycles/limb\n", cycles);

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	mpn_lshift(c, a, size, 1);
			mpn_lshift(c, a, size, 2);
			mpn_lshift(c, a, size, 3);
			mpn_lshift(c, a, size, 4);
			mpn_lshift(c, a, size, 5);

			mpn_lshift(c, a, size, 1);
			mpn_lshift(c, a, size, 2);
			mpn_lshift(c, a, size, 3);
			mpn_lshift(c, a, size, 4);
			mpn_lshift(c, a, size, 5);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "mpn_lshift: \t\t%10.2f cycles/limb\n", cycles);


	 mpn_random2(a, size);

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	refmpn_rshift(c, a, size, 1);
			refmpn_rshift(c, a, size, 2);
			refmpn_rshift(c, a, size, 3);
			refmpn_rshift(c, a, size, 4);
			refmpn_rshift(c, a, size, 5);

			refmpn_rshift(c, a, size, 1);
			refmpn_rshift(c, a, size, 2);
			refmpn_rshift(c, a, size, 3);
			refmpn_rshift(c, a, size, 4);
			refmpn_rshift(c, a, size, 5);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "refmpn_rshift: \t\t%10.2f cycles/limb\n", cycles);

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	mpn_rshift(c, a, size, 1);
			mpn_rshift(c, a, size, 2);
			mpn_rshift(c, a, size, 3);
			mpn_rshift(c, a, size, 4);
			mpn_rshift(c, a, size, 5);

			mpn_rshift(c, a, size, 1);
			mpn_rshift(c, a, size, 2);
			mpn_rshift(c, a, size, 3);
			mpn_rshift(c, a, size, 4);
			mpn_rshift(c, a, size, 5);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "mpn_rshift: \t\t%10.2f cycles/limb\n", cycles);


	 mpn_random2(a, size);
	 mpn_random2(b, size);

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	refmpn_add_n(c, a, b, size);
			refmpn_add_n(c, a, b, size);
			refmpn_add_n(c, a, b, size);
			refmpn_add_n(c, a, b, size);
			refmpn_add_n(c, a, b, size);

			refmpn_add_n(c, a, b, size);
			refmpn_add_n(c, a, b, size);
			refmpn_add_n(c, a, b, size);
			refmpn_add_n(c, a, b, size);
			refmpn_add_n(c, a, b, size);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "refmpn_add_n: \t\t%10.2f cycles/limb\n", cycles);

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	mpn_add_n(c, a, b, size);
			mpn_add_n(c, a, b, size);
			mpn_add_n(c, a, b, size);
			mpn_add_n(c, a, b, size);
			mpn_add_n(c, a, b, size);

			mpn_add_n(c, a, b, size);
			mpn_add_n(c, a, b, size);
			mpn_add_n(c, a, b, size);
			mpn_add_n(c, a, b, size);
			mpn_add_n(c, a, b, size);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "mpn_add_n: \t\t%10.2f cycles/limb\n", cycles);


	 mpn_random2(a, size);
	 mpn_random2(b, size);

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	refmpn_sub_n(c, a, b, size);
			refmpn_sub_n(c, a, b, size);
			refmpn_sub_n(c, a, b, size);
			refmpn_sub_n(c, a, b, size);
			refmpn_sub_n(c, a, b, size);

			refmpn_sub_n(c, a, b, size);
			refmpn_sub_n(c, a, b, size);
			refmpn_sub_n(c, a, b, size);
			refmpn_sub_n(c, a, b, size);
			refmpn_sub_n(c, a, b, size);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "refmpn_sub_n: \t\t%10.2f cycles/limb\n", cycles);

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	mpn_sub_n(c, a, b, size);
			mpn_sub_n(c, a, b, size);
			mpn_sub_n(c, a, b, size);
			mpn_sub_n(c, a, b, size);
			mpn_sub_n(c, a, b, size);

			mpn_sub_n(c, a, b, size);
			mpn_sub_n(c, a, b, size);
			mpn_sub_n(c, a, b, size);
			mpn_sub_n(c, a, b, size);
			mpn_sub_n(c, a, b, size);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "mpn_sub_n: \t\t%10.2f cycles/limb\n", cycles);


	 mpn_random2(a, size);
	 limb = 0x12345678;

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	refmpn_mul_1(c, a, size, limb);
			refmpn_mul_1(c, a, size, limb);
			refmpn_mul_1(c, a, size, limb);
			refmpn_mul_1(c, a, size, limb);
			refmpn_mul_1(c, a, size, limb);

			refmpn_mul_1(c, a, size, limb);
			refmpn_mul_1(c, a, size, limb);
			refmpn_mul_1(c, a, size, limb);
			refmpn_mul_1(c, a, size, limb);
			refmpn_mul_1(c, a, size, limb);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "refmpn_mul_1: \t\t%10.2f cycles/limb\n", cycles);


	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	mpn_mul_1(c, a, size, limb);
			mpn_mul_1(c, a, size, limb);
			mpn_mul_1(c, a, size, limb);
			mpn_mul_1(c, a, size, limb);
			mpn_mul_1(c, a, size, limb);

			mpn_mul_1(c, a, size, limb);
			mpn_mul_1(c, a, size, limb);
			mpn_mul_1(c, a, size, limb);
			mpn_mul_1(c, a, size, limb);
			mpn_mul_1(c, a, size, limb);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "mpn_mul_1: \t\t%10.2f cycles/limb\n", cycles);


	 mpn_random2(a, size);
	 mpn_random2(c, size);
	 limb = 0x12345678;

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	refmpn_addmul_1(c, a, size, limb);
			refmpn_addmul_1(c, a, size, limb);
			refmpn_addmul_1(c, a, size, limb);
			refmpn_addmul_1(c, a, size, limb);
			refmpn_addmul_1(c, a, size, limb);

			refmpn_addmul_1(c, a, size, limb);
			refmpn_addmul_1(c, a, size, limb);
			refmpn_addmul_1(c, a, size, limb);
			refmpn_addmul_1(c, a, size, limb);
			refmpn_addmul_1(c, a, size, limb);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "refmpn_addmul_1: \t%10.2f cycles/limb\n", cycles);

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	mpn_addmul_1(c, a, size, limb);
			mpn_addmul_1(c, a, size, limb);
			mpn_addmul_1(c, a, size, limb);
			mpn_addmul_1(c, a, size, limb);
			mpn_addmul_1(c, a, size, limb);

			mpn_addmul_1(c, a, size, limb);
			mpn_addmul_1(c, a, size, limb);
			mpn_addmul_1(c, a, size, limb);
			mpn_addmul_1(c, a, size, limb);
			mpn_addmul_1(c, a, size, limb);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "mpn_addmul_1: \t\t%10.2f cycles/limb\n", cycles);


	 mpn_random2(a, size);
	 mpn_random2(c, size);
	 limb = 0x12345678;

	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	refmpn_submul_1(c, a, size, limb);
			refmpn_submul_1(c, a, size, limb);
			refmpn_submul_1(c, a, size, limb);
			refmpn_submul_1(c, a, size, limb);
			refmpn_submul_1(c, a, size, limb);

			refmpn_submul_1(c, a, size, limb);
			refmpn_submul_1(c, a, size, limb);
			refmpn_submul_1(c, a, size, limb);
			refmpn_submul_1(c, a, size, limb);
			refmpn_submul_1(c, a, size, limb);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "refmpn_submul_1: \t%10.2f cycles/limb\n", cycles);


	 start = gettimer();
	 for (i = 1; i <= LOOPS; ++i)
	 {	mpn_submul_1(c, a, size, limb);
			mpn_submul_1(c, a, size, limb);
			mpn_submul_1(c, a, size, limb);
			mpn_submul_1(c, a, size, limb);
			mpn_submul_1(c, a, size, limb);

			mpn_submul_1(c, a, size, limb);
			mpn_submul_1(c, a, size, limb);
			mpn_submul_1(c, a, size, limb);
			mpn_submul_1(c, a, size, limb);
			mpn_submul_1(c, a, size, limb);
	 }
	 tyme = gettimer() - loop - start;
	 cycles = CPU_CLOCK * tyme / ((double) size * iterations);
	 fprintf(out, "mpn_submul_1: \t\t%10.2f cycles/limb\n", cycles);

   fprintf(out, "=======================================================\n\n\n");



	 free(a);
	 free(b);
	 free(c);
}


int main(void)
{
	out = fopen("report.txt", "w");
	do_timings(10);
	do_timings(30);
	do_timings(100);
	do_timings(300);
  /*
	do_timings(1000);
	do_timings(3000);
	*/
	fclose(out);
	exit(EXIT_SUCCESS);
}
