{
Title:   LIBRARY S708Drv
Summary: INMOS S708 link driver for WIServer.
Ref.:    23.03.93.01.mjm
Author:  mjm
Version: 1.1

-----------------------------------------------------------------------------------------------

Change history:

0.0      Under construction
1.0      First working version. 5/6/93
24/6/93  Attempt to detect and report link contention.
----------------------------------------------------------------------------------------------

Associated documentation:
Ref.                  Title
18.01.93.09.mjm       WIServer software description
18.01.93.01.mjm       WIServer (Windows IServer)
18.01.93.07.mjm       UNIT LinkInterface
18.01.93.08.mjm       WIServer user manual


----------------------------------------------------------------------------------------------


Known problems/bugs to sort out:

1: The link driver does not protect against another WIServer instance already using the
   link. An error is generated if two WIServer instances are generated using the same link.
   Also there will be problems if two WIServer instances use different links. This is because
   there is only one copy of LinkErrorValue which gets shared between instances. Hence, if
   there is an error generated in one instance it will also appear in another instance.


----------------------------------------------------------------------------------------------

Possible future improvements:

1:

----------------------------------------------------------------------------------------------

Introduction

This library is a link driver for WIServer using the standard Inmos S708 link driver.

}



library S708DRV;

{$D-,G+,R-,S-,W-} {No debug info,
                   286 code generated,
                   No range checking,
                   no stack overflow checking,
                   Protected mode only}

uses
  WinProcs, WinTypes, WinDos;

const
  NULL_LINK_HANDLE       = -1;     {Used to identify a bad link handle.}
  LE_OK                  = 0;      {Link error codes}
  LE_TRANSPUTER_ERROR    = 1;
  LE_LINK_OUTPUT_SLOW    = 2;
  LE_LINK_INPUT_SLOW     = 3;
  LE_LINK_IN_USE         = 4;
  LE_UNKNOWN_LINK_HANDLE = 5;
  LE_LINK_IO_SLOW        = 6;
  LE_UNKNOWN_LINK        = 7;

var
  LinkErrorValue: Word;    {Stores the current link error value.}
  ErrorFlagMask: Word;     {Decides if Transputer error is checked. 0 = no checking.}

function Device_read (LinkHandle: Integer): Word;
  {Uses a DOS call to get information about the device.}
  var
    t: Word;
  begin
    asm
      mov ax, $4400
      mov bx, LinkHandle
    end;
    Dos3Call;
    asm
      mov t, dx
    end;
    Device_read := t;
  end;

procedure Device_write (LinkHandle: Integer; write_value: Word);
  {Uses a DOS call to set the device status. (ASCII to Binary mode)}
  begin
    asm
      mov ax, $4401
      mov bx, LinkHandle
      mov dx, write_value
    end;
    Dos3Call;
  end;

function IOCTLread (LinkHandle: Integer): LongInt;
  {Used to read the Transputer Error status.}
  var
    a, b: Word;
    t: LongInt;
    r: Word;
  begin
    a := Ofs (t);
    b := Seg (t);
    asm
      push ds
      mov ax, b
      mov ds, ax
      mov dx, a
      mov ax, $4402
      mov bx, LinkHandle
      mov cx, $04
    end;
    Dos3Call;
    asm
      pop ds
      mov r, ax
    end;
    IOCTLread := t;
  end;

procedure IOCTLwrite (LinkHandle: Integer; IOCTL_oper: LongInt);
  {Used to assert analyse and reset.}
  var
    a, b, r: Word;
  begin
    a := Ofs (IOCTL_oper);
    b := Seg (IOCTL_oper);
    asm
      push ds
      mov ax, b
      mov ds, ax
      mov dx, a
      mov bx, LinkHandle
      mov cx, $04
      mov ax, $4403
    end;
    Dos3Call;
    asm
      pop ds
      mov r, ax
    end;
  end;

function lwrite (LinkHandle: Integer; pbuf: Pointer; count: Word): Word;
  {Used to send data to the link using a DOS call.}
  type
    Trec = record
             a1: Word;
             a2: Word;
           end;
  var
    p: Trec;
    r: Word;
  begin
    p := Trec (pbuf);
    asm
      push ds
      mov ax, p.a2
      mov ds, ax
      mov dx, p.a1
      mov bx, LinkHandle
      mov cx, count
      mov ax, $4000
    end;
    Dos3Call;
    asm
      pop ds
      mov r, ax
    end;
    lwrite := r;
  end;

function lread (LinkHandle: Integer; pbuf: Pointer; count: Word): Word;
  {Used to get data from the link using a DOS call.}
  type
    Trec = record
             a1: Word;
             a2: Word;
           end;
  var
    p: Trec;
    r: Word;
  begin
    p := Trec (pbuf);
    asm
      push ds
      mov ax, p.a2
      mov ds, ax
      mov dx, p.a1
      mov bx, LinkHandle
      mov cx, count
      mov ax, $3F00
    end;
    Dos3Call;
    asm
      pop ds
      mov r, ax
    end;
    lread := r;
  end;

function LinkOutputReady (LinkHandle: Integer): Byte; export;
  {If returns non zero then link is expecting input.}
  begin
    LinkOutputReady := IOCTLread (LinkHandle) and 4;
  end; {function LinkOutputReady ---------------------------------------------------------------------------------------------}


function LinkInputReady (LinkHandle: Integer): Byte; export;
  {If returns non zero then link has sent a byte.}
  begin
    LinkInputReady := IOCTLread (LinkHandle) and 8;
  end; {function LinkInputReady ----------------------------------------------------------------------------------------------}


function TransputerError (LinkHandle: Integer): Byte; export;
  {Checks transputer error flag if ErrorFlagMask is zero.}
  {Returns non zero if error pin is set.}
  begin
    if ErrorFlagMask = 0 then
      TransputerError := IOCTLread (LinkHandle) and 1
    else
      TransputerError := 0;
  end; {function TransputerError ---------------------------------------------------------------------------------------------}

function LinkError (LinkHandle: Integer): Word; export;
  {Returns the current error and tests the transputer error pin.}
  begin
    if TransputerError (LinkHandle) <> 0 then
      if ErrorFlagMask = 0 then
        LinkErrorValue := LE_TRANSPUTER_ERROR;
    LinkError := LinkErrorValue;
  end; {function LinkError ---------------------------------------------------------------------------------------------------}

procedure SendBlock (PBlock: Pointer; count, LinkHandle: Integer); export;
  {Sends a block of memory to the link, checking for errors as sent out.}
  {PBlock points to the block to get sent,
   count hold the size of the block in bytes,
   LinkHandle is the link identifier.}
  type
    Buffer  = array[0..$FFFE] of Byte;
  var
    PBuffer: ^Buffer;
    result: Word;
  begin
    IOCTLwrite (LinkHandle, $00100002);
    result := lwrite (Integer (LinkHandle), PBlock, count);
    if result <> count then
      if LinkError (LinkHandle) = LE_OK then
        LinkErrorValue := LE_LINK_OUTPUT_SLOW;
  end; {procedure SendBlock --------------------------------------------------------------------------------------------------}


procedure GetBlock (PBlock: Pointer; count, LinkHandle: Integer); export;
  {Gets a block of memory from the link, checking for errors as recieved.}
  {PBlock points to a block of memory used to store,
   count hold the size of the block to get in bytes,
   LinkHandle is the link identifier.}
  type
    Buffer  = array[0..$FFFE] of Byte;
  var
    PBuffer: ^Buffer;
    result: Word;
  begin
    IOCTLwrite (LinkHandle, $00100002);
    result := lread (Integer (LinkHandle), PBlock, count);
    if result <> count then
      if LinkError (LinkHandle) = LE_OK then
        LinkErrorValue := LE_LINK_INPUT_SLOW;
  end; {procedure GetBlock ---------------------------------------------------------------------------------------------------}

procedure ResetLink (analyse : Byte; LinkHandle: Integer); export;
  {Resets the transputer. The transputer has the analyse pin set if analyse is non zero.}
  begin
    if analyse = 0 then
      IOCTLwrite (LinkHandle, $00000000)
    else
      IOCTLwrite (LinkHandle, $00010000);
  end; {procedure ResetLink --------------------------------------------------------------------------------------------------}

procedure Setuplink (LinkOptions: PChar; var LinkHandle:Integer; var LinkErrorVal: Word); export;
  {Sets up the link for sending, receiving and reseting.}
  var
    device_status: Word;
    buf: TOFStruct;
  begin
    ErrorFlagMask := 255;
    LinkHandle := _lopen (LinkOptions, of_ReadWrite or of_Share_Exclusive);  {24/6/93 Attempt to detect link contention.}
                                                                             {Perhaps one of the future Inmos drivers   }
    if LinkHandle = -1 then                                                  {will prevent multiple accesss.            }
      begin
        if OpenFile (LinkOptions, buf, of_Exist) = -1 then {24/6/93 Added a bit of code to   }
          LinkErrorVal := LE_UNKNOWN_LINK                  {make error message more explicit.}
        else
          LinkErrorVal := LE_LINK_IN_USE;
        LinkHandle := NULL_LINK_HANDLE;
      end
    else
      begin
        device_status := Device_read (LinkHandle);
        if (device_status and 128) = 0 then
          LinkErrorVal := LE_UNKNOWN_LINK
        else
          LinkErrorVal := LE_OK;

        device_status := (device_status and $FF) or $20;
        Device_write (LinkHandle, device_status);
        LinkErrorValue := LinkErrorVal;
      end;
  end;

procedure CloseLink (LinkHandle: Integer); export;
  {This procedure closes the link down.}
  begin
    _lclose (LinkHandle);
  end;

procedure LinkErrorMsg (var Msg: PChar; ErrorCode: Word); export;
  {This option translates error codes to error messages.}
  begin
    if
      ErrorCode = LE_OK then
        Msg := ''
      else if ErrorCode = LE_TRANSPUTER_ERROR then
        Msg := 'Error on transputer.'
      else if ErrorCode = LE_LINK_OUTPUT_SLOW then
        Msg := 'Link output is slow.'
      else if ErrorCode = LE_LINK_INPUT_SLOW then
        Msg := 'Link input is slow.'
      else if ErrorCode = LE_LINK_IN_USE then
        Msg := 'Link already in use.'
      else if ErrorCode = LE_LINK_IO_SLOW then
        Msg := 'Link I/O is slow.'
      else if ErrorCode = LE_UNKNOWN_LINK then
        Msg := 'Unknown link.'
      else
        Msg := 'Unkown error!';
  end;

procedure ResetLinkError (MaskErrorFlagValue: Byte); export;
  {This option resets any link error and decides if the transputer error pin should be checked.}
  begin
    LinkErrorValue := LE_OK;
    if MaskErrorFlagValue = 0 then
      ErrorFlagMask := 255
    else
      ErrorFlagMask := 0;
  end;

function BoardID : Byte; export;
  {This function returns the identification for the board as defined by Inmos.}
  begin
    BoardID := 128; {This is the value IServer must return when asked what sort of board is being used.}
                    {Values for boards are: B004  - 1
                                            B008  - 2
                                            B010  - 3
                                            B011  - 4
                                            B014  - 5
                                            DRX11 - 6
                                            QT0   - 7
                                Reserved by InMOS - 8 to 127
                  NOTE: I cannot find a way of finding the board ID from the S708 driver.
                  }
  end;


exports
  {Pascal proc name, DLL index, DLL name.}
  LinkInputReady     index 1    name 'LINKINPUTREADY',
  LinkOutputReady    index 2    name 'LINKOUTPUTREADY',
  LinkError          index 3    name 'LINKERROR',
  SetUpLink          index 4    name 'SETUPLINK',
  SendBlock          index 5    name 'SENDBLOCK',
  GetBlock           index 6    name 'GETBLOCK',
  ResetLink          index 7    name 'RESETLINK',
  CloseLink          index 8    name 'CLOSELINK',
  LinkErrorMsg       index 9    name 'LINKERRORMSG',
  BoardID            index 10   name 'BOARDID',
  ResetLinkError     index 11   name 'RESETLINKERROR';

begin
  LinkErrorValue := 255;
  ErrorFlagMask := $00;
end.
