{
Title:   LIBRARY B008Drv
Summary: B008 link driver for WIServer.
Ref.:    18.01.93.06.mjm
Author:  mjm
Version: 2.0

-----------------------------------------------------------------------------------------------

Change history:

2.0      This is the first version with a reference number.
24/6/93  Change to Link Use check to allow proper checking.

----------------------------------------------------------------------------------------------

Associated documentation:
Ref.                  Title
18.01.93.09.mjm       WIServer software description
18.01.93.01.mjm       WIServer (Windows IServer)
18.01.93.07.mjm       UNIT LinkInterface
18.01.93.08.mjm       WIServer user manual


----------------------------------------------------------------------------------------------


Known problems/bugs to sort out:

1:


----------------------------------------------------------------------------------------------

Possible future improvements:

1:

----------------------------------------------------------------------------------------------

Introduction

This library is a B008 link driver for WIServer. The filename (B008DRV.DLL) should be
specified in the WISERVER.INI file if you are using a B008 board. This source is set up
to use the B008 (with no DMA) using the port address of #150.

The procedures provided by this DLL are:

  function LinkOutputReady (LinkHandle: Word): Byte; export;
    Returns a non-zero result if the link interface will accept at least one byte.

  function LinkInputReady (LinkHandle: Word): Byte; export;
    Returns a non-zero result if the link interface has at least one byte to send.

  function TransputerError (LinkHandle: Word): Byte; export;
    Returns a non-zero result if the link interface detects an error condition on the transputer.

  function LinkError (LinkHandle: Word): Word; export;
    Returns a link error condition of the link (See the LE_xxxx constants).

  procedure SendBlock (PBlock: Pointer; count, LinkHandle: Word); export;
    Sends a block of bytes to the link.

  procedure GetBlock (PBlock: Pointer; count, LinkHandle: Word); export;
    Gets a block of bytes from the link.

  procedure ResetLink (analyse : Byte; LinkHandle: Word); export;
    Resets the transputer (possibly with analyse).

  procedure Setuplink (LinkOptions: PChar; var LinkHandle, LinkErrorVal: Word); export;
    Opens a link for communication. Returns a link handle.

  procedure CloseLink (LinkHandle: Word); export;
    Closes a link.

  procedure LinkErrorMsg (var Msg: PChar; ErrorCode: Word); export;
    Converts a link error number to an error string.

  procedure ResetLinkError (MaskErrorFlagValue: Byte); export;
    Clears the link error condition without affecting the link.

  function BoardID : Byte; export;
    Returns an IServer standard identifier for the board implemented.

This DLL can only open on link to a B008 board at a time. However, if you have two B008 boards,
you can configure them to operate at different link addresses and use a similar B008 driver
that is configured for the second link address.

}



library B008DRV;

{$D-,G+,R-,S-,W-} {No debug info,
                   286 code generated,
                   No range checking,
                   no stack overflow checking,
                   Protected mode only}

uses
  WinProcs, WinTypes;

const
  MAX_RETRIES: LongInt   = 300000; {No of retries before 'Slow output/input link error'.}
  NULL_LINK_HANDLE       = 0;      {Used to identify a bad link handle.}

  LE_OK                  = 0;      {Link error codes}
  LE_TRANSPUTER_ERROR    = 1;
  LE_LINK_OUTPUT_SLOW    = 2;
  LE_LINK_INPUT_SLOW     = 3;
  LE_LINK_IN_USE         = 4;
  LE_UNKNOWN_LINK_HANDLE = 5;
  LE_LINK_IO_SLOW        = 6;

  LinkPortBaseAddr                = $150;          {Save file as B008DRV.DLL}
  {LinkPortBaseAddr                = $200;}          {Save file as B008DRV2.DLL}
  {LinkPortBaseAddr                = $300;}          {Save file as B008DRV3.DLL}
  LinkInPortAddr                  = LinkPortBaseAddr + $00;   {Port addresses - note not all the ports are used.}
  LinkOutPortAddr                 = LinkPortBaseAddr + $01;
  LinkInStatusPortAddr            = LinkPortBaseAddr + $02;
  LinkOutStatusPortAddr           = LinkPortBaseAddr + $03;
  LinkResetErrorPortAddr          = LinkPortBaseAddr + $10;
  LinkAnalysePortAddr             = LinkPortBaseAddr + $11;
  LinkDMARequestPortAddr          = LinkPortBaseAddr + $12;
  LinkInterruptEnablePortAddr     = LinkPortBaseAddr + $13;
  LinkInterruptChanSelectPortAddr = LinkPortBaseAddr + $14;

var
  LinkInUse     : Boolean; {Used to stop link contention.}
  LinkErrorValue: Word;    {Stores the current link error value.}
  ErrorFlagMask : Byte;    {Used to mask the transputer error pin during analyse.}


function LinkOutputReady (LinkHandle: Word): Byte; export;
  {If returns non zero then link is expecting input.}
  begin
    LinkOutputReady := Port[LinkOutStatusPortAddr] and $01;
  end; {function LinkOutputReady ---------------------------------------------------------------------------------------------}


function LinkInputReady (LinkHandle: Word): Byte; export;
  {If returns non zero then link has sent a byte.}
  begin
    LinkInputReady := Port[LinkInStatusPortAddr] and $01;
  end; {function LinkInputReady ----------------------------------------------------------------------------------------------}


function TransputerError (LinkHandle: Word): Byte; export;
  {Checks transputer error flag if ErrorFlagMask is zero.}
  {Returns non zero if error pin is set.}
  begin
    if ErrorFlagMask = 0 then
      TransputerError := (Port[LinkResetErrorPortAddr] and $01) xor 1
    else
      TransputerError := 0;
  end; {function TransputerError ---------------------------------------------------------------------------------------------}

function LinkError (LinkHandle: Word): Word; export;
  {Returns the current error and tests the transputer error pin.}
  begin
    if (Port[LinkResetErrorPortAddr] and $01) = 0 then
      if ErrorFlagMask = 0 then
        LinkErrorValue := LE_TRANSPUTER_ERROR;
    LinkError := LinkErrorValue;
  end; {function LinkError ---------------------------------------------------------------------------------------------------}

procedure SendBlock (PBlock: Pointer; count, LinkHandle: Word); export;
  {Sends a block of memory to the link, checking for errors as sent out.}
  {PBlock points to the block to get sent,
   count hold the size of the block in bytes,
   LinkHandle is the link identifier - Not used in this DLL because it can only hold a single link.}
  type
    Buffer  = array[0..$FFFE] of Byte;
  var
    retries: LongInt;
    index: Integer;
    PBuffer: ^Buffer;
  begin
    index := 0;
    PBuffer := PBlock;
    while count > 0 do
      begin
        retries := 0;
        while ((Port[LinkOutStatusPortAddr] and $01) = 0) and (LinkErrorValue = LE_OK) do
          if retries > MAX_RETRIES then
            begin
              count := 1;
              LinkErrorValue := LE_LINK_OUTPUT_SLOW;
              LinkError (LinkHandle);
            end
          else
            Inc (retries);
        Port[LinkOutPortAddr] := PBuffer^[index];
        Dec (count);
        Inc (index);
      end;
  end; {procedure SendBlock --------------------------------------------------------------------------------------------------}


procedure GetBlock (PBlock: Pointer; count, LinkHandle: Word); export;
  {Gets a block of memory from the link, checking for errors as recieved.}
  {PBlock points to a block of memory used to store,
   count hold the size of the block to get in bytes,
   LinkHandle is the link identifier - Not used in this DLL because it can only hold a single link.}
  type
    Buffer  = array[0..$FFFE] of Byte;
  var
    retries: LongInt;
    index: Integer;
    PBuffer: ^Buffer;
  begin
    index := 0;
    PBuffer := PBlock;
    while count > 0 do
      begin
        retries := 0;
        while ((Port[LinkInStatusPortAddr] and $01) = 0) and (LinkErrorValue = LE_OK) do
          if retries > MAX_RETRIES then
            begin
              count := 1;
              LinkErrorValue := LE_LINK_INPUT_SLOW;
              LinkError (LinkHandle);
            end
          else
            Inc (retries);
        PBuffer^[index] := Port[LinkInPortAddr];
        Dec (count);
        Inc (index);
      end;
  end; {procedure GetBlock ---------------------------------------------------------------------------------------------------}


procedure delay;                                                                                        {Produce a 100ms delay}
  var
    StartTime : LongInt;
  begin
    StartTime := GetTickCount;                                                  {The IFOPT ensures that if range checking     }
    {$IFOPT R+}                                                                 {  is on then it is off for the repeat-until  }
    {$R-}                                                                       {  loop. There is a possibility of rollover in}
    repeat until (GetTickCount - StartTime) > 100;                              {  GetTickCount.                              }
    {$R+}
    {$ELSE}
    repeat until (GetTickCount - StartTime) > 100;
    {$ENDIF}
  end; {procedure delay ------------------------------------------------------------------------------------------------------}


procedure ResetLink (analyse : Byte; LinkHandle: Word); export;
  {Resets the transputer. The transputer has the analyse pin set if analyse is non zero.}
  begin
    if analyse <> 0 then
      begin
        Port[LinkAnalysePortAddr] := $01;
        Port[LinkResetErrorPortAddr] := $00;
        delay;
      end;
    Port[LinkResetErrorPortAddr] := $01;
    delay;
    Port[LinkResetErrorPortAddr] := $00;
    delay;
    if analyse <> 0 then
      begin
        Port[LinkAnalysePortAddr] := $00;
        delay;
      end;
    LinkErrorValue := LE_OK;
  end; {procedure ResetLink --------------------------------------------------------------------------------------------------}

procedure Setuplink (LinkOptions: PChar; var LinkHandle, LinkErrorVal: Word); export;
  {Sets up the link for sending, receiving and reseting.}
  {If LinkOptions contains an 'X' then SetupLink will not complain if there is another Application using it.}
  {LinkHandle is returned as an identifier for the link (always 1 in this DLL).}
  var
    LinkInUseOverride: Boolean;
    tempErrorFlagMask: Byte;
    i: Integer;
  begin
    tempErrorFlagMask := 255;                                                          {Error pin should be checked.}
    i := 0;                                                                        {Check to see if the X option is used.}
    LinkInUseOverride := FALSE;
    while LinkOptions[i] <> chr(0) do
      begin
        LinkInUseOverride := LinkInUseOverride or (Upcase (LinkOptions[i]) = 'X'); {Turn off 'in use' check.}
        if Upcase (LinkOptions[i]) = 'E' then tempErrorFlagMask := 0;                  {Turn off error pin checking.}
        Inc (i);
      end;

    LinkHandle := NULL_LINK_HANDLE;

    if LinkInUse and not LinkInUseOverride then                                    {Check for use.}
      begin
        LinkErrorVal := LE_LINK_IN_USE;
        LinkHandle := NULL_LINK_HANDLE; {24/6/93 - Allows proper link use checking.}
      end
    else
      begin
        LinkInUse := TRUE;
        Port[LinkInterruptEnablePortAddr]     := $00; {No interrupts}
        Port[LinkInStatusPortAddr]            := $00; {No interrupt for input}
        Port[LinkOutStatusPortAddr]           := $00; {No interrupts for output}
        Port[LinkInterruptChanSelectPortAddr] := $08; {Select no DMA and IRQ chan 3}
        Port[LinkAnalysePortAddr]             := $00; {Reset the analyse line}
        Port[LinkResetErrorPortAddr]          := $00; {Reset the reset line}
        LinkErrorVal   := LE_OK;
        LinkErrorValue := LE_OK;
        ErrorFlagMask  := tempErrorFlagMask;
        LinkHandle     := 1;
      end;
  end;

procedure CloseLink (LinkHandle: Word); export;
  {This procedure closes the link down.}
  begin
    LinkInUse := FALSE;
  end;

procedure LinkErrorMsg (var Msg: PChar; ErrorCode: Word); export;
  {This option translates error codes to error messages.}
  begin
    if
      ErrorCode = LE_OK then
        Msg := ''
      else if ErrorCode = LE_TRANSPUTER_ERROR then
        Msg := 'Error on transputer.'
      else if ErrorCode = LE_LINK_OUTPUT_SLOW then
        Msg := 'Link output is slow.'
      else if ErrorCode = LE_LINK_INPUT_SLOW then
        Msg := 'Link input is slow.'
      else if ErrorCode = LE_LINK_IN_USE then
        Msg := 'Link already in use.'
      else if ErrorCode = LE_LINK_IO_SLOW then
        Msg := 'Link I/O is slow.'
      else
        Msg := 'Unkown error!';
  end;

procedure ResetLinkError (MaskErrorFlagValue: Byte); export;
  {This option resets any link error and decides if the transputer error pin should be checked.}
  begin
    LinkErrorValue := LE_OK;
    if MaskErrorFlagValue = 0 then
      ErrorFlagMask := 255
    else
      ErrorFlagMask := 0;
  end;

function BoardID : Byte; export;
  {This function returns the identification for the board as defined by Inmos.}
  begin
    BoardID := 2; {This is the value IServer must return when asked what sort of board is being used.}
                  {Values for boards are: B004  - 1
                                          B008  - 2
                                          B010  - 3
                                          B011  - 4
                                          B014  - 5
                                          DRX11 - 6
                                          QT0   - 7
                              Reserved by InMOS - 8 to 127
                  }
  end;


exports
  {Pascal proc name, DLL index, DLL name.}
  LinkInputReady     index 1    name 'LINKINPUTREADY',
  LinkOutputReady    index 2    name 'LINKOUTPUTREADY',
  LinkError          index 3    name 'LINKERROR',
  SetUpLink          index 4    name 'SETUPLINK',
  SendBlock          index 5    name 'SENDBLOCK',
  GetBlock           index 6    name 'GETBLOCK',
  ResetLink          index 7    name 'RESETLINK',
  CloseLink          index 8    name 'CLOSELINK',
  LinkErrorMsg       index 9    name 'LINKERRORMSG',
  BoardID            index 10   name 'BOARDID',
  ResetLinkError     index 11   name 'RESETLINKERROR';

begin
  {Initial setting is - Link not in use,
                        unknown error value (Link should be opened before looking at error),
                        error pin of transputer if checked.}
  LinkInUse := FALSE;
  LinkErrorValue := 255;
  ErrorFlagMask := $00;
end.
