
/*
 -- ---------------------------------------------------------------------------
 --
 --      link.c   -   INMOS standard link interface
 --
 -- ---------------------------------------------------------------------------
*/

#include <stdio.h>
#include <fcntl.h>
#include "b014.h"

extern	int	errno;

static	int	gTimeout = -1;

/*
 *
 *   OpenLink
 *
 *   Ready the link associated with `Name'.
 *   If `Name' is NULL or "" then any free link can be used.
 *   Returns any positive integer as a link id or
 *   a negative value if the open fails.
 *
 */

int OpenLink ( Name )
   char *Name;
{
	int	i, fd;
	static	char	*name[] = {
				"/dev/b014",
				"/dev/vmtm1",
				"/dev/vmtm2",
				"/dev/vmtm4",
				"/dev/vmtm8"
			};

	if (Name == NULL || *Name == '\0') {
		for (i=0; i<(sizeof(name)/sizeof(char*)); i++) {
			if ((fd=open(name[i], O_RDWR)) >= 0) {
				return fd;
			}
			printf("Returning: %d\n",fd);
		}
		return -1;
	} else {
		if ((fd=open(Name, O_RDWR)) >= 0)
			return fd;
		else
			return -1;
	}
}



/*
 *
 *   CloseLink
 *
 *   Close the active link `LinkId'.
 *   Returns 1 on success or negative if the close failed.
 *
 */

int CloseLink ( LinkId )
   int LinkId;
{
	if (close(LinkId) == -1)
		return -1;
	else
		return 1;
}



/*
 *
 *   ReadLink
 *
 *   Read `Count' chars into `Buffer' from the specified link.
 *   LinkId is a vaild link identifier, opened with OpenLink.
 *   `Timeout' is a non negative integer representing tenths
 *   of a second.  A `Timeout' of zero is an infinite timeout.
 *   The timeout is for the complete operation.
 *   If `Timeout' is positive then ReadLink may return having
 *   read less that the number of chars asked for.
 *   Returns the number of chars placed in `Buffer' (which may
 *   be zero) or negative to indicate an error.
 *
 */
 
int ReadLink ( LinkId, Buffer, Count, Timeout )
   int LinkId;
   char *Buffer;
   unsigned int Count;
   int Timeout;
{
	if (Timeout != gTimeout) {
		gTimeout = Timeout;
		ioctl(LinkId, SLEEP, 10*Timeout);
	}

	return read(LinkId, Buffer, Count);
}



/*
 *
 *   WriteLink
 *
 *   Write `Count' chars from `Buffer' to the specified link.
 *   LinkId is a vaild link identifier, opened with OpenLink.
 *   `Timeout' is a non negative integer representing tenths
 *   of a second.  A `Timeout' of zero is an infinite timeout.
 *   The timeout is for the complete operation.
 *   If `Timeout' is positive then WriteLink may return having
 *   written less that the number of chars asked for.
 *   Returns the number of chars actually written (which may
 *   be zero) or negative to indicate an error.
 *
 */
 
int WriteLink ( LinkId, Buffer, Count, Timeout )
   int LinkId;
   char *Buffer;
   unsigned int Count;
   int Timeout;
{
	if (Timeout != gTimeout) {
		gTimeout = Timeout;
		ioctl(LinkId, SLEEP, 10*Timeout);
	}

	return write(LinkId, Buffer, Count);
}



/*
 *
 *   ResetLink
 *
 *   Reset the subsystem associated with the specified link.
 *   Returns 1 if the reset is successful, negative otherwise.
 *
 */
 
int ResetLink ( LinkId )
   int LinkId;
{
	errno = 0;

	ioctl(LinkId, RESET);

	return errno ? -1 : 1;
}



/*
 *
 *   AnalyseLink
 *
 *   Analyse the subsystem associated with the specified link.
 *   Returns 1 if the analyse is successful, negative otherwise.
 *
 */
 
int AnalyseLink ( LinkId )
   int LinkId;
{
	errno = 0;

	ioctl(LinkId, ANALYZE);

	return errno == 0 ? 1 : -1;
}



/*
 *
 *   TestError
 *
 *   Test the error status associated with the specified link.
 *   Returns 1 if error is set, 0 if it is not and
 *   negative to indicate an error.
 *
 */
 
int TestError ( LinkId )
   int LinkId;
{
	int	ErrorSet;

	errno = 0;

	ioctl(LinkId, ERROR, &ErrorSet);

	if (errno == 0)
		return ErrorSet ? 1 : 0;
	else
		return -1;
}



/*
 *
 *   TestRead
 *
 *   Test input status of the link.
 *   Returns 1 if ReadLink will return one byte without timeout,
 *   0 if it may not and negative to indicate an error.
 *
 */

int TestRead ( LinkId )
   int LinkId;
{
	int	Ready;

	ioctl(LinkId, RREADY, &Ready);

	return Ready;
}



/*
 *
 *   TestWrite
 *
 *   Test output status of the link.
 *   Returns 1 if WriteLink can write one byte without timeout,
 *   0 if it may not and negative to indicate an error.
 *
 */

int TestWrite ( LinkId )
   int LinkId;
{
	int	Ready;

	ioctl(LinkId, WREADY, &Ready);

	return Ready;
}



/*
 *   Eof
 */
