/***********************************************
 *
 * File: iserver.c
 *
 * 29-sep-92 lr added subsystem reset
 * 02-Jun-92 lr
 *   link name is now 'transputer'
 * 03-Jun-91 lr
 *   changes in Core()
 * 09-sep-94 lv
 *   added SPEED_DEB
 *   added PCS_AND_LOC to compile using pcs_and_loc_link
 */
/*
 -- ---------------------------------------------------------------------------
 --
 --      ISERVER  -  INMOS standard file server
 --
 --      iserver.c
 --
 --      The main body
 --
 --      Copyright (c) INMOS Ltd., 1988.
 --      All Rights Reserved.
 --
 -- ---------------------------------------------------------------------------
*/


#include <stdio.h>
#include <signal.h>
#include <string.h> 
#include <stdlib.h>	/* for malloc */

#ifdef HELIOS
#include <stdlib.h>
#include <posix.h>
#endif

#include "inmos.h"
#include "iserver.h"


EXTERN VOID SpOpen();
EXTERN VOID SpClose();
EXTERN VOID SpRead();
EXTERN VOID SpWrite();
EXTERN VOID SpGetBlock();
EXTERN VOID SpPutBlock();
EXTERN VOID SpGets();
EXTERN VOID SpPuts();
EXTERN VOID SpFlush();
EXTERN VOID SpSeek();
EXTERN VOID SpTell();
EXTERN VOID SpEof();
EXTERN VOID SpError();
EXTERN VOID SpRemove();
EXTERN VOID SpRename();

EXTERN VOID SpGetkey();
EXTERN VOID SpPollkey();
EXTERN VOID SpGetenv();
EXTERN VOID SpTime();
EXTERN VOID SpSystem();
EXTERN VOID SpCommand();

EXTERN VOID SpCore();
EXTERN VOID SpId();
EXTERN int  SpExit();
EXTERN VOID SpUnknown();

#ifdef MSC
EXTERN VOID SpMsdos();
#endif

EXTERN LINK OpenLink();
EXTERN int  CloseLink();
EXTERN int  ReadLink();
EXTERN int  WriteLink();
EXTERN int  ResetLink();
EXTERN int  AnalyseLink();
EXTERN int  TestError();
EXTERN int  TestRead();
EXTERN int  TestWrite();

EXTERN VOID HostBegin();
EXTERN VOID HostEnd();
EXTERN VOID ResetTerminal();



PUBLIC BYTE Tbuf[TRANSACTION_BUFFER_SIZE];
	/*  buffer for all server operations  */

PUBLIC BYTE RealCommandLine     [ MAX_COMMAND_LINE_LENGTH+1 ];
PUBLIC BYTE DoctoredCommandLine [ MAX_COMMAND_LINE_LENGTH+1 ];

/***  command line switches  ***/
PUBLIC BOOL AnalyseSwitch;
PUBLIC BOOL TestErrorSwitch;
PUBLIC BOOL VerboseSwitch;
PUBLIC BOOL LinkSwitch;
PUBLIC BOOL ResetSwitch;
PUBLIC BOOL ResetSubsysSwitch;
PUBLIC BOOL ServeSwitch;
PUBLIC BOOL LoadSwitch;
PUBLIC BOOL CocoPops;

#ifdef PCS_AND_LOC
PUBLIC int  UsedLinker;
#endif

PRIVATE BYTE BootFileName[MAX_BOOT_FILE_LENGTH+1];

PRIVATE BYTE LinkName[MAX_LINK_NAME_LENGTH+1];  


PUBLIC LINK TheLink = -1 ;	/*  the server's idea of the active link  */
PUBLIC BYTE *CoreDump;		/*  for analyse  */
PUBLIC int PeekKBytes;
PUBLIC int PeekChunk;
PUBLIC int PeekBase;

PRIVATE char *HardwareNames[] = BOARD_NAMES;



extern BYTE GetAKey();

PRIVATE int
AskIfAnalyse (ExitCode)
	int ExitCode;
{
	BYTE c = 0;

	fputc('\n', stderr); 
	while ( c == 0 ) {
		fprintf(stderr,"%s: (r)reboot or (q)uit? ", PROGRAM_NAME);
		fflush(stderr);
		c = GetAKey(); 
		fputc(c, stderr);
		if ( (c != 10) && (c != 13) ) fputc('\n', stderr); 
		fflush(stderr);
		switch(c) {
		case  0 :
		case 10 :
		case 13 :
		case 'q':
		case 'Q':
			HostEnd();
			exit(EXIT_SUCCESS);
		case 'r':
		case 'R':
			/* should be reboot! ***/
			HostEnd();
			exit(ExitCode);
		default:
			c = 0;
			break;
		}
	}
	HostEnd();
	return(ExitCode);
}

PRIVATE VOID
TransputerError ()
{
   fprintf(STANDARD_ERROR, "\nError - %s - ", PROGRAM_NAME);
   fprintf(SE, "Transputer error flag set");
   fputs(".\n",STANDARD_ERROR);
   if (TheLink != -1 ) CloseLink(TheLink);
   exit(AskIfAnalyse(EXIT_ERRORFLAG));
}

/*
 *   Boot  -  copy the boot file to the link
 */

PRIVATE VOID
Boot()
{
#ifdef PCS_AND_LOC
if(!UsedLinker){
#endif PCS_AND_LOC
#if PCSERVER | PCS_AND_LOC
	if (isatty(1)) {
		INFO(("Booting root transputer..."));
	}
	load_code(BootFileName);
#endif /* PCSERVER | PCS_AND_LOC */
#ifdef PCS_AND_LOC
} else {
#endif PCS_AND_LOC
#ifndef PCSERVER 

   FILE *Fd;
   BYTE Buffer[ BOOT_BUFFER_LENGTH ];
   int Length=0, Count=0;
   INT32 Size=0;
   static BYTE Flicks[]="|/-\\|/-\\" ;
   int Flick=0;

   if ( ( Fd=fopen( BootFileName, "rb" ) ) == NULL )
      ABORT((SE, "cannot find boot file \"%s\"", BootFileName));
   INFO(("Booting root transputer..."));
   while ( ( Length = fread( Buffer, 1, BOOT_BUFFER_LENGTH, Fd ) ) > 0 ) {
	 DEBUG(("%d", Length));
	 if ( ( Count = WriteLink( TheLink, Buffer, Length, BOOT_TIMEOUT ) ) < 0 )
	    ABORT(( SE,"internal error - WriteLink in Boot was bad"));
	 Size += Count;
	 if ( Count != Length )
	    ABORT((SE, "unable to write byte %d to the boot link", Size ));
	 if ( isatty(1) ) {
	       INFO(( "%c\b", Flicks[Flick] ));
	       if ( ++Flick == 8 ) Flick -= 8;
	 }
   }
   if ( isatty(1) ) {
	 INFO(( "\r                           \r" ));
   } else {
	 INFO(("ok\n"));
   }
   DEBUG(( "booted %ld bytes", Size ));
#endif /* !PCSERVER */
#ifdef PCS_AND_LOC
}
#endif PCS_AND_LOC
}




/*
 *   Core  -  peek the transputer memory into a buffer
 */

PRIVATE VOID
Core()
{
	BYTE Buf[512*5] /* , *c */;
	INT32 a;
	INT32 l,p;
	INT32 PeekBytes;
	static BYTE Flicks[]="|\\-/|\\-/" ;
	int Flick=0;

	PeekBytes = 1024 * PeekKBytes;
	INFO(( "Peeking root transputer - %ld bytes", PeekBytes));
	CoreDump = (BYTE *)malloc (PeekBytes);
	if (CoreDump == NULL)
		ABORT((SE, "cannot allocate %ld bytes for core dump peek block", PeekBytes));
	for( a = 0 ; a < PeekBytes ;) {
		for (p=0;p< 5*PeekChunk;) {
			l = (INT32)(a + 0x80000000L + PeekBase*1024);
			Buf[p++] = 1;  /*  peek  */
			Buf[p++] = l & 0xff;
			Buf[p++] = (l>>8) & 0xff;
			Buf[p++] = (l>>16) & 0xff;
			Buf[p++] = (l>>24) & 0xff;
			a += 4;
		}
	 	if ( WriteLink( TheLink, &Buf[0], 5*PeekChunk, TIMEOUT ) != 5*PeekChunk )
			ABORT((SE, "timed out peeking word %d", a/4 ));
		if ( ReadLink( TheLink, &Buf[1], 4*PeekChunk, TIMEOUT ) != 4*PeekChunk )
			ABORT((SE, "timed out in peek of word %d", a/4 ));
		memcpy(CoreDump+a, Buf, 4*PeekChunk);
		if ( isatty(1) &&  (a & 0xfff) == 0 ) {
			INFO(( "%c\b", Flicks[Flick] ));
			if ( ++Flick == 8 ) Flick -= 8;
		}
	}
	if ( isatty(1) ) {
		INFO(( "\r                                        \r" ));
	} else {
		INFO(("ok\n"));
	}
	INFO(( "peeked %ld bytes\n", a ));
}




/*
 *   PrintHelp 
 */

PRIVATE VOID
PrintHelp ()
{
   fprintf( stdout, "\n%s : INMOS host file server\n", PROGRAM_NAME);
   fprintf( stdout, "%s & %s\nVersion %s.\n", HOST, HardwareNames[BOARD_ID], VERSION_NAME );
   fprintf( stdout, "%s\n", Copyright );
   fprintf( stdout, "Usage:  %s {%coption}\n\n", PROGRAM_NAME, SWITCH_CHAR );
   fprintf( stdout, "Options: \n" );
   fprintf( stdout, "        SA            analyse and peek the root transputer\n");
   fprintf( stdout, "        SB  filename  boot the named file (same as %cSR%cSS%cSI%cSC filename)\n", SWITCH_CHAR, SWITCH_CHAR, SWITCH_CHAR, SWITCH_CHAR );
   fprintf( stdout, "        SBB filename  boot the named file (same as %cSRR%cSS%cSI%cSC filename)\n", SWITCH_CHAR, SWITCH_CHAR, SWITCH_CHAR, SWITCH_CHAR );
   fprintf( stdout, "        SC filename   copy the named file to the link\n");
   fprintf( stdout, "        SE            test the error flag\n");
   fprintf( stdout, "        SI            verbose mode\n");
   fprintf( stdout, "        SL linkname   use the named link\n");
   fprintf( stdout, "        SP peeksize   peek peeksize Kbytes (<=4096)\n");
   fprintf( stdout, "        SPB peekbase  peek base address (Kbytes)\n");
   fprintf( stdout, "        SPP peekchunk peek chunk bytes (<=512)\n");
   fprintf( stdout, "        SR            reset the root transputer\n");
   fprintf( stdout, "        SRR           reset the root transputer and subsystem\n");
   fprintf( stdout, "        SS            serve the link\n\n");
}




/*
 *   ParseCommandLine 
 */

PRIVATE VOID
ParseCommandLine ()
{
   register BYTE *s, *t;
   int i;
   BOOL Gobble=TRUE;
   /* BYTE *getenv(); */

   AnalyseSwitch = 0;
   PeekKBytes = 8;
   PeekBase = 0;
#ifdef PCS_AND_LOC
if(!UsedLinker){
#endif PCS_AND_LOC
#if PCSERVER | PCS_AND_LOC
   PeekChunk=512;
#endif
#ifdef PCS_AND_LOC
}else{
#endif PCS_AND_LOC
#ifndef PCSERVER 
   PeekChunk = 1;
#endif
#ifdef PCS_AND_LOC
}
#endif PCS_AND_LOC
   TestErrorSwitch = 0;
   VerboseSwitch = 0;
   LinkSwitch = 0;
   ResetSwitch = 0;
   ResetSubsysSwitch = 0;
   ServeSwitch = 0;
   LoadSwitch = 0;
   CocoPops = 0;

   BootFileName[0]=0;
   LinkName[0]=0;
   if ( (s=(BYTE *)getenv("TRANSPUTER")) != NULL ) {
	strcpy(LinkName,s);
   };

   s = RealCommandLine;
   t = DoctoredCommandLine;

   while( (*s) && (*s != ' ') )  /* skip the first argv */
      ++s;
   if ( *s == ' ' )
      ++s;

   for(;;) {
	if ( *s == '\0' ) return;      /*  end of command line  */
	if ( (*s==SWITCH_CHAR) &&
	     ((*(s+1)=='s')||(*(s+1)=='S'))) { /* server option */
	   s += 2;
	   switch ( *s ) {
		case 'a' :
		case 'A' :
		    ++AnalyseSwitch;
		    ++s;
		    break;

		case 'b' :
		case 'B' :
		    ++s;
		    ++ResetSwitch;
		    ++LoadSwitch;
		    ++ServeSwitch;
		    ++VerboseSwitch;
		    if (*s=='b' || *s=='B') {
			++s;
		        ++ResetSubsysSwitch;
		    }
		    while( *s == ' ' ) ++s;
		    i = 0;
		    while( (*s) && (*s != SWITCH_CHAR) && (*s != ' ') ) {
		        if ( i == MAX_BOOT_FILE_LENGTH )
			    ABORT((SE, "boot filename is too long, maximum size is %d characters", MAX_BOOT_FILE_LENGTH ));
		   	 BootFileName[i++] = *s++;
		    }
		    if ( i == 0 )
			ABORT((SE, "expected a filename after %csb option", SWITCH_CHAR ));
		    BootFileName[i] = 0;
		    break;

		case 'c' :
		case 'C' :
		    ++s;
		    ++LoadSwitch;
		    while( *s == ' ' ) ++s;
		    i = 0;
		    while( (*s) && ( (*s!=SWITCH_CHAR) && (*s!=' ') ) ) {
			if ( i == MAX_BOOT_FILE_LENGTH )
			   ABORT((SE, "copy filename is too long, maximum size is %d characters", MAX_BOOT_FILE_LENGTH ));
			BootFileName[i++] = *s++;
		    }
		    if ( i == 0 )
			ABORT((SE, "expected a filename after %csc option", SWITCH_CHAR ));
		    BootFileName[i] = 0;
		    break;

		case 'e' :
		case 'E' :  ++s;
		    ++TestErrorSwitch; break;

		case 'i' :
		case 'I' :  ++s;
		    		 ++VerboseSwitch; break;

		case 'l' :
		case 'L' :  ++s;
		    		 ++LinkSwitch;
				 while( *s == ' ' )
				    ++s;
				 i = 0;
				 while( (*s) && ( (*s!=SWITCH_CHAR) && (*s!=' ') ) )
				    {
				       if ( i == MAX_LINK_NAME_LENGTH )
					  ABORT((SE, "link name is too long, maximum size is %d characters", MAX_LINK_NAME_LENGTH ));
				       LinkName[i++] = *s++;
				    }
				 if ( i == 0 )
				    ABORT((SE, "expected a name after %csl option", SWITCH_CHAR ));
				 LinkName[i] = 0;
				 break;

		case 'p' :
		case 'P' :
			++s;
			{
			BYTE SpNumber[16];  
			int dest=0;
			if (*s=='p'||*s=='P') {
				s++;
				dest=1; /* peekchunk */
			} else if (*s=='b'||*s=='B') {
				s++;
				dest=2; /* PeekBase */
			}
			while( *s == ' ' ) ++s;
			i = 0;
			while( (*s) && ( (*s!=SWITCH_CHAR) && (*s!=' ') ) ) {
				if ( i >= 16 )
					ABORT((SE, "number is too long, maximum size is %d characters", 16 ));
					SpNumber[i++] = *s++;
			}
			if ( i == 0 )
				ABORT((SE, "expected a number"));
				SpNumber[i] = 0;
				if (dest==1) {
					PeekChunk=atoi(SpNumber);
					if (PeekChunk > 512)
						ABORT((SE, "cannot peek more than 512 bytes at a time"))
				} else if (dest==2) {
					PeekBase=atoi(SpNumber);
					if (PeekBase > 4096)
						ABORT((SE, "cannot peek after 4096K"))
				} else {
					PeekKBytes = atoi(SpNumber);
					if (PeekKBytes > 4096)
						ABORT((SE, "cannot peek more than 4096K bytes"))
				}
			}
			break;

		case 'r' :
		case 'R' :
		    ++s;
		    ++ResetSwitch;
		    if (*s=='b' || *s=='B') {
			++s;
		        ++ResetSubsysSwitch;
		    }
		    break;

		case 's' :
		case 'S' :  ++s; ++ServeSwitch; break;

		case 'z' :
		case 'Z' :  ++s; ++CocoPops; break;

		default  :  
			 *t++ = SWITCH_CHAR;
			 *t++ = *(s-1);
			 Gobble = FALSE;
			 break;
	   }
	   if ( (Gobble == TRUE) && (*s == ' ') ) ++s;
	} else {
	   *t++ = *s++;
	}
    }
}




/*
 *   Serve  -  the main loop (read buffer from link,
 *             call a function, write buffer to link)
 */

PRIVATE int
Serve ()
{
   register int Count, Size;
   BOOL TerminateFlag = FALSE;
   int Result= -1; /* initialization */

   for(;;) {
	 if( TestErrorSwitch && TestError( TheLink ) == 1 ) TransputerError ();
	 DEBUG(("-=-"));
	 for(;;) {
	       Count = ReadLink( TheLink, &Tbuf[0], 8, TIMEOUT );
	       if( TestErrorSwitch && TestError( TheLink ) == 1 ) TransputerError ();
	       if( Count ) {
		     if ( Count == 8 ) break;
		     if( Count < 0 ) ABORT((SE, "internal error, got %d from ReadLink", Count ));
		     Count += ReadLink( TheLink, &Tbuf[Count], 8-Count, TIMEOUT );
		     if( Count != 8 ) ABORT((SE, "protocol error, got %d bytes at start of a transaction", Count ));
		  }
	       DEBUG(("waiting"));
#ifdef MSC
	       kbhit();                                     /*  test break  */
#endif
	    }
	/*  have 8  */
	 Size = Tbuf[0]+Tbuf[1]*256-6 ;
	 if ( Size ) {
	       if ( Size > TRANSACTION_BUFFER_SIZE - 8 ) ABORT((SE, "protocol error, packet size is too large"));
	       if ( Size < 0 ) ABORT((SE, "protocol error, read nonsense from the link"));
	       Count = ReadLink( TheLink, &Tbuf[8], Size, TIMEOUT );
	       if( TestErrorSwitch && TestError( TheLink ) == 1 ) TransputerError ();
	       if( Count != Size ) ABORT((SE, "protocol error, timed out getting a further %d", Count));
	       DEBUG(( "packet is %d", Count+6 ));
	    }
	 switch( Tbuf[2] ) {
	       case SP_OPEN     : SpOpen();     break; 
	       case SP_CLOSE    : SpClose();    break; 
	       case SP_READ     : SpRead();     break; 
	       case SP_WRITE    : SpWrite();    break; 
	       case SP_GETBLOCK : SpGetBlock(); break;
	       case SP_PUTBLOCK : SpPutBlock(); break;
	       case SP_GETS     : SpGets();     break; 
	       case SP_PUTS     : SpPuts();     break; 
	       case SP_FLUSH    : SpFlush();    break; 
	       case SP_SEEK     : SpSeek();     break; 
	       case SP_TELL     : SpTell();     break; 
	       case SP_EOF      : SpEof();      break; 
	       case SP_FERROR   : SpError();    break; 
	       case SP_REMOVE   : SpRemove();   break; 
	       case SP_RENAME   : SpRename();   break; 
	       case SP_GETKEY   : SpGetkey();   break; 
	       case SP_POLLKEY  : SpPollkey();  break; 
	       case SP_GETENV   : SpGetenv();   break; 
	       case SP_TIME     : SpTime();     break; 
	       case SP_SYSTEM   : SpSystem();   break; 
	       case SP_EXIT     : TerminateFlag = TRUE;
				  Result = SpExit();
				  TestErrorSwitch = FALSE;
				  break;
	       case SP_COMMAND  : SpCommand();  break; 
	       case SP_CORE     : SpCore();     break; 
	       case SP_ID       : SpId();       break; 
#ifdef MSC
	       case SP_MSDOS    : SpMsdos();    break;
#endif
#ifdef SPEED_DEB
	       case SP_SPEED_DEB_IN  : SpSpeedDebIn();       break; 
	       case SP_SPEED_DEB_OUT : SpSpeedDebOut();      break; 
#endif
	       default          : SpUnknown(); break;
	    }

	 Size = Tbuf[0]+Tbuf[1]*256+2 ;
	 if ( Size ) {
	       if ( Size > TRANSACTION_BUFFER_SIZE ) ABORT((SE, "internal error, oversize packet write back"));
	       DEBUG(( "%d:%d", Size-2, Tbuf[2] ));
	       Count = WriteLink( TheLink, &Tbuf[0], Size, TIMEOUT );
	       if(TestErrorSwitch && TestError(TheLink)==1) TransputerError ();
	       if( Count != Size ) ABORT((SE, "protocol error, timed out sending reply message"));
	 } else ABORT((SE, "internal error, zero size write back"));
	 if( TerminateFlag == TRUE ) return( Result );
      }
}




PUBLIC void
BreakHandler()
{
   BYTE c = 0;
int pi=0;

#ifndef UNIX
   signal(SIGINT, BreakHandler);
#endif

   fputc('\n', stderr); 
fflush(stdin);
   while ( c == 0 ) {
	 fprintf(stderr,"%s: (x)exit, (s)hell or (c)ontinue? ", PROGRAM_NAME); fflush(stderr);
fflush(stdin);
	 c = GetAKey(); 
	 fputc(c, stderr);
	 if ( (c != 10) && (c != 13) ) fputc('\n', stderr); 
fprintf(stderr,"%d \n",pi++);
	 fflush(stderr);
	 switch(c) {
	       case  0 :
	       case 10 :
	       case 13 :
	       case 'x':
	       case 'X':
	       case 'q':
	       case 'Q':
			  DEBUG(("break exit"));
			  if ( CloseLink( TheLink ) < 1 )
			  ABORT((SE, "unable to free transputer link"));
/*			  HostEnd(); */
			  exit(AskIfAnalyse(EXIT_USERBREAK));
	       case 'c':
	       case 'C':
			  break;
	       case 'S':
	       case 's':
			  ResetTerminal();
#ifdef MSC
			  (void)system(getenv("COMSPEC"));
#endif
#if UNIX | HELIOS
			  (void)system(getenv("SHELL"));
#endif
#ifdef VMS
			  {
			     long substatus;
			     INFO(("[Starting DCL]"));
			     (void)lib$spawn(NULL, 0,0,0,0,0,&substatus,0,0,0);
			  }
#endif
			  fprintf(stderr,"\n");
			  break;

	       case 'i':
			  fprintf( stdout, "%s&%s version %s.\n", HOST, HardwareNames[BOARD_ID], VERSION_NAME );
			  if (VerboseSwitch)
			     VerboseSwitch = 0;
			  else
			     VerboseSwitch = 1;
			  break;

	       case 'z':
			  if (CocoPops)
			     CocoPops = 0;
			  else
			     CocoPops = 1;
			  break;

	       default:
			  c = 0;
			  break;
	    }
      }
}




/*
 *   main 
 */

PUBLIC int
main (argc, argv)
   int argc;
   char **argv;
{
   int ExitStatus;
   BYTE *c, *ALinkName;

   if ( argc == 1 ) {
	 PrintHelp();
	 return ( EXIT_SUCCESS );
   }

   HostBegin();                        /*  terminal initialisation  */

   *RealCommandLine = 0;
   while ( argc-- > 0 ) {
	 if ( ( MAX_COMMAND_LINE_LENGTH - strlen( RealCommandLine) ) < strlen( *argv ) ) {
	       ABORT((SE, "Command line too long (at \"%s\")", *argv) )
	 }
	 (void)strcat( RealCommandLine, *argv );
	 (void)strcat( RealCommandLine, " " );
	 ++argv;
   }
   for ( c=RealCommandLine; *c ; ++c )    /*  strip of the last space  */ ;
   if ( *--c == ' ' ) *c = 0;
   ParseCommandLine();

#ifdef NOTDEF
   for ( c=RealCommandLine; *c ; ++c ) {
	if (*c== '/') *c='\\';
	if (*c== '-') *c='/';
   }
   for ( c=DoctoredCommandLine; *c ; ++c ) {
	if (*c== '/') *c='\\';
	if (*c== '-') *c='/';
   }
#endif /* NOTDEF */
   DEBUG(("original line is \"%s\"", RealCommandLine));
   DEBUG(("doctored line is \"%s\"", DoctoredCommandLine));

   signal( SIGINT, BreakHandler );
#ifdef UNIX
   signal( SIGTSTP, SIG_IGN );	/*  disable ctrl-z  */
#endif

   if ( CocoPops )
      printf("(analyse=%d error=%d verbose=%d link=%d reset=%d serve=%d load=%d coco=%d)\n", AnalyseSwitch, TestErrorSwitch, VerboseSwitch, LinkSwitch, ResetSwitch, ServeSwitch, LoadSwitch, CocoPops );

   DEBUG(("using \"%s\" as the boot file", BootFileName));

   if ( *LinkName == 0 ) {
	 if ( ( ALinkName = (BYTE *)getenv("TRANSPUTER") ) != NULL )
	    strcpy( LinkName, ALinkName );
   }

   DEBUG(("and \"%s\" as the link name", LinkName));

   if ( ( TheLink = OpenLink( LinkName ) ) < 1 )
      if(TheLink == ER_LINK_BAD){
	 ABORT((SE, "Bad link specification"));
      } else{
	 ABORT((SE, "unable to access a transputer"));
      }

   if ( ResetSwitch && AnalyseSwitch )
      ABORT((SE, "reset and analyse are incompatible"));

   if ( ResetSwitch ) {
	 if ( ResetLink( TheLink ) != 1 )
	    ABORT((SE, "failed to reset root transputer"));
	 DEBUG(("reset root"));
	 if (ResetSubsysSwitch) {
		char rst_buf[9];
		int i;
		for (i=0;i<9;i++) rst_buf[i]=0;	/* poke 0,0 */
		rst_buf[5]=1;
	 	if ( WriteLink( TheLink, rst_buf, sizeof(rst_buf), TIMEOUT ) != sizeof(rst_buf) ) {
	    		ABORT((SE, "Failed to reset subsystem"));
		}
		rst_buf[5]=0;
	 	if ( WriteLink( TheLink, rst_buf, sizeof(rst_buf), TIMEOUT ) != sizeof(rst_buf) ) {
	    		ABORT((SE, "Failed to deassert reset subsystem"));
		}
#if PCSERVER | PCS_AND_LOC
#ifdef PCS_AND_LOC
if(!UsedLinker){
#endif
		CloseLink(TheLink);
		if ( ( TheLink = OpenLink( LinkName ) ) < 1 )
			ABORT((SE,"failed to reopen the link after reset subsystem"));
#ifdef PCS_AND_LOC
}
#endif
#endif /* PCSERVER */
	 }
   }

   if ( AnalyseSwitch ) {
	 if ( AnalyseLink( TheLink ) != 1 )
	    ABORT((SE, "failed to analyse root transputer"));
	 DEBUG(("analyse root"));
	 Core();
#if PCSERVER | PCS_AND_LOC
#ifdef PCS_AND_LOC
if(!UsedLinker){
#endif
	CloseLink(TheLink);
	if ( ( TheLink = OpenLink( LinkName ) ) < 1 )
		ABORT((SE,"failed to reopen the link after peek"));
#ifdef PCS_AND_LOC
}
#endif
#endif /* PCSERVER */
      }

   if ( LoadSwitch ) Boot();

   if ( ServeSwitch ) ExitStatus = Serve();
   else   {
	 if ( ResetSwitch || AnalyseSwitch || LoadSwitch || TestErrorSwitch )
	    ExitStatus = EXIT_SUCCESS;
	 else {
	       INFO(("(no action taken)\n"));
	 }
      }
#ifdef notdef
   /* If Serve terminated we are not interested if error was subsequently set!*/
   if( TestErrorSwitch )
       if( TestError( TheLink ) == 1 )
           TransputerError ();

   signal(SIGINT, SIG_IGN); /*  dont want to be interrupted while closing  */
#endif
   if ( CloseLink( TheLink ) < 1 )
      ABORT((SE, "unable to free transputer link"));

   HostEnd();

   return( ExitStatus );
}



/*
 *   Eof
 */

