/*{{{Comments*/
/*
 * File        : linkio.c
 * Author      : Stuart Menefy
 * Description : IServer file reading and terminal I/O routines.
 * Header      : $Id: linkio.c,v 1.5 1994/02/21 11:22:21 stuart Exp $
 *   
 * This file supplies the IO functions to support cmonitor.c and csecond.c.
 * As such, it contains only the routines common to both.
 * This is used when booting from link and using the hosts's file system
 * via an IServer.
 *
 * History:
 *   04/07/93  SIM  Created.
 */
/*}}}*/

/*{{{#include*/
#include <string.h>
#include "io.h"

#undef BLOCK_SIZE
#define BLOCK_SIZE 507


/*}}}*/
/*{{{externs and globals*/
void print_int(const int i);

extern void* bootLinkIn;
extern void* bootLinkOut;

int file_inode; /* Make global for r_readline */
static int offset;

#define B002 0
#define B016 1

/*}}}*/
/*{{{Channel communications macros*/
#define ChanOut(link, data, size) \
  __asm{ ldabc size, link, data; out; }
#define ChanIn(link, data, size) \
  __asm{ ldabc size, link, data; in; }
/*}}}*/

/*{{{disk_init*/
void disk_init(void)
{
    ;
}
/*}}}*/
/*{{{r_open*/
void r_open(const char* filename)
{
    int len = strlen(filename);
    char hdr[3] = {10,              /* Fopen */
                   0,0};            /* Length (patched) */
    char trailer[2] = {1,           /* Binary */
                       1};          /* Open for input */                   
    int pkt_len = sizeof(hdr) + sizeof(trailer) + len;
    int pad = pkt_len & 1;

    hdr[1] = len % 256;
    hdr[2] = len / 256;

    pkt_len += pad;

    ChanOut(bootLinkOut, &pkt_len, 2);
    ChanOut(bootLinkOut, hdr, sizeof(hdr));
    ChanOut(bootLinkOut, filename, len);
    ChanOut(bootLinkOut, trailer, sizeof(trailer));
    if (pad)
        ChanOut(bootLinkOut, hdr, 1);

    ChanIn(bootLinkIn, &pkt_len, 2);
    ChanIn(bootLinkIn, &trailer, 1);    /* Result */
    pkt_len--;
    if (trailer[0] == 0) {
        ChanIn(bootLinkIn, &file_inode, sizeof(file_inode));
        pkt_len -= sizeof(file_inode);
    }
    if (pkt_len > 0)
        ChanIn(bootLinkIn, hdr, pkt_len);

    if (trailer[0] != 0) {
        printk("Unable to find file \"");
        printk(filename);
        printk("\" (reason ");
        print_int(trailer[0]);
        printk(")\n");
        file_inode = -1;
    }

    offset = 0;    
}
/*}}}*/
/*{{{r_read*/
int r_read(char* address)
{
    char hdr[7] = {12,                  /* Fread */
                   0,0,0,0,             /* Handle (patched) */
                   BLOCK_SIZE & 0xff,   /* Length (lsb) */
                   BLOCK_SIZE >> 8};    /* Length (msb) */
    int pkt_len = 8;    /* 7 + padding */
    int len =0;         /* Clear msb */

    memcpy(&hdr[1], &file_inode, 4);
    
    ChanOut(bootLinkOut, &pkt_len, 2);
    ChanOut(bootLinkOut, hdr, pkt_len);

    ChanIn(bootLinkIn, &pkt_len, 2);
    ChanIn(bootLinkIn, &hdr, 1);
    ChanIn(bootLinkIn, &len, 2);
    ChanIn(bootLinkIn, address, len);

    pkt_len -= (3 + len);
    if (pkt_len > 0)
        ChanIn(bootLinkIn, hdr, pkt_len);
    
    return len;
}
/*}}}*/
/*{{{r_close*/
void r_close()
{
    ;
}
/*}}}*/

/*{{{term_init*/
void term_init(void)
{
    ;
}
/*}}}*/
/*{{{putchar*/
void putchar(const char c)
{
    char string[2];

    string[0] = c;
    string[1] = '\0';
    printk(string);
}

/*}}}*/
/*{{{printk*/
void printk(const char* msg)
{
    int len = strlen(msg);
    char hdr[7] = {13,              /* Fwrite */
                   1,0,0,0,         /* Stream = 1 = stdout */
                   0,0};            /* Length (patched) */
    int pkt_len = 7 + len;
    int pad = pkt_len & 1;

    if (len == 0)
        return;
    
    hdr[5] = len % 256;
    hdr[6] = len / 256;

    pkt_len += pad;
    
    ChanOut(bootLinkOut, &pkt_len, 2);
    ChanOut(bootLinkOut, hdr, 7);
    ChanOut(bootLinkOut, msg, len);
    if (pad)
        ChanOut(bootLinkOut, hdr, 1);

    ChanIn(bootLinkIn, &pkt_len, 2);
    ChanIn(bootLinkIn, hdr, pkt_len);
}
/*}}}*/
/*{{{getchar*/
char getchar(void)
{
    char hdr[6] = {30,              /* GetKey */
                   0,0,0,0,0};      /* padding */
    int pkt_len = 6;
    char result[2];    
    
    ChanOut(bootLinkOut, &pkt_len, 2);
    ChanOut(bootLinkOut, hdr, pkt_len);

    ChanIn(bootLinkIn, &pkt_len, 2);
    ChanIn(bootLinkIn, result, pkt_len);

    return result[1];
}
/*}}}*/
