/*{{{Comments*/
/*
 * File        : fileio.c
 * Author      : Stuart Menefy
 * Description : Minix file system resing routines
 * Header      : $Id: fileio.c,v 1.3 1994/03/07 12:14:48 stuart Exp $
 *   
 * This file supplies the file input routines to support cmonitor.c.
 * This is used when running on a B016 and using a MINIX filesystem via
 * an M212.
 * As a result it also uses rawfs.c, and so provides readblock.
 *
 * History:
 *   06/02/94  SIM  Created.
 */
/*}}}*/

/*{{{#include*/
#include <sys/types.h>
#include <sys/stat.h>
#include <minix/config.h>
#include <minix/const.h>
#include "rawfs.h"
#include "io.h"
/*}}}*/
/*{{{externs and globals*/
extern void* bootLinkIn;
extern void* bootLinkOut;

extern int errno;

int file_inode;
static int offset;
static ino_t dir_inode;
static struct stat file_stat;
/*}}}*/
/*{{{Channel communications macros*/
#define ChanOut(link, data, size) \
  __asm{ ldabc size, link, data; out; }
#define ChanIn(link, data, size) \
  __asm{ ldabc size, link, data; in; }
/*}}}*/

/*{{{select_block*/
static void select_block(int block)
{
    static char data[] = {
        3, 0x4, 0,      /* WriteParam(LogicalSector0, x) */
        3, 0x5, 0,      /* WriteParam(LogicalSector1, x) */
        3, 0x6, 0       /* WriteParam(LogicalSector2, x) */
    };

    data[2] = (block      ) & 0xff;
    data[5] = (block >>  8) & 0xff;
    data[8] = (block >> 16) & 0xff;
    ChanOut(bootLinkOut, data, sizeof(data));
}
/*}}}*/
/*{{{select_buffer*/
static void select_buffer(int buffer)
{
    static char data[] = {
        3, 0x1d, 0      /* WriteParam(DesiredSectorBuffer, n) */
    };

    data[2] = buffer;
    ChanOut(bootLinkOut, data, sizeof(data));
}
/*}}}*/
/*{{{check_error*/
static void check_error(int block)
{
    static char data2[] = {
        2, 0x20         /* ReadParam(Error) */
    };
    char error;

    ChanOut(bootLinkOut, data2, sizeof(data2));
    ChanIn(bootLinkIn, &error, 1);
    if (error != 0) {
        printk("Unable to read sector ");
        print_int(block);
        printk(" (reason ");
        print_int(error);
        printk(")\n");
    }
}
/*}}}*/
/*{{{readblock*/
void readblock(off_t block, char* address)
{
    static char data1[] = {
        6,              /* ReadSector() */
        4               /* ReadBuffer() */
    };

    select_block(block + 2);
    select_buffer(0);   /* Ready for next sector */
    ChanOut(bootLinkOut, data1, sizeof(data1));
    ChanIn(bootLinkIn, address, BLOCK_SIZE);

    check_error(block+2);
}
/*}}}*/
/*{{{dump - not used*/
#if 0
void dump(char* address)
{
    char data[4];
    int line, row;
    char val;

    data[2] = ' ';
    data[3] = '\0';
    
    for (line = 0; line < BLOCK_SIZE; line += 16) {
        print_int(line);
        printk("  ");
        for (row = 0; row < 16; row++) {
            val = (address[line + row] & 0xf0) >> 4;
            data[0] = (val > 9) ? val + 'a' - 10 : val + '0';
            val = (address[line + row] & 0x0f);
            data[1] = (val > 9) ? val + 'a' - 10 : val + '0';
            printk(data);
        }
        printk("\n");
    }
}
#endif
/*}}}*/

/*{{{disk_init*/
void disk_init(void)
{
    static char select_drive[] = {
        3, 0x1e, 1,     /* WriteParam(DesiredDrive, 1) */
        1, 1,           /* Initialise(1 = winny) */
        8,              /* Restore() */
        3, 0x09, 10,    /* WriteParam(sectorSizeLg2, 10 = 1024) */
        3, 0x0a,  9     /* WriteParam(NumberOfSectors, 9) */
    };
    off_t disk_size;

    printk("About to init disk\n");
    ChanOut(bootLinkOut, select_drive, sizeof(select_drive));
    printk("Init disk OK\n");

    /* Can now call readblock */
    
    disk_size = r_super();
    printk("size of disk ");
    print_int(disk_size);
    printk("\n");

    dir_inode = r_lookup(ROOT_INO, "minix");
    if (dir_inode == 0) {
        printk("Unable to find minix directory");
        print_int(errno);
        printk("\n");
        return;
    }
}
/*}}}*/
/*{{{r_open*/
void r_open(const char* filename)
{
    file_inode = r_lookup(dir_inode, filename);
    if (file_inode == 0) {
        printk("Unable to find file \"");
        printk(filename);
        printk("\" (reason ");
        print_int(errno);
        printk(")\n");
        return;
    }

    offset = 0;

    /* Need to stat the file to set 'curfil' for future reads. */
    r_stat(file_inode, &file_stat);
}
/*}}}*/
/*{{{r_read*/
int r_read(char* address)
{
    int block;
    
    block = r_vir2abs(offset);
    readblock(block, address);
    offset++;

    return BLOCK_SIZE;
}
/*}}}*/
/*{{{r_readline*/
int r_readline(char* buffer, int max_len)
{
    static char block[BLOCK_SIZE];
    static int remaining = 0;   /* Remaining chars in block */
    static char *ptr;           /* Ptr into block */

    while (1) {
        while (remaining-- > 0) {
            if ((*buffer = *(ptr++)) == '\n') {
                *buffer = '\0';
                return 1;
            }
            buffer++;
        }

        if (file_stat.st_size == 0) {
            *buffer = '\0';
            return 0;
        }
    
        r_read(block);
        if (file_stat.st_size > BLOCK_SIZE)
            remaining = BLOCK_SIZE;
        else
            remaining = file_stat.st_size;
        file_stat.st_size -= remaining;
        ptr = block;
    }
}
/*}}}*/
/*{{{r_close*/
void r_close(void)
{
    ;
}
/*}}}*/
