/*{{{}}}*/
/*{{{  #includes*/
#include <ctype.h>
#include <errno.h>
#include <getopt.h>
#include <string.h>
#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>

#include "link.h"

typedef enum { FALSE, TRUE } Bool;

#ifdef READLINE
extern char *readline(char *prompt);
extern void add_history(char *line);
#endif
/*}}}  */
/*{{{  #defines*/
#define TIMEOUT 10
/*}}}  */

/*{{{  variables*/
static unsigned long MinInt=0x80000000;
/*}}}  */

/*{{{  number*/
static unsigned long int number(s, endptr) char *s; char **endptr;
{
  int base=10;
  char c;
  register char *nptr = s;
  unsigned long int result = 0L;
  Bool saw_a_digit = FALSE;

  if (*nptr == '0')
  {
    if ((c = *++nptr) == 'x' || c == 'X')
    {
      ++nptr;
      base = 16;
    }
    else
    {
      saw_a_digit = TRUE;        /* in case '0' is only digit */
      base = 8;
    }
  }

  --nptr;
  while ((c=*++nptr))
  {
    if (isdigit(c)) c -= '0';
    else if (isupper(c)) c -= ('A'-10);
    else if (islower(c)) c -= ('a'-10);
    else break;
    if (c>=base) break;
    saw_a_digit = TRUE;
    result = result*base+c;
  }

  *endptr=(saw_a_digit ? nptr : s);
  return result;
}
/*}}}  */
/*{{{  snumber*/
static unsigned long int snumber(char *s, char **endptr)
{
  if (*s=='-' && isdigit(*(s+1))) return(-number(++s, endptr));
  else return(number(s, endptr));
}
/*}}}  */
/*{{{  words*/
static unsigned long words(unsigned long n)
{
  return (n>>2);
}
/*}}}  */
/*{{{  readword*/
static unsigned long readword(int LinkID, unsigned long address, Bool *error)
{
  unsigned char buffer[5];

  buffer[0]=1;
  buffer[1]=(address & 0xff);
  buffer[2]=((address & 0xff00)>>8);
  buffer[3]=((address & 0xff0000)>>16);
  buffer[4]=(address>>24);

  if (WriteLink(LinkID,buffer,5,TIMEOUT)!=5)
  {
    printf("Timeout!\n");
    *error=TRUE;
    return 0;
  }
  else
  {
    if (ReadLink(LinkID,buffer,4,TIMEOUT)!=4)
    {
      printf("Timeout\n");
      *error=TRUE;
      return 0;
    }
    else
    {
      *error=FALSE;
      return((buffer[3]<<24) | (buffer[2]<<16) | (buffer[1]<<8) | (buffer[0]));
    }
  }
}
/*}}}  */
/*{{{  writeword*/
static void writeword(int LinkID, unsigned long address, unsigned long data)
{
  unsigned char buffer[9];

  buffer[0]=0;

  buffer[1]=(address & 0xff);
  buffer[2]=((address & 0xff00)>>8);
  buffer[3]=((address & 0xff0000)>>16);
  buffer[4]=(address>>24);

  buffer[5]=(data & 0xff);
  buffer[6]=((data & 0xff00)>>8);
  buffer[7]=((data & 0xff0000)>>16);
  buffer[8]=(data>>24);

  if (WriteLink(LinkID,buffer,9,TIMEOUT)!=9) printf("Timeout\n");
}
/*}}}  */

/*{{{  main*/
int main(int argc, char *argv[], char *env[])
{
  /*{{{  variables*/
  int LinkID,c;
  char *linkname=getenv("TRANSPUTER");
  Bool err=FALSE;
  Bool reset=FALSE;
  /*}}}  */

  /*{{{  parse arguments*/
  while ((c=getopt(argc,argv,"?l:r"))!=-1)
  switch (c)
  {
    /*{{{  l linkname*/
    case 'l':
    {
      linkname=optarg;
      break;
    }
    /*}}}  */
    /*{{{  r*/
    case 'r': reset=TRUE; break;
    /*}}}  */
    /*{{{  ?, default*/
    case '?':
    default: err=TRUE; break;
    /*}}}  */
  }
  if (err)
  /*{{{  if err print usage*/
  {
    fprintf(stderr,"Usage: linkdiag [-r] [-l linkname]\n");
    exit(2);
  }
  /*}}}  */
  else
  /*{{{  process arguments*/
  {
    /*{{{  open link*/
    if ((LinkID=OpenLink(linkname))<0)
    {
      fprintf(stderr,"linkdiag: Can't open link %s: %s\n",linkname,strerror(errno));
      exit(1);
    }
    /*}}}  */
    /*{{{  reset if asked for*/
    if (reset)
    {
      ResetLink(LinkID);
      CloseLink(LinkID);
      exit(0);
    }
    /*}}}  */
  }
  /*}}}  */
  /*}}}  */
  /*{{{  command loop*/
  while (1)
  {
    /*{{{  variables*/
    char *p,*command,*arg1,*arg2,*arg3;
    char ln[128];
    /*}}}  */

    /*{{{  read command*/
#    ifdef READLINE
    p=readline("linkdiag> ");
#    else
    printf("linkdiag> "); fflush(stdout);
    p=fgets(ln,sizeof(ln),stdin);
#    endif
    if (p==(char*)0) break;
#    ifdef READLINE
    strcpy(ln,p);
    add_history(ln);
#    endif
    /*}}}  */
    /*{{{  parse command*/
    /*{{{  first word is command*/
    command=ln; p=ln;
    while (*p && *p!='\n' && *p!=' ') p++;
    /*}}}  */
    if (*p)
    /*{{{  there are arguments*/
    {
      *p++='\0'; while (*p==' ') p++;
      arg1=p;
      while (*p && *p!='\n' && *p!=' ') p++;
      if (*p)
      /*{{{  there are more arguments*/
      {
        *p++='\0'; while (*p==' ') p++;
        arg2=p;
        while (*p && *p!='\n' && *p!=' ') p++;
        if (*p)
        /*{{{  and more arguments*/
        {
          *p++='\0'; while (*p==' ') p++;
          arg3=p;
          while (*p && *p!='\n' && *p!=' ') p++;
          if (*p)
          /*{{{  and even more :)*/
          {
            *p++='\0';
          }
          /*}}}  */
        }
        /*}}}  */
        else arg3="";
      }
      /*}}}  */
      else arg2=arg3="";
    }
    /*}}}  */
    else arg1=arg2=arg3="";
    /*}}}  */
    /*{{{  process command*/
    /*{{{  res                    reset link*/
    if (!strcmp("res",command)) ResetLink(LinkID);
    /*}}}  */
    /*{{{  ana                    analyse link*/
    else if (!strcmp("ana",command)) AnalyseLink(LinkID);
    /*}}}  */
    /*{{{  de                     display error flag*/
    else if (!strcmp("de",command))
    {
      printf("error flag is %d\n",TestError(LinkID));
    }
    /*}}}  */
    /*{{{  se                     set error flag*/
    else if (!strcmp("se",command))
    {
      static unsigned char set_error[5]=
      {
        0x04,      /*  d8 0x04  */
        0x21,0xf0, /*  seterr   */
        /* loop      */
        0x60,0x0e  /*  j loop   */
      };

      ResetLink(LinkID);
      if (WriteLink(LinkID,set_error,5,TIMEOUT)!=5) printf("Timeout\n");
    }
    /*}}}  */
    /*{{{  ce                     clear error flag*/
    else if (!strcmp("ce",command))
    {
      static unsigned char clear_error[5]=
      {
        0x04,      /*  d8 0x04      */
        0x22,0xf9, /*  testerr      */
        /* loop          */
        0x60, 0x0e /*  j loop       */
      };

      ResetLink(LinkID);
      if (WriteLink(LinkID,clear_error,5,TIMEOUT)!=5) printf("Timeout\n");
    }
    /*}}}  */
    /*{{{  dl                     display link status*/
    else if (!strcmp("dl",command))
    {
      printf("transputer is %swriting and %sreading\n",
      TestRead(LinkID) ? "" : "not ",
      TestWrite(LinkID) ? "" : "not ");
    }
    /*}}}  */
    /*{{{  sl byte                send data to link*/
    else if (!strcmp("sl",command))
    {
      /*{{{  variables*/
      char *p=arg1;
      unsigned char c;
      unsigned long byte;
      /*}}}  */

      if (*arg1=='\0') printf("Usage: sl byte\n");
      else
      {
        byte=number(arg1,&p); c=byte;
        if (arg1==p || byte>255) printf("Invalid byte\n");
        else if (WriteLink(LinkID,&c,1,TIMEOUT)!=1) printf("Timeout\n");
      }
    }
    /*}}}  */
    /*{{{  b file                 boot transputer with file*/
    else if (!strcmp("b",command))
    {
      if (*arg1=='\0') printf("Usage: s file\n");
      else
      /*{{{  open file and boot*/
      {
        FILE *fp;

        if ((fp=fopen(arg1,"r"))==(FILE*)0) printf("Can't open %s:%s\n",arg1,strerror(errno));
        else
        /*{{{  boot*/
        {
          /*{{{  variables*/
          int c;
          unsigned char byte;
          /*}}}  */

          ResetLink(LinkID);
          while ((c=fgetc(fp))!=EOF)
          {
            byte=c;
            if (WriteLink(LinkID,&byte,1,TIMEOUT)!=1) { printf("Timeout\n"); break; }
          }
          fclose(fp);
        }
        /*}}}  */
      }
      /*}}}  */
    }
    /*}}}  */
    /*{{{  rl                     receive data from link*/
    else if (!strcmp("rl",command))
    {
      unsigned char c;

      if (*arg1=='\0')
      {
        if (ReadLink(LinkID,&c,1,TIMEOUT)!=1) printf("Timeout\n");
        else printf("0x%x\n",(unsigned int)c);
      }
      else
      {
        unsigned long len=number(arg1,&p), to=len-1, to2=to | 0xf;
        char ascii[17];
        Bool error;

        ascii[16]='\0';
        for (to=0; to<=to2; to++)
        {
          if (to<=len && (to & 0xf)==0) printf("%08lx:",to);
          if (to>=len)
          /*{{{  print blanks*/
          {
            printf("   ");
            ascii[(int)(to & 0xf)]=' ';
          }
          /*}}}  */
          else
          /*{{{  print memory contents*/
          {
            error=(ReadLink(LinkID,&c,1,TIMEOUT)!=1);
            if (!error)
            {
              unsigned int byte;

              printf(" %02x",byte=(unsigned int)(c & 0xff));
              ascii[(int)(to & 0xf)]=(byte<32 || byte>127 ? '.' : byte);
            }
          }
          /*}}}  */
          if ((to & 0xf)==0xf) printf("  %s\n",ascii);
        }
      }
    }
    /*}}}  */
    /*{{{  d address length       dump memory*/
    else if (!strcmp("d",command))
    {
      if (*arg1=='\0' || *arg2=='\0') printf("Usage: d address length\n");
      else
      /*{{{  process arguments and dump*/
      {
        unsigned long address, length;

        address=number(arg1,&p);
        if (arg1==p) printf("Invalid address\n");
        else
        {
          length=number(arg2,&p);
          if (arg2==p || length==0) printf("Invalid length\n");
          else
          /*{{{  dump*/
          {
            /*{{{  variables*/
            unsigned long from=address & ~0xf, to=address+length-1, to2=to | 0xf, data;
            char ascii[17];
            Bool error;
            /*}}}  */

            data=readword(LinkID,from,&error);
            if (!error)
            {
              ascii[16]='\0';
              for (; from<=to2; from++)
              {
                if ((from & 0xf)==0) printf("%08lx:",from);
                if (from<address || from>to)
                /*{{{  print blanks*/
                {
                  printf("   ");
                  ascii[(int)(from & 0xf)]=' ';
                }
                /*}}}  */
                else
                /*{{{  print memory contents*/
                {
                  data=readword(LinkID,from,&error);
                  if (!error)
                  {
                    unsigned int byte;

                    printf(" %02x",byte=(unsigned int)(data & 0xff));
                    ascii[(int)(from & 0xf)]=(byte<32 || byte>127 ? '.' : byte);
                  }
                }
                /*}}}  */
                if ((from & 0xf)==0xf) printf("  %s\n",ascii);
              }
            }
          }
          /*}}}  */
        }
      }
      /*}}}  */
    }
    /*}}}  */
    /*{{{  w file address length  dump memory*/
    else if (!strcmp("w",command))
    {
      if (*arg1=='\0' || *arg2=='\0' || arg3=='\0') printf("Usage: w file address length\n");
      else
      /*{{{  process arguments and dump*/
      {
        unsigned long address, length;
        FILE *fp;

        address=number(arg2,&p);
        if (arg2==p) printf("Invalid address\n");
        else
        {
          length=number(arg3,&p);
          if (arg3==p || length==0) printf("Invalid length\n");
          else if ((fp=fopen(arg1,"w"))==(FILE*)0) printf("Can't open %s:%s\n",arg1,strerror(errno));
          else
          /*{{{  dump*/
          {
            /*{{{  variables*/
            unsigned long to=address+length-1, data;
            Bool error;
            /*}}}  */

            data=readword(LinkID,address,&error);
            if (!error)
            {
              for (; address<=to; address++)
              {
                data=readword(LinkID,address,&error);
                if (!error) fputc((unsigned char)(data & 0xff),fp);
              }
            }
            fclose(fp);
          }
          /*}}}  */
        }
      }
      /*}}}  */
    }
    /*}}}  */
    /*{{{  h                      show available commands*/
    else if (!strcmp("h",command))
    {
      printf("res                    reset link\n");
      printf("ana                    analyse link\n");
      printf("de                     display error flag\n");
      printf("se                     set error flag\n");
      printf("ce                     clear error flag\n");
      printf("dl                     display link status\n");
      printf("st                     show saved status\n");
      printf("sl byte                send byte to link\n");
      printf("b file                 boot transputer with file\n");
      printf("rl [length]            receive byte(s) from link\n");
      printf("d address length       dump memory\n");
      printf("w file address length  write dump to file\n");
      printf("h                      show available commands\n");
      printf("q                      quit linkdiag\n");
    }
    /*}}}  */
    /*{{{  q                      quit*/
    else if (!strcmp("q",command)) break;
    /*}}}  */
    /*{{{  st                     status*/
    else if (!strcmp("st",command))
    {
      long val;
      Bool error;

      val=readword(LinkID,MinInt+words(9),&error);  if (!error) printf("TPtrLoc0           : 0x%lx\n",val);
      val=readword(LinkID,MinInt+words(10),&error); if (!error) printf("TPtrLoc1           : 0x%lx\n",val);
      val=readword(LinkID,MinInt+words(11),&error); if (!error) printf("WdescIntSaveLoc    : 0x%lx\n",val);
      val=readword(LinkID,MinInt+words(12),&error); if (!error) printf("IPtrIntSaveLoc     : 0x%lx\n",val);
      val=readword(LinkID,MinInt+words(13),&error); if (!error) printf("ARegIntSaveLoc     : 0x%lx\n",val);
      val=readword(LinkID,MinInt+words(14),&error); if (!error) printf("BRegIntSaveLoc     : 0x%lx\n",val);
      val=readword(LinkID,MinInt+words(15),&error); if (!error) printf("CRegIntSaveLoc     : 0x%lx\n",val);
      val=readword(LinkID,MinInt+words(16),&error); if (!error) printf("StatusIntSaveLoc   : 0x%lx\n",val);
      val=readword(LinkID,MinInt+words(17),&error); if (!error) printf("ERegOntSaveLoc     : 0x%lx\n",val);
      {
        static unsigned char clear_error[18]=
        {
          0x11,      /*   d8 0x11 */
          0xd0,      /*   stl 0x0 */
          0xd1,      /*   stl 0x1 */
          0x24,0xf2, /*   mint    */
          0x21,0xf8, /*   sthf    */
          0x24,0xf2, /*   mint    */
          0x21,0xfc, /*   stlf    */
          0x10,      /*   ldlp    */
          0x24,0xf2, /*   mint    */
          0x48,      /*   ldc 0x8 */
          0xfb,      /*   out     */
          /*            loop      */
          0x60,0x0e  /*   j loop  */
        };
        
        if (WriteLink(LinkID,clear_error,18,TIMEOUT)!=18) printf("Timeout\n");
        else
        {
          char buffer[4];

          if (ReadLink(LinkID,buffer,4,TIMEOUT)==4)
          {
            val=((buffer[3]<<24) | (buffer[2]<<16) | (buffer[1]<<8) | (buffer[0]));
            printf("IPtr    : 0x%lx\n",val);
          }
          else printf("Timeout\n");
          if (ReadLink(LinkID,buffer,4,TIMEOUT)==4)
          {
            val=((buffer[3]<<24) | (buffer[2]<<16) | (buffer[1]<<8) | (buffer[0]));
            printf("Wdesc    : 0x%lx\n",val);
          }
          else printf("Timeout\n");
        }
      }
    }
    /*}}}  */
    /*{{{  unknown command*/
    else if (*command!='\0') printf("Unknown command\n");
    /*}}}  */
    /*}}}  */
  }
  /*}}}  */
  /*{{{  clean up*/
  CloseLink(LinkID);
  /*}}}  */
  exit(0);
}
/*}}}  */
