/*{{{}}}*/
/*{{{  Notes*/
/*

The actual code is written by me, but the Notes are taken from the Inmos
iserver link.c stubs.  I left them unchanged to ensure that the "specs"
stay visible.  This version runs on Linux 0.98.5 with GCC 2.2.2d.  Please
have a look at the correctness of the ResetLink and AnalyseLink functions,
I was told this timing is good, but I have no reference from a databook.
Any comments are appreciated.

Michael Haardt (mhaardt@ftp.thp.uni-koeln.de)

*/
/*}}}  */
/*{{{  #includes*/
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include "linkio.h"
/*}}}  */
/*{{{  #defines*/
#define LinkIn     0x00
#define LinkOut    0x01
#define StatIn     0x02
#define StatOut    0x03
#define ResetFF    0x10
#define AnalyseFF  0x11
#define ErrorFF    0x10
/*}}}  */

/*{{{  Notes*/
/*

Thanks Linus!  I took this from the kernel.

*/
/*}}}  */
/*{{{  outb*/
static void inline outb(char value, unsigned short port)
{
  __asm__ volatile ("outb %0,%1"
  ::"a" ((char) value),"d" ((unsigned short) port));
}
/*}}}  */
/*{{{  inb*/
static unsigned char inline inb(unsigned short port)
{
  unsigned char _v;
  __asm__ volatile ("inb %1,%0"
  :"=a" (_v):"d" ((unsigned short) port));
  return _v;
}
/*}}}  */

/*{{{  B004OpenLink*/
/*{{{  Notes*/
/*
 *
 *   OpenLink
 *
 *   Ready the link associated with `Name'.
 *   If `Name' is NULL or "" then any free link can be used.
 *   Returns any positive integer as a link id or
 *   a negative value if the open fails.
 *
*/
/*}}}  */
int B004OpenLink(char *Name)
{
  unsigned short base;
  int res;

  if (Name==(char*)0) base=0x150;
  else if (*Name=='\0') base=0x150;
  else
  {
    if (*Name=='#') Name++;
    if (!strcmp(Name,"150")) base=0x150;
    else if (!strcmp(Name,"200")) base=0x200;
    else if (!strcmp(Name,"250")) base=0x250;
    else if (!strcmp(Name,"300")) base=0x300;
    else return ER_LINK_BAD;
  }
  if (geteuid()>0) setreuid(geteuid(),getuid());
  res=(ioperm(base+LinkIn,4,1)==-1 || ioperm(base+ResetFF,2,1)==-1);
  setreuid(0,getuid());
  return(res ? ER_LINK_CANT : base);
}
/*}}}  */
/*{{{  B004CloseLink*/
/*{{{  Notes*/
/*
 *
 *   CloseLink
 *
 *   Close the active link `LinkId'.
 *   Returns 1 on success or negative if the close failed.
 *
*/
/*}}}  */
int B004CloseLink(int LinkId)
{
  int res;

  if (geteuid()>0) setreuid(geteuid(),getuid());
  res=(ioperm(LinkId+LinkIn,4,0)==-1 || ioperm(LinkId+ResetFF,2,0)==-1);
  setreuid(0,getuid());
  return(res ? ER_LINK_CANT : SUCCEEDED);
}
/*}}}  */
/*{{{  B004ReadLink*/
/*{{{  Notes*/
/*
 *
 *   ReadLink
 *
 *   Read `Count' chars into `Buffer' from the specified link.
 *   LinkId is a vaild link identifier, opened with OpenLink.
 *   `Timeout' is a non negative integer representing tenths
 *   of a second.  A `Timeout' of zero is an infinite timeout.
 *   The timeout is for the complete operation.
 *   If `Timeout' is positive then ReadLink may return having
 *   read less that the number of chars asked for.
 *   Returns the number of chars placed in `Buffer' (which may
 *   be zero) or negative to indicate an error.
 *
*/
/*}}}  */
int B004ReadLink(int LinkId, register char *Buffer, unsigned int Count, int Timeout)
{
  /*{{{  variables*/
  register unsigned int c=Count;
  register unsigned short int Stat=LinkId+StatIn;
  register unsigned short int In=LinkId+LinkIn;
  /*}}}  */

  if (Count<1) return ER_LINK_CANT;
  if (Timeout)
  /*{{{  non blocking read*/
  {
    c++;
    while (--c)
    {
      while (Timeout>0 && (inb(Stat)&1)==0) { usleep(10000); Timeout--; }
      if (Timeout==0) break;
      *Buffer++=inb(In);
    }
  }
  /*}}}  */
  else
  /*{{{  blocking read*/
  {
    c++;
    while (--c)
    {
      /* This prevents eating all cpu time when waiting.  Once the transputer
      decides to send data, the usleep will never happen anyway. */
      while ((inb(Stat)&1)==0) usleep(10000);
      *Buffer++=inb(In);
    }
  }
  /*}}}  */
  return (Count-c);
}
/*}}}  */
/*{{{  B004WriteLink*/
/*{{{  Notes*/
/*
 *
 *   WriteLink
 *
 *   Write `Count' chars from `Buffer' to the specified link.
 *   LinkId is a vaild link identifier, opened with OpenLink.
 *   `Timeout' is a non negative integer representing tenths
 *   of a second.  A `Timeout' of zero is an infinite timeout.
 *   The timeout is for the complete operation.
 *   If `Timeout' is positive then WriteLink may return having
 *   written less that the number of chars asked for.
 *   Returns the number of chars actually written (which may
 *   be zero) or negative to indicate an error.
 *
*/
/*}}}  */
int B004WriteLink(register int LinkId, register char *Buffer, unsigned int Count, int Timeout)
{
  /*{{{  variables*/
  register unsigned int c=Count;
  register unsigned short int Stat=LinkId+StatOut;
  register unsigned short int Out=LinkId+LinkOut;
  /*}}}  */

  if (Count<1) return ER_LINK_CANT;
  if (Timeout)
  /*{{{  non blocking write*/
  {
    c++;
    while (--c)
    {
      while (Timeout>0 && (inb(Stat)&1)==0) { usleep(10000); Timeout--; }
      if (Timeout==0) break;
      outb(*Buffer++,Out);
    }
  }
  /*}}}  */
  else
  /*{{{  blocking write*/
  {
    c++;
    while (--c)
    {
      while ((inb(Stat)&1)==0) usleep(10000);
      outb(*Buffer++,Out);
    }
  }
  /*}}}  */
  return (Count-c);
}
/*}}}  */
/*{{{  B004ResetLink*/
/*{{{  Notes*/
/*
 *
 *   ResetLink
 *
 *   Reset the subsystem associated with the specified link.
 *   Returns 1 if the reset is successful, negative otherwise.
 *
*/
/*}}}  */
int B004ResetLink(int LinkId)
{
  outb(0,(unsigned short)(LinkId+AnalyseFF));
  outb(0,(unsigned short)(LinkId+ResetFF)); usleep(100000);
  outb(1,(unsigned short)(LinkId+ResetFF)); usleep(100000);
  outb(0,(unsigned short)(LinkId+ResetFF)); usleep(100000);
  return SUCCEEDED;
}
/*}}}  */
/*{{{  B004AnalyseLink*/
/*{{{  Notes*/
/*
 *
 *   AnalyseLink
 *
 *   Analyse the subsystem associated with the specified link.
 *   Returns 1 if the analyse is successful, negative otherwise.
 *
*/
/*}}}  */
int B004AnalyseLink(int LinkId)
{
  outb(0,(unsigned short)(LinkId+AnalyseFF));
  outb(0,(unsigned short)(LinkId+ResetFF)); usleep(100000);
  outb(1,(unsigned short)(LinkId+AnalyseFF)); usleep(100000);
  outb(1,(unsigned short)(LinkId+ResetFF)); usleep(100000);
  outb(0,(unsigned short)(LinkId+ResetFF)); usleep(100000);
  outb(0,(unsigned short)(LinkId+AnalyseFF)); usleep(100000);
  return SUCCEEDED;
}
/*}}}  */
/*{{{  B004TestError*/
/*{{{  Notes*/
/*
 *
 *   TestError
 *
 *   Test the error status associated with the specified link.
 *   Returns 1 if error is set, 0 if it is not and
 *   negative to indicate an error.
 *
*/
/*}}}  */
int B004TestError(int LinkId)
{
  return (!(inb((unsigned short)(LinkId+ErrorFF))&1));
}
/*}}}  */
/*{{{  B004TestRead*/
/*{{{  Notes*/
/*
 *
 *   TestRead
 *
 *   Test input status of the link.
 *   Returns 1 if ReadLink will return one byte without timeout,
 *   0 if it may not and negative to indicate an error.
 *
*/
/*}}}  */
int B004TestRead(int LinkId)
{
  return (inb((unsigned short)(LinkId+StatIn))&1);
}
/*}}}  */
/*{{{  B004TestWrite*/
/*{{{  Notes*/
/*
 *
 *   TestWrite
 *
 *   Test output status of the link.
 *   Returns 1 if WriteLink can write one byte without timeout,
 *   0 if it may not and negative to indicate an error.
 *
*/
/*}}}  */
int B004TestWrite(int LinkId)
{
  return (inb((unsigned short)(LinkId+StatOut))&1);
}
/*}}}  */
