/*
-- ----------------------------------------------------------------------------
--
--     Object Name : version.c
--     Revision    : 1
--
--     Copyright INMOS Limited, 1987, 1988.
--     All Rights Reserved.
--
--     DESCRIPTION
--         Afserver host and version specific command functions.
--
--     NOTES
--         This module should be changed if porting to a new host.
--
--     HISTORY
--         25-Feb-1988    Antony King    Last change.
--         12-Apr-1988    Antony King    Added this comment.
--         12-Apr-1988    Antony King    Change call to rename function, now
--                                       compatable only with MS C V4.0 & V5.0.
--         20-Apr-1988    Antony King    Fixed bug in testing of i/o streams
--                                       for errors - now use ferror().
--         19-May-1988    Russell Wayman Changed call to rename back to make
--                                       it compile under MS C V3.0.
--
-- ----------------------------------------------------------------------------
*/

/* Included files */

#include <linux/types.h>
#include <linux/termios.h>

#include "version.h"
#include "afserver.h"

/* External procedures */

extern char *getenv();

extern void host_end();

#define ORG_MODE 0
#define GET_MODE 1
#define POLL_MODE 2

static int TermMode = ORG_MODE;

static struct termios OrgMode, CurMode;


/*
-- ----------------------------------------------------------------------------
--
--     Function Name : sys_break
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Terminate afserver if a user break (control-break on the keyboard).
--
-- ----------------------------------------------------------------------------
*/

static void sys_break ()
{
    fprintf(stderr, "Server terminated: aborted by user\n");
    host_end(EXIT_ABORT);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_terminal
--
--     Input Parameters :
--         int record_length           - length of string to be read.
--         struct STREAM_DESC *streamp - pointer to input stream.
--
--     Output Parameters :
--         char *buffer                - string read from keyboard.
--         int *bytes_read             - actual length of string read.
--
--     Result :
--         int                         - result of operation.
--
--     DESCRIPTION
--         Read from the keyboard a string of a given length.
--
-- ----------------------------------------------------------------------------
*/

int read_terminal (buffer, record_length, streamp, bytes_read)
    char *buffer;
    struct STREAM_DESC *streamp;
    int record_length, *bytes_read;
{
    int result = OPERATION_OK;

    if (fgets(buffer, record_length + 1, streamp->fileptr) != NULL)
        *bytes_read = strlen(buffer);
    else if (ferror(streamp->fileptr))
    {
        streamp->result = make_af_err(ferror(streamp->fileptr));
        result = OPERATIONFAILED_ERR;
    }
    else if (feof(streamp->fileptr))
    {
        streamp->result = ENDOFFILE;
        result = ENDOFFILE;
    }
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : write_terminal
--
--     Input Parameters :
--         char *buffer                - string sent to the screen.
--         int record_length           - length of string to be written.
--         struct STREAM_DESC *streamp - pointer to output stream.
--
--     Output Parameters :
--         int *bytes_written          - actual length of string written.
--
--     Result :
--         int                         - result of operation.
--
--     DESCRIPTION
--         Write to the screen a string of a given length.
--
-- ----------------------------------------------------------------------------
*/

int write_terminal (buffer, record_length, streamp, bytes_written)
    char *buffer;
    struct STREAM_DESC *streamp;
    int record_length, *bytes_written;
{
    int result = OPERATION_OK;

    *bytes_written = fwrite(buffer, 1, record_length, streamp->fileptr);
    if (ferror(streamp->fileptr))
    {
        streamp->result = make_af_err(ferror(streamp->fileptr));
        result = OPERATIONFAILED_ERR;
    }
    fflush(streamp->fileptr);
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : delete_file
--
--     Input Parameters :
--         char *name - name of file to be deleted.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int        - result of operation.
--
--     DESCRIPTION
--         Delete the given file name from file system.
--
-- ----------------------------------------------------------------------------
*/

int delete_file (name)
    char *name;
{
    return(unlink(name));
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : host_end
--
--     Input Parameters :
--         int result - value of exit result.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Close down afserver and exit.
--
-- ----------------------------------------------------------------------------
*/

void host_end (result)
    int result;
{
    ioctl(0,TCSETS,&OrgMode);
    filer_close();
    exit(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : sys_error
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Terminate afserver if error in the transputer system.
--
-- ----------------------------------------------------------------------------
*/

void sys_error ()
{
    fprintf(stderr, "Server terminated: error in transputer system\n");
    host_end(EXIT_ABORT);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : host_begin
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Setup up interrupt traps for afserver.
--
-- ----------------------------------------------------------------------------
*/

void host_begin ()
{
    signal(SIGINT, sys_break);
    ioctl(0,TCGETS,&OrgMode);
    ioctl(0,TCGETS,&CurMode);

}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_key_call
--
--     Input Parameters :
--         int wait - flag to determine wether to wait for a key to be pressed.
--
--     Output Parameters :
--         int *chr - character read from the keyboard (if any).
--
--     Result :
--         int      - result of operation.
--
--     DESCRIPTION
--         Performs afserver commands ReadKey.Cmd and ReadKeyWait.Cmd.
--
-- ----------------------------------------------------------------------------
*/


int read_key_call (chr, wait)
    int wait, *chr;
{
   if (wait)
   {
     if ( TermMode == ORG_MODE )
     {
  	CurMode.c_iflag &= ~ICRNL;
	CurMode.c_lflag &= ~(ICANON | ECHO);
	CurMode.c_cc[VTIME] = 0;
	CurMode.c_cc[VMIN] = 1;
	ioctl( 0, TCSETS, &CurMode );
	TermMode = GET_MODE;
     }
     else if ( TermMode == POLL_MODE )
     {
       CurMode.c_cc[VTIME] = 0;
       CurMode.c_cc[VMIN] = 1;
       ioctl( 0, TCSETS, &CurMode );
       TermMode = GET_MODE;
     }
   }
   else
   {
     if ( TermMode == ORG_MODE )
     {
       CurMode.c_iflag &= ~ICRNL;
       CurMode.c_lflag &= ~(ICANON | ECHO);
       CurMode.c_cc[VTIME] = 1;
       CurMode.c_cc[VMIN] = 0;
       ioctl( 0, TCSETS, &CurMode );
       TermMode = POLL_MODE;
     }
     else if ( TermMode == GET_MODE )
     {
       CurMode.c_cc[VTIME] = 1;
       CurMode.c_cc[VMIN] = 0;
       ioctl( 0, TCSETS, &CurMode );
       TermMode = POLL_MODE;
     }
   }

   (void)read(0, chr , 1);

   if (*chr == 0)
     return( OPERATIONFAILED_ERR );
   else
     return( OPERATION_OK );
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_time_call
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         long int *seconds - current time in seconds.
--
--     Result :
--         int               - result of operation.
--
--     DESCRIPTION
--         Performs afserver command ReadTime.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int read_time_call (seconds)
    long int *seconds;
{
    (void) time(seconds);
    return(OPERATION_OK);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : rename_file_call
--
--     Input Parameters :
--         char old_name[] - old name of file.
--         int old_length  - old file name length.
--         char new_name[] - new name of file.
--         int new_length  - new file name length.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int             - result of operation.
--
--     DESCRIPTION
--         Performs afserver command RenameFile.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int rename_file_call (old_name, old_length, new_name, new_length)
    int old_length, new_length;
    char old_name[], new_name[];
{
    int result = OPERATION_OK;

    if ((old_length >= 0) && (old_length <= FILE_NAME_LENGTH))
    {
        if ((new_length >= 0) && (new_length <= FILE_NAME_LENGTH))
        {
            int res;

            old_name[old_length] = '\0';
            new_name[new_length] = '\0';
            /*res = rename(new_name, old_name);*/       /* If ok, returns 0 */

            res = rename(old_name, new_name);

            if (res != 0)                           /* Not ok, returns non-zero */
                result = OPERATIONFAILED_ERR;
        }
        else
            result = INVALIDFILENAMELENGTH_ERR;
    }
    else
        result = INVALIDFILENAMELENGTH_ERR;
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : run_command_call
--
--     Input Parameters :
--         char buffer[] - command line to be executed.
--         int length    - command line length.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int           - result of operation.
--
--     DESCRIPTION
--         Performs afserver command RunCommand.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int run_command_call (buffer, length)
    int length;
    char buffer[];
{
    int result = OPERATION_OK;

    if ((length >= 0) && (length <= RECORD_LENGTH))
    {
        int res;

        buffer[length] = '\0';
        res = system(buffer);                   /* If ok, returns 0 */
        if (res != 0)                           /* Not ok, returns non-zero */
            result = OPERATIONFAILED_ERR;
    }
    else
        result = INVALIDRECORDLENGTH_ERR;
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_environment_call
--
--     Input Parameters :
--         char old_name[] - name of variable.
--         int old_length  - variable length.
--
--     Output Parameters :
--         char new_name[] - value of variable.
--         int *new_length - value length.
--
--     Result :
--         int             - result of operation.
--
--     DESCRIPTION
--         Performs afserver command ReadEnvironment.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int read_environment_call (old_name, old_length, new_name, new_length)
    int old_length, *new_length;
    char old_name[], new_name[];
{
    char *ptr;
    int result = OPERATION_OK;

    *new_length = 0;
    if ((old_length >= 0) && (old_length <= RECORD_LENGTH))
    {
        old_name[old_length] = '\0';
        if ((ptr = getenv(old_name)) != NULL)
        {
            if ((*new_length = strlen(ptr)) <= RECORD_LENGTH)
                strcpy(new_name, ptr);
            else
            {
                *new_length = RECORD_LENGTH;
                new_name[*new_length] = '\0';
                strncpy(new_name, ptr, *new_length);
            }
        }
        else
            result = OPERATIONFAILED_ERR;
    }
    else
        result = INVALIDRECORDLENGTH_ERR;
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : send_block_call
--
--     Input Parameters :
--         long int destination - destination address of data block.
--         char buffer[]        - data block written.
--         int length           - data block length.
--
--     Output Parameters :
--         int *bytes_written   - actual length written.
--
--     Result :
--         int                  - result of operation.
--
--     DESCRIPTION
--         Performs afserver command SendBlock.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int send_block_call (destination, buffer, length, bytes_written)
    char buffer[];
    long int destination;
    int length, *bytes_written;
{
    fprintf(stderr, "afserver: The SendBlock.Cmd is not supported by linux\n");

    return(NOTIMPLEMENTED_ERR);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : receive_block_call
--
--     Input Parameters :
--         long int source - source address of data block.
--         int length      - data block length to be read.
--
--     Output Parameters :
--         char buffer[]   - data block read.
--         int *bytes_read - actual length read.
--
--     Result :
--         int             - result of operation.
--
--     DESCRIPTION
--         Performs afserver command ReceiveBlock.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int receive_block_call (source, length, buffer, bytes_read)
    char buffer[];
    long int source;
    int length, *bytes_read;
{
    fprintf(stderr, "afserver: The ReceiveBlock.Cmd is not supported by linux\n");

    return(NOTIMPLEMENTED_ERR);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_regs_call
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         char buffer[] - register block read.
--         int *length   - register block length.
--
--     Result :
--         int           - result of operation.
--
--     DESCRIPTION
--         Performs afserver command ReadRegs.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int read_regs_call (buffer, length)
    int *length;
    char buffer[];
{
    fprintf(stderr, "afserver: The ReadRegs.Cmd is not supported by linux\n");

    return(NOTIMPLEMENTED_ERR);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : call_interrupt_call
--
--     Input Parameters :
--         int intrpt_no   - interrupt number.
--         char input[]    - register block before interrupt.
--         int in_length   - register block length.
--
--     Output Parameters :
--         int *flag       - value of carry flag.
--         char output[]   - register block after interrupt.
--         int *out_length - register block length.
--
--     Result :
--         int             - result of operation.
--
--     DESCRIPTION
--         Performs afserver command CallInterrupt.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int call_interrupt_call (intrpt_no, input, in_length, flag, output, out_length)
    char input[], output[];
    int intrpt_no, in_length, *flag, *out_length;
{
    fprintf(stderr, "afserver: The CallInterrupt.Cmd is not supported by linux\n");

    return(NOTIMPLEMENTED_ERR);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : port_read_call
--
--     Input Parameters :
--         int address - address of the port to be read from.
--
--     Output Parameters :
--         int *value  - value read from the port.
--
--     Result :
--         int         - result of operation.
--
--     DESCRIPTION
--         Performs afserver command PortRead.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int port_read_call (address, value)
    int address, *value;
{
    fprintf(stderr, "afserver: The PortRead.Cmd is not supported by linux\n");

    return(NOTIMPLEMENTED_ERR);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : port_write_call
--
--     Input Parameters :
--         int address - address of the port to be wrtten to.
--         int value   - value to be wrtten to the port.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int         - result of operation.
--
--     DESCRIPTION
--         Performs afserver command PortWrite.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int port_write_call (address, value)
    int address, value;
{
    fprintf(stderr, "afserver: The PortWrite.Cmd is not supported by linux\n");

    return(NOTIMPLEMENTED_ERR);
}
