/*
-- ----------------------------------------------------------------------------
--
--     Object Name : linkio.c
--     Revision    : 1
--
--     Copyright INMOS Limited, 1987, 1988.
--     All Rights Reserved.
--
--     DESCRIPTION
--         Afserver host specific link driver routines.
--
--     NOTES
--         This module should be changed if porting to a new host or transputer
--         board.
--
--     HISTORY
--         25-Feb-1988    Antony King    Last change.
--         12-Apr-1988    Antony King    Added this comment.
--         20-Apr-1988    Antony King    Added extra delay loops in reset_root
--                                       and reset_analyse_root.
--	   18-Feb-1993	  Christoph Niemann  Linux Version
--
-- ----------------------------------------------------------------------------
*/

/* Included files */

#include "version.h"
#include "afserver.h"
#include "../linkdiag/link.h"
#include <fcntl.h>
#include <sys/stat.h>
#include <stdio.h>
#include <errno.h>

/* External variables */

extern int test_error_flag;


/* Global variables */

int Link;

char *link_device;

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : error_test
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Test if an error is present in the transputer system. If set then
--         immediately exit afserver.
--
-- ----------------------------------------------------------------------------
*/

void error_test ()
{
    if (test_error_flag && TestError(Link))
        sys_error();
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : byte_from_link
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int - byte read from transputer.
--
--     DESCRIPTION
--         Receive a byte from the transputer.
--
-- ----------------------------------------------------------------------------
*/

int byte_from_link ()
{
    char ret;
#ifdef DEBUG
    printf("DEBUG: Reading a byte from link\n");
#endif
    ReadLink(Link, &ret, 1, 0);
    return (int) ret;
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : byte_to_link
--
--     Input Parameters :
--         int ch - byte to be sent.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Send a byte to the transputer.
--
-- ----------------------------------------------------------------------------
*/

void byte_to_link (ch)
    int ch;
{
	char Byte = (int) ch;
#ifdef DEBUG
	printf("DEBUG: Writing a byte to the link\n");
#endif
	WriteLink(Link, &Byte, 1, 0);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : byte_to_link_t
--
--     Input Parameters :
--         int ch - byte to be sent.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int    - TRUE if the byte is sent, FALSE otherwise.
--
--     DESCRIPTION
--         Send a byte to the transputer within a specified time, or timeout.
--
-- ----------------------------------------------------------------------------
*/

int byte_to_link_t (ch)
    int ch;
{
  char Byte = (int) ch;
#ifdef DEBUG
  printf("DEBUG: writing a byte to the link\n");
#endif
  if ( WriteLink(Link, &Byte, 1, 0) == 1)
    return(TRUE);
  else
    return(FALSE);
}

void reset_root ()
{
#ifdef DEBUG
  printf("DEBUG: resetting transputer\n");
#endif
  ResetLink(Link);
}


void reset_analyse_root ()
{
#ifdef DEBUG
  printf("DEBUG: analysing transputer\n");
#endif
  AnalyseLink(Link);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : word_from_link
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int - word received from the transputer.
--
--     DESCRIPTION
--         Receive a word from the transputer. This routine is host- and
--         transputer- word-length dependent. Here we assume a host int length
--         of two bytes, and a transputer int length of four bytes.
--
-- ----------------------------------------------------------------------------
*/

int word_from_link ()
{
    short Word, Dummy;
    ReadLink(Link, &Word, 2, 0);
    ReadLink(Link, &Dummy, 2, 0);
#ifdef DEBUG
  printf("DEBUG: reading word %d from link\n", Word);
#endif
    return( (int) Word );
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : slice_from_link
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         int *length - length of slice received from the transputer.
--         char *slice - slice received from the transputer.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Receive a slice from the transputer. The slice consists of a length
--         followed by a sequence of bytes.
--
-- ----------------------------------------------------------------------------
*/

void slice_from_link (length, slice)
    int *length;
    char *slice;
{
    int real_len = word_from_link();

    *length = real_len;
#ifdef REV_A_FIX
    if (real_len == 1) real_len = 2;
#endif
#ifdef DEBUG
    printf("DEBUG: reading char* from link, length = %d\n", real_len);
#endif
    ReadLink(Link, slice, real_len, 0);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : trunc_slice_from_link
--
--     Input Parameters :
--         int max_length - maximum length of slice that can be read.
--
--     Output Parameters :
--         char *slice    - slice received from the transputer.
--
--     Result :
--         int            - length of slice received.
--
--     DESCRIPTION
--         Receive a slice from the transputer whose maximum length is
--         max_length. If the length read in is longer than this, the remainder
--         of the slice is read in, but ignored.
--
-- ----------------------------------------------------------------------------
*/

int trunc_slice_from_link (max_length, slice)
    char *slice;
    int max_length;
{
    char* temp = slice;
    int real_len, length, i;
    char dummy;


    length = real_len = word_from_link();
#ifdef REV_A_FIX
    if (real_len == 1) real_len = 2;
#endif
    if (real_len > max_length)
    {
	read(Link, slice, max_length);
        for (i = max_length; i < real_len; i++)
            ReadLink(Link, &dummy, 1, 0);
	length=max_length;
    }
    else
    {
	ReadLink(Link, slice, real_len, 0);
	length=real_len;
    }
    return(length);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : long_word_from_link
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         long int - word received from the transputer.
--
--     DESCRIPTION
--         Receive a word from the transputer. This routine is host- and
--         transputer- word-length dependent. Here we assume a host int length
--         of four bytes, and a transputer int length of four bytes.
--
-- ----------------------------------------------------------------------------
*/

long int long_word_from_link ()
{
    int t;
    ReadLink(Link, &t, 4, 0);
#ifdef DEBUG
  printf("DEBUG: reading long word %d from link\n", t);
#endif
    return(t);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : word_to_link
--
--     Input Parameters :
--         int w - word to be sent to the transputer.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Send a word to the transputer, least significant byte first. This
--         routine is host- and transputer- word-length dependent. Here we
--         assume a host int length of two bytes, and a transputer int length
--         of four bytes.
--
-- ----------------------------------------------------------------------------
*/

void word_to_link (w)
    int w;
{
#ifdef DEBUG
  printf("DEBUG: writing long word %d to link\n", w);
#endif
	WriteLink(Link, &w, 4, 0);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : slice_to_link
--
--     Input Parameters :
--         int length  - length of slice sent to the transputer.
--         char *slice - slice sent to the transputer.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Send a slice to the transputer. The slice consists of a length
--         followed by a sequence of bytes.
--
-- ----------------------------------------------------------------------------
*/

void slice_to_link (length, slice)
    int length;
    char *slice;
{
    int real_len = length;

    word_to_link(real_len);
#ifdef REV_A_FIX
    if (real_len == 1) real_len = 2;
#endif
	WriteLink(Link, slice, real_len, 0);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : init_root
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int - always returns TRUE.
--
--     DESCRIPTION
--         Initialise the data used by the afserver to poll and set the
--         transputer with.
--
-- ----------------------------------------------------------------------------
*/

int init_root ()
{
  Link = OpenLink(link_device ? link_device : getenv("TRANSPUTER") );
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : peek_word
--
--     Input Parameters :
--         int i    - offset of word from MININT to be peeked.
--
--     Output Parameters :
--         None.
--
--     Result :
--         long int - value of word that has been peeked.
--
--     DESCRIPTION
--         Read the contents of the i'th word above MININT of the transputer.
--
-- ----------------------------------------------------------------------------
*/

long int peek_word(i)
    int i;
{
    byte_to_link(T_PEEK);
    word_to_link(MOSTNEG_INT + i);
    return(word_from_link());
}
