/* #includes *//*{{{C}}}*//*{{{*/
#ifndef NO_POSIX_SOURCE
#undef  _POSIX_SOURCE
#define _POSIX_SOURCE   1
#undef  _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 2
#endif

#ifdef DMALLOC
#include "dmalloc.h"
#endif

#include <assert.h>
#include <curses.h>
#include <ctype.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#ifdef NEED_BCOPY
#define memmove(dst,src,len) bcopy(src,dst,len)
#endif

#include "cat.h"
#include "complete.h"
#include "display.h"
#include "macro.h"
#include "misc.h"
#include "msgline.h"
/*}}}*/
/* #defines */ /*{{{*/
#ifdef BROKEN_CURSES_PROTOTYPES
#define CAST_CHAR_PTR (char*)
#else
#define CAST_CHAR_PTR
#endif
/*}}}*/

/* variables *//*{{{*/
static int msg=0;
/*}}}*/

/* showmsg      -- show a message *//*{{{*/
void showmsg(const char *fmt, ...)
{
  /* variables *//*{{{*/
  va_list ap;
  int x,maxx,maxy;
  char c;
  int oldy,oldx;
  /*}}}*/
  
  if (!msg && fmt==(const char*)0) return;
  (void)getyx(stdscr,oldy,oldx);
  (void)move(LINES-1,0);
  (void)clrtoeol();
  if (fmt==(const char*)0) msg=0;
  else
  {
    va_start(ap,fmt);
    x=0;
    getmaxyx(stdscr,maxy,maxx);
    while (*fmt) if ((c=*fmt++)=='%') switch (c=*fmt++)
    {
      /* d *//*{{{*/
      case 'd':
      {
        int u;
        char ustr[10],*s;

        u=va_arg(ap,int);
        if (u<0)
        {
          (void)addch('-');
          u=-u;
        }
        s=ustr;
        do { *s=(u%10)+'0'; u/=10; if (u) ++s; } while (u);
        while (x<maxx-1 && s>=ustr) { (void)addch((chtype)((unsigned char)*s--)); ++x; }
        break;
      }
      /*}}}*/
      /* s *//*{{{*/
      case 's':
      {
        char *s;

        s=va_arg(ap,char*);
        while (x<maxx-1 && *s) { (void)addch((chtype)((unsigned char)*s)); ++s; ++x; }
        break;
      }
      /*}}}*/
      /* c *//*{{{*/
      case 'c':
      {
        char ch;

        ch=va_arg(ap,char);
        if (x<maxx-1) { (void)addch((chtype)ch); ++x; }
        break;
      }
      /*}}}*/
      /* o *//*{{{*/
      case 'o':
      {
        char s[10],*p;
        int i,oc;

        oc=va_arg(ap,int);
        for (i=0; oc; oc>>=3,++i)
        {
          s[i]=(((unsigned int)((unsigned char)oc))&7)+'0';
        }
        s[i]='\\';
        p=s+i;
        while (x<maxx-1 && p>=s) { (void)addch((chtype)((unsigned char)*p--)); ++x; }
        break;
      }
      /*}}}*/
    }
    else if (x<maxx-1)
    {
      (void)addch((chtype)((unsigned char)c));
      ++x;
    }
    va_end(ap);
    msg=1;
  }
  (void)move(oldy,oldx);
}
/*}}}*/
/* editline     -- line editor *//*{{{*/
int editline(char *buf, size_t size, const char *prompt, size_t *x, size_t *offx, char *history[HISTORY_LEN])
{
  /* variables */ /*{{{*/
  size_t promptlen;
  int i,mx,my,insert,inhist;
  chtype c;
  /*}}}*/

  /* asserts */ /*{{{*/
  assert(buf!=(char*)0);
  assert(prompt!=(char*)0);  
  assert(x!=(size_t*)0);  
  assert(offx!=(size_t*)0);
  /*}}}*/
  getmaxyx(stdscr,my,mx);
  promptlen=strlen(prompt)+1;
  (void)mvwaddstr(stdscr,LINES-1,0,CAST_CHAR_PTR prompt); (void)addch(' ');
  insert=1;
  inhist=-1;
  do
  {
    int dx,di;
    const char *ds;

    /* correct offx so cursor stays visible */ /*{{{*/
    if (*x<*offx) *offx=*x;
    again:
    assert(*x>=*offx);
    ds=buf;
    for (dx=0; dx<(int)*offx && *ds; ++dx,++ds);
    for (dx=0,di=0; di<(int)(*x-*offx) && *ds; ++dx,++di,++ds) if (*ds>='\0' && *ds<='\037') ++dx;
    if (dx>(mx-(int)promptlen-1)) { ++*offx; goto again; }
    /*}}}*/
    /* display buffer */ /*{{{*/
    (void)move(LINES-1,(int)promptlen);
    for (i=di=promptlen; buf[i-promptlen+*offx]!='\0' && di<mx; ++i,++di)
    {
      if (buf[i-promptlen+*offx]>='\0' && buf[i-promptlen+*offx]<='\037')
      {
        (void)addch('^');
        if (i<mx) (void)addch(buf[i-promptlen+*offx]+'A'-1);
        ++di;
      }
      else if (di<mx) (void)addch((chtype)((unsigned char)buf[i-promptlen+*offx]));
    }
    if (di!=mx) (void)clrtoeol();
    /*}}}*/
    /* show cursor */ /*{{{*/
    (void)move(LINES-1,(int)(dx+promptlen));
    /*}}}*/
    c=mc_get();
    switch (c)
    {
      /* C-a, KEY_BEG       -- beginning of line         */ /*{{{*/
      case '\001':
      case KEY_BEG:
      {
        *x=0;
        break;
      }  
      /*}}}*/
      /* C-b, KEY_LEFT      -- backward character        */ /*{{{*/
      case KEY_LEFT:
      case '\002': if (*x>0) --(*x); break;
      /*}}}*/
      /* C-f, KEY_RIGHT     -- forward character         */ /*{{{*/
      case '\006':
      case KEY_RIGHT: if (*x<strlen(buf)) ++(*x); break;
      /*}}}*/
      /* C-h, KEY_BACKSPACE -- delete previous character */ /*{{{*/
      case '\010':
      case KEY_BACKSPACE:
      {
        if (*x>0)
        {
          memmove(buf+*x-1,buf+*x,strlen(buf+*x)+1);
          --(*x);
        }
        break;
      }
      /*}}}*/
      /* C-d, KEY_DC        -- delete character          */ /*{{{*/
      case '\004':
      case KEY_DC:
      {
        if (*x<strlen(buf)) memmove(buf+*x,buf+*x+1,strlen(buf+*x));
        break;
      }
      /*}}}*/
      /* C-e, KEY_END       -- end of line               */ /*{{{*/
      case '\005':
      case KEY_END:
      {
        *x=strlen(buf);
        break;
      }  
      /*}}}*/
      /* C-i                -- complete file             */ /*{{{*/
      case '\t':
      {
        char *s,*r,ch;
        
        ch=buf[*x];
        buf[*x]='\0';
        if ((r=s=completefile(buf)))
        {
          buf[*x]=ch;
          for (; *r; ++r)
          {
            if (strlen(buf)==(size-1)) break;
            memmove(buf+*x+1,buf+*x,strlen(buf)-*x+1);
            *(buf+*x)=*r;
            ++(*x);
          }  
          free(s);
        }
        else buf[*x]=ch;
        break;
      }
      /*}}}*/
      /* C-l                -- redraw screen             */ /*{{{*/
      case '\014': (void)touchwin(curscr); (void)wrefresh(curscr); break;
      /*}}}*/
      /* KEY_IC             -- toggle insert mode        */ /*{{{*/
      case KEY_IC:
      {
        insert=1-insert;
        break;
      }  
      /*}}}*/
      /* KEY_EIC *//*{{{*/
      case KEY_EIC:
      {
        insert=0;
        break;
      }
      /*}}}*/
      /* ^T *//*{{{*/
      case '\024':
      {
        if (*x>0 && (strlen(buf)<(size-1) || *x!=strlen(buf)))
        {
          char tc;

          tc=*(buf+*x);
          *(buf+*x)=*(buf+*x-1);
          if (tc=='\0')
          {
            tc=' ';
            *(buf+*x+1)='\0';
          }
          *(buf+*x-1)=tc;
          ++(*x);
        }
        break;
      }
      /*}}}*/
      /* ^K, KEY_DL *//*{{{*/
      case '\013':
      case KEY_DL:
      {
        *(buf+*x)='\0';
        break;
      }
      /*}}}*/
      /* ^P, KEY_UP *//*{{{*/
      case KEY_UP:
      case '\020':
      {
        if (history!=(char**)0 && inhist<(HISTORY_LEN-1) && history[inhist+1]!=(char*)0)
        {
          strcpy(buf,history[++inhist]);
          *x=0;
          *offx=0;
        }
        break;
      }
      /*}}}*/
      /* ^N, KEY_DOWN *//*{{{*/
      case KEY_DOWN:
      case '\016':
      {
        if (history!=(char**)0 && inhist>0)
        {
          strcpy(buf,history[--inhist]);
          *x=0;
          *offx=0;
        }
        break;
      }
      /*}}}*/
      /* C-q               -- quote character *//*{{{*/
      case '\021':
      {
        chtype qc,qc2;

        qc=mc_get();
        if (qc=='\0' || strlen(buf)>=(size-2)) break;
        if (ds_compose(qc,0))
        {
          qc2=mc_get();
          if (qc2=='\0') break;
          if ((qc=ds_compose(qc,qc2))==0) break;
        }
        if (insert) memmove(buf+*x+1,buf+*x,strlen(buf)-*x+1);
        else if (*x==strlen(buf)) *(buf+*x+1)='\0';
        *(buf+*x)=qc;
        ++(*x);
        break;
      }
      /*}}}*/
      /* default *//*{{{*/
      default:
      {
        if (c<' ' || c>=256 || strlen(buf)==(size-1)) break;
        if (insert) memmove(buf+*x+1,buf+*x,strlen(buf)-*x+1);
        else if (*x==strlen(buf)) *(buf+*x+1)='\0';
        *(buf+*x)=c;
        ++(*x);
        break;
      }
      /*}}}*/
    }
  } while (c!=KEY_ENTER && c!=(chtype)'\n' && c!=(chtype)'\r' && c!='\007' && c!=KEY_CANCEL);
  (void)move(LINES-1,0);
  (void)clrtoeol();
  if (c==KEY_CANCEL || c=='\007')
  {
    return 0;
  }
  else
  {
    if (history!=(char**)0)
    {
      if (history[HISTORY_LEN-1]) free(history[HISTORY_LEN-1]);
      memmove(history+1,history,sizeof(history[0])*(HISTORY_LEN-1));
      history[0]=mystrmalloc(buf);
    }
    return 1;
  }
}
/*}}}*/
/* menuline     -- one line menu *//*{{{*/
/* Notes *//*{{{*/
/*

The choices are terminated by the last element having a (const char*)str
field.  Each item can be chosen by tolower(*str) or by the key stored in
the c field.  Use a space as first character of str, if you only want the
function key to work.

*/
/*}}}*/
int menuline(const char *prompt, const MenuChoice *choice, int curx)
{
  /* variables *//*{{{*/
  int maxx,x,offx,width,mx,my;
  chtype c;
  size_t promptlen;
  int oldy,oldx;
  /*}}}*/

  /* asserts *//*{{{*/
  assert(prompt!=(const char*)0);
  assert(choice!=(const MenuChoice*)0);
  assert(curx>=0);
  /*}}}*/
  getyx(stdscr,oldy,oldx);
  getmaxyx(stdscr,my,mx);
  (void)mvwaddstr(stdscr,LINES-1,0,CAST_CHAR_PTR prompt);
  promptlen=strlen(prompt);
  for (maxx=0; (choice+maxx)->str!=(const char*)0; ++maxx);
  offx=0;
  do
  {
    (void)move(LINES-1,(int)promptlen);
    /* correct offset so choice is visible *//*{{{*/
    if (curx<=offx) offx=curx;
    else do
    {
      for (width=(int)promptlen,x=offx; x<maxx && width+(int)strlen((choice+x)->str+1)+1<=mx; width+=strlen((choice+x)->str)+1,++x);
      --x;
      if (x<curx) ++offx;
    } while (x<curx);
    /*}}}*/
    /* show visible choices *//*{{{*/
    for (width=promptlen,x=offx; x<maxx && width+(int)strlen((choice+x)->str+1)+1<=mx; width+=strlen((choice+x)->str+1)+1,++x)
    {
      (void)addch(' ');
      if (x==curx) (void)attron(A_REVERSE);
      (void)addstr((char*)(choice+x)->str+1);
      if (x==curx) (void)attroff(A_REVERSE);
    }
    /*}}}*/
    if (width<mx) (void)clrtoeol();
    (void)move(oldy,oldx);
    switch (c=mc_get())
    {
      /* KEY_LEFT              -- move to previous item *//*{{{*/
      case '\002':
      case KEY_BACKSPACE:
      case KEY_LEFT: if (curx>0) --curx; else curx=maxx-1; break;
      /*}}}*/
      /* Space, Tab, KEY_RIGHT -- move to next item *//*{{{*/
      case '\006':
      case ' ':
      case '\t':
      case KEY_RIGHT: if (curx<(maxx-1)) ++curx; else curx=0; break;
      /*}}}*/
      /* default               -- search choice keys *//*{{{*/
      default:
      {
        int i;

        for (i=0; (choice+i)->str!=(const char*)0; ++i)
        {
          if ((c<256 && tolower(c)==*((choice+i)->str)) || (choice+i)->c==c)
          {
            c=KEY_ENTER;
            curx=i;
          }
        }
      }
      /*}}}*/
    }
  }
  while (c!=KEY_ENTER && c!='\n' && c!='\r' && c!=KEY_CANCEL && c!='\007');
  (void)move(LINES-1,0);
  (void)clrtoeol();
  (void)move(oldy,oldx);
  return ((c==KEY_CANCEL || c=='\007') ? -1 : curx);
}
/*}}}*/
/* ok           -- one line yes/no menu *//*{{{*/
int ok(const char *prompt, int curx)
{
  /* variables *//*{{{*/
  MenuChoice menu[3];
  int result;
  /*}}}*/

  /* asserts *//*{{{*/
  assert(curx==0 || curx==1);
  /*}}}*/
  menu[0].str=mystrmalloc(NO); menu[0].c='\0';
  menu[1].str=mystrmalloc(YES); menu[1].c='\0';
  menu[2].str=(char*)0;
  result=menuline(prompt,menu,curx);
  free(menu[0].str);
  free(menu[1].str);
  return result;
}
/*}}}*/




