/* #includes */ /*{{{C}}}*//*{{{*/
#ifndef NO_POSIX_SOURCE
#undef  _POSIX_SOURCE
#define _POSIX_SOURCE   1
#undef  _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 2
#endif

#ifdef DMALLOC
#include "dmalloc.h"
#endif

#include <assert.h>
#include <ctype.h>
#include <curses.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef OLD_REALLOC
#define realloc(s,l) myrealloc(s,l)
#endif
extern char *optarg;
extern int optind,opterr,optopt;
int getopt(int argc, char * const *argv, const char *optstring);
#include <string.h>
#ifdef NEED_BCOPY
#define memmove(dst,src,len) bcopy(src,dst,len)
#endif
#include <time.h>
#include <unistd.h>

#include "misc.h"
#include "libfe.h"
/*}}}*/
/* #defines */ /*{{{*/
#define KILL_LEN 128
/*}}}*/

/* types */ /*{{{*/
struct Emdisplay
{
  struct Display *d;
  enum { YANKMARK=1, APPENDKILL=2, AUTOINDENT=4 } flags;
};
/*}}}*/
/* variables */ /*{{{*/
static char *filehist[HISTORY_LEN];
static char *gotohist[HISTORY_LEN];
static char *searchhist[HISTORY_LEN];
static char *cmdhist[HISTORY_LEN];
static struct Buffer *killbf[KILL_LEN];
static int kill_cur, kill_len;
static struct Emdisplay **e=(struct Emdisplay**)0;
static unsigned int esz=0;
static unsigned int ce=0;
static int counter=0, increment=1;
/*}}}*/

/* em_name       -- name a buffer */ /*{{{*/
static int em_name(struct Emdisplay *e, char *history[HISTORY_LEN])
{
  char path[_POSIX_PATH_MAX];
  size_t x,offx;
    
  x=offx=0;
  path[0]='\0';
  if (editline(path,sizeof(path),FILENAME,&x,&offx,history)==0) return -1;
  if (bf_name(e->d->buffer,path)==0) return 0;
  return 1;
}
/*}}}*/
/* em_save       -- save a file */ /*{{{*/
static int em_save(struct Emdisplay *e)
{
  unsigned int chars;

  if (e->d->buffer->name==(char*)0)
  {
    switch (em_name(e,filehist))
    {
      case -1: return 0; /* abort by user */
      case 0: showmsg(NAMEFAILED); return 0;
      case 1: break; /* ok */
      default: assert(0);
    }
  }
  if (bf_save(e->d->buffer,e->d->buffer->name,"{{{","}}}",&chars))
  {
    showmsg(WROTE,chars);
    return 1;
  }
  else
  {
    showmsg(NOTWRITTEN);
    return 0;
  }
}
/*}}}*/
/* em_saveas     -- save as another file */ /*{{{*/
static void em_saveas(struct Emdisplay *e)
{
  char file[_POSIX_PATH_MAX];
  size_t x,offx;
  unsigned int chars;
  
  x=offx=0;
  file[0]='\0';
  if (editline(file,sizeof(file),SAVEAS,&x,&offx,filehist)==1 && file[0])
  {
    if (ok(NOFOLDS,0)==1)
    {
      if (bf_write(e->d->buffer,file,&chars)==1) showmsg(WROTE,chars);
      else showmsg(NOTWRITTEN);
    }
    else
    {
      if (bf_save(e->d->buffer,file,"{{{","}}}",&chars)==1) showmsg(WROTE,chars);
      else showmsg(NOTWRITTEN);
    }
  }
}
/*}}}*/
/* em_load       -- load a file */ /*{{{*/
static void em_load(void)
{
  e[ce]->flags&=~(YANKMARK|APPENDKILL);
  if (e[ce]->d->buffer->refcnt>1 || !(e[ce]->d->buffer->mode&CHANGED) || ok(LOADOK,0)==1)
  {
    struct Buffer *b;

    if ((b=bf_alloc())==(struct Buffer*)0) showmsg(NOTLOADED);
    else
    {
      ds_realloc(e[ce]->d,b);
      if (em_name(e[ce],filehist)==1)
      {
        int i;

        for (i=0; i<MAXDISPLAYS; ++i)
        {
          if (buffers[i] && buffers[i]->name && strcmp(buffers[i]->name,e[ce]->d->buffer->name)==0 && e[ce]->d->buffer!=buffers[i])
          {
            ds_free(e[ce]->d);
            ds_realloc(e[ce]->d,buffers[i]);
            ds_current(e[ce]->d);
            showmsg(ATTACHED);
            return;
          }
        }
        if (bf_load(e[ce]->d->buffer,(const char*)0,e[ce]->d->buffer->name,-1,"{{{","}}}"))
        {
          e[ce]->d->buffer->mode&=~CHANGED;
          ds_begin();
          return;
        }
        else showmsg(LOADFAILED,strerror(errno));
      }
    }
  }
  else showmsg(NOTLOADED);
}
/*}}}*/
/* em_insert     -- insert a file */ /*{{{*/
static void em_insert(struct Emdisplay *e)
{
  struct Buffer *b;
  char file[_POSIX_PATH_MAX];
  size_t x,offx;
  
  e->flags&=~(YANKMARK|APPENDKILL);
  x=offx=0;
  file[0]='\0';
  if (editline(file,sizeof(file),INSERT,&x,&offx,filehist)==1 && file[0])
  {
    if ((b=bf_alloc())==(struct Buffer*)0)
    {
      showmsg(NOTLOADED);
      return;
    }
    if (bf_load(b,(const char*)0,file,-1,"{{{","}}}")==0) showmsg(LOADFAILED,strerror(errno));
    ds_insertbuf(b,1);
    bf_free(b);
  }
}
/*}}}*/
/* em_gotoline   -- goto numbered line */ /*{{{*/
static void em_gotoline(struct Emdisplay *e, int argument)
{
  e->flags&=~(YANKMARK|APPENDKILL);
  if (argument>0)
  {
    if (ds_gotoline(argument-1)==0) showmsg(NOSUCHLINE);
  }
  else
  {
    char buf[20],*bufp;
    size_t x,offx;
            
    buf[0]='\0';
    x=offx=(size_t)0;
    if (editline(buf,sizeof(buf),GOTOLINE,&x,&offx,gotohist)==1)
    {
      if ((argument=strtod(buf,&bufp))>0 && bufp>buf && *bufp=='\0')
      {
        if (ds_gotoline(argument-1)==0) showmsg(NOSUCHLINE);
      }
      else showmsg(NONUMBER);
    }
  }
}
/*}}}*/
/* em_search     -- search */ /*{{{*/
static void em_search(struct Emdisplay *e, int forward)
{
  char buf[128];
  size_t x,offx;
            
  e->flags&=~(YANKMARK|APPENDKILL);
  buf[0]='\0';
  x=offx=(size_t)0;
  if (editline(buf,sizeof(buf),forward ? STRFSEARCH : STRBSEARCH,&x,&offx,searchhist)==1)
  {
    if (e->d->mode&MAGIC)
    {
      regex_t needle;
      int err;

      if ((err=regcomp(&needle,buf,REG_NEWLINE)))
      {
        size_t sz;
        char errstr[256];

        sz=regerror(err,&needle,errstr+1,sizeof(errstr)-2);
        errstr[0]='[';
        errstr[sz]=']';
        errstr[sz+1]='\0';
        showmsg("%s",errstr);
      }
      else if ((forward ? ds_regfsearch : ds_regbsearch)(&needle)==0) showmsg(NOTFOUND);
      regfree(&needle);
    }
    else if ((forward ? ds_strfsearch : ds_strbsearch)(buf,strlen(buf))==0) showmsg(NOTFOUND);
  }
}
/*}}}*/
/* em_page       -- move a page but stay into fold */ /*{{{*/
static void em_page(int argument, int forward)
{
  int dy;
  int hitstart;

  e[ce]->flags&=~(YANKMARK|APPENDKILL);
  if (argument==0) argument=1;
  dy=e[ce]->d->y;
  if (argument<0)
  {
    forward=!forward;
    argument=-argument;
  }
  while (argument>0)
  {
    hitstart=(forward ? ds_nextpage() : ds_prevpage());
    if (hitstart==1) --argument;
    else if (hitstart==0)
    {
      showmsg(forward ? ATBOTTOM : ATTOP);
      break;
    }
    else if (hitstart==2) break;
  }
  ds_torow(e[ce]->d,dy);
}
/*}}}*/
/* em_kill       -- store buffer in kill ring */ /*{{{*/
static int em_kill(struct Buffer *b, int append)
{
  if (kill_len==0) append=0;
  if (append)
  {
    char c,mark;
    unsigned int count;
    
    bf_end(killbf[kill_cur]);
    bf_begin(b);
    bf_lchar(killbf[kill_cur],&c,&mark);
    if (c!='\n' && bf_isfold(killbf[kill_cur],FOLDEND,LEFT) && bf_isfold(b,FOLDSTART,RIGHT))
    {
      bf_insert(killbf[kill_cur],'\n','\0');
      showmsg(XYZZY);
      count=1;
    }
    else count=0;
    while (bf_rchar(b,&c,&mark))
    {
      if (bf_insert(killbf[kill_cur],c,mark)) ++count;
      else
      {
        while (count>0) { bf_delete(killbf[kill_cur]); --count; }
        return 0;
      }
      bf_forward(b);
    }
    bf_free(b);
  }
  else
  {
    kill_cur=(kill_cur+1)%KILL_LEN;
    if (kill_len<KILL_LEN) ++kill_len;
    bf_free(killbf[kill_cur]);
    killbf[kill_cur]=b;
  }
  return 1;
}
/*}}}*/
/* em_itssearch  -- incremental search */ /*{{{*/
static void em_itssearch(struct Emdisplay *e, int direction, char *history[HISTORY_LEN])
{
  /* variables *//*{{{*/
  size_t bufsize;
  char buf[128];
  char *start;
  chtype c;
  int oops;
  int inhist=-1;
  const char *estr;
  char estrbuf[256];
  regex_t regex;
  /*}}}*/

  buf[0]='\0';
  bufsize=0;
  oops=0;
  estr=SEARCHFAIL;
  start=e->d->buffer->gapstart;
  do
  {
    showmsg(direction ? ISEARCH : RSEARCH,buf,oops ? estr : "");
    ds_refresh();
    switch (c=mc_get())
    {
      /* C-h, KEY_BACKSPACE */ /*{{{*/
      case '\010':
      case KEY_BACKSPACE:
      {
        if (bufsize) buf[--bufsize]='\0';
        break;
      }
      /*}}}*/
      /* C-s */ /*{{{*/
      case '\023':
      {
        direction=1;
        start=e->d->buffer->gapstart;
        break;
      }
      /*}}}*/
      /* C-r */ /*{{{*/
      case '\022':
      {
        direction=0;
        start=e->d->buffer->gapstart;
        break;  
      }
      /*}}}*/
      /* C-p */ /*{{{*/
      case KEY_UP:
      case '\020':
      {
        if (inhist<(HISTORY_LEN-1) && history[inhist+1]!=(char*)0)
        {
          strcpy(buf,history[++inhist]);
          bufsize=strlen(buf);
        }
        break;
      }
      /*}}}*/
      /* C-n -- next line in history */ /*{{{*/
      case KEY_DOWN:
      case '\016':
      {
        if (inhist>0)
        {
          strcpy(buf,history[--inhist]);
          bufsize=strlen(buf);
        }
        break;
      }
      /*}}}*/
      /* C-q -- quote character */ /*{{{*/
      case '\021': 
      {
        chtype c2;
        
        c=mc_get();
        if (c==(c&0xff))
        {
          if (ds_compose(c,0))
          {
            c2=mc_get();
            if (c2==(c2&0xff) && ds_compose(c,c2)) c=((unsigned char)ds_compose(c,c2));
            else break;
          }
        }
        else break;
      }
      /*}}}*/
      default:     /*     -- add to search string */ /*{{{*/
      {
        if (c==(c&0xff) && c>=(unsigned char)' ' && bufsize<(sizeof(buf)-1))
        {
          buf[bufsize]=(char)c;
          buf[++bufsize]='\0';
        }
      }
      /*}}}*/
    }
    ds_topos(e->d,start);
    if (e->d->mode&MAGIC)
    {
      if ((oops=regcomp(&regex,buf,REG_NEWLINE)))
      {
        size_t sz;

        estr=estrbuf;
        sz=regerror(oops,&regex,estrbuf+1,sizeof(estrbuf)-2);
        estrbuf[0]='[';
        estrbuf[sz]=']';
        estrbuf[sz+1]='\0';
      }
      else
      {
        if ((oops=!(direction ? ds_regfsearch(&regex) : ds_regbsearch(&regex))))
        estr=SEARCHFAIL;
      }
      regfree(&regex);
    }
    else
    {
      if ((oops=!(direction ? ds_strfsearch(buf,bufsize) : ds_strbsearch(buf,bufsize))))
      estr=SEARCHFAIL;
    }
  } while (c!='\r' && c!='\n' && c!=KEY_ENTER && c!='\007' && c!=KEY_CANCEL);
  if (c=='\007' || c==KEY_CANCEL) ds_topos(e->d,start);
  else
  {
    if (history[HISTORY_LEN-1]) free(history[HISTORY_LEN-1]);
    memmove(history+1,history,sizeof(history[0])*(HISTORY_LEN-1));
    history[0]=mystrmalloc(buf);
  }
  showmsg((const char*)0);
}
/*}}}*/
/* em_replace    -- search and replace */ /*{{{*/
static void em_replace(struct Emdisplay *e, int argument, int query)
{
  /* variables *//*{{{*/
  char search[128],replace[128];
  size_t x,offx;
  MenuChoice menu[6];
  int replacements=0,oops;
  regex_t regex;
  /*}}}*/
            
  e->flags&=~(YANKMARK|APPENDKILL);
  search[0]='\0';
  x=offx=(size_t)0;
  if (editline(search,sizeof(search),argument<0 ? STRBSEARCH : STRFSEARCH,&x,&offx,searchhist)==1)
  {
    if ((e->d->mode&MAGIC) && (oops=regcomp(&regex,search,REG_NEWLINE)))
    {
      size_t sz;
      char estrbuf[256];

      sz=regerror(oops,&regex,estrbuf+1,sizeof(estrbuf)-2);
      estrbuf[0]='[';
      estrbuf[sz]=']';
      estrbuf[sz+1]='\0';
      showmsg("%s",estrbuf);
      return;
    }
    /* alloc menu *//*{{{*/
    menu[0].str=mystrmalloc(YES);      menu[0].c='\0';
    menu[1].str=mystrmalloc(NO);       menu[1].c='\0';
    menu[2].str=mystrmalloc(REST);     menu[2].c='!';
    menu[3].str=mystrmalloc(ABORT);    menu[3].c='\0';
    menu[4].str=mystrmalloc(ONLYTHIS); menu[4].c='.';
    menu[5].str=(char*)0;
    /*}}}*/
    replace[0]='\0';
    x=offx=(size_t)0;
    if (editline(replace,sizeof(replace),REPLACE,&x,&offx,searchhist)==1)
    {
      int noarg=0;

      if (argument==0) noarg=1;
      while
      (
        (noarg || argument)
        &&
        (
          argument<0
          ? (e->d->mode&MAGIC?ds_regbsearch(&regex):ds_strbsearch(search,strlen(search)))
          : (e->d->mode&MAGIC?ds_regfsearch(&regex):ds_strfsearch(search,strlen(search)))
          ==1
        )
      )
      {
        /* variables *//*{{{*/
        struct Buffer *b;
        char *r,*s;
        int doit=0;
        /*}}}*/

        if (query) /*{{{*/
        {
          ds_refresh();
          switch (menuline(QUERYREPLACE,menu,0))
          {
            case 0: doit=1; break;
            case 1: doit=0; break;
            case 2: query=0; break;
            case 3: noarg=argument=doit=0; break;
            case 4: doit=1; noarg=argument=0; break;
            case -1: noarg=argument=doit=0; break;
            default: assert(0);
          }
        }
        /*}}}*/
        if (!query || doit) /* replace string *//*{{{*/
        {
          for (r=replace; *r; ++r) ds_insertch(*r);
          ds_mark();
          for (s=search; *s; ++s) bf_forward(e->d->buffer);
          b=ds_covebuf(e->d,1,1);
          if (b==(struct Buffer*)0)
          {
            for (s=search; *s; ++s) bf_backward(e->d->buffer);
            for (r=replace; *r; ++r) ds_deletech(1);
            break;
          }
          else bf_free(b);
          ++replacements;
        }
        /*}}}*/
        if (!noarg)
        {
          if (argument<0) ++argument;
          else if (argument>0) --argument;
        }
      }
      showmsg(REPLACEMENTS,replacements);
    }
    /* free menu *//*{{{*/
    free(menu[0].str);
    free(menu[1].str);
    free(menu[2].str);
    free(menu[3].str);
    free(menu[4].str);
    free(menu[5].str);
    /*}}}*/
  }
}
/*}}}*/
/* em_filter     -- filter region */ /*{{{*/
static void em_filter(struct Emdisplay *e)
{
  /* variables */ /*{{{*/
  struct Buffer *old,*new=(struct Buffer*)0;
  char cmd[_POSIX_PATH_MAX];
  size_t x,offx;
  char *tmp;
  /*}}}*/

  e->flags&=~(YANKMARK|APPENDKILL);
  x=offx=0;
  cmd[0]='\0';
  if (editline(cmd,sizeof(cmd),COMMAND,&x,&offx,cmdhist)==0) return;
  if (cmd[0])
  {
    if ((old=ds_covebuf(e->d,1,1))!=(struct Buffer*)0)
    {
      int ok=1;

      tmp=tmpnam((char*)0);
      if (bf_save(old,tmp,"{{{","}}}",(unsigned int*)0))
      {
        if ((new=bf_alloc())!=(struct Buffer*)0)
        {
          if (bf_load(new,cmd,tmp,-1,"{{{","}}}"))
          {
            if (ds_insertbuf(new,1)==0)
            {
              showmsg(NOTFILTERED2);
              ok=0;
            }
          }
          else
          {
            showmsg(NOTFILTERED3);
            ok=0;
          }
        }
        else
        {
          showmsg(NOTFILTERED3);
          ok=0;
        }
      }
      else
      {
        showmsg(NOTFILTERED4);
        ok=0;
      }
      unlink(tmp);
      if (ok)
      {
        em_kill(old,0);
        em_kill(new,0);
        e->flags|=YANKMARK;
      }
      else
      {
        ds_insertbuf(old,1);
        bf_free(old);
        if (new) bf_free(new);
      }
    }
    else showmsg(NOTFILTERED1);
  }
}
/*}}}*/
/* em_new        -- allocate new emdisplay by splitting current emdisplay */ /*{{{*/
static int em_new(void)
{
  int i;

  if (esz>0 && e[ce]->d->maxy<4) return 0;
  ++esz;
  e=realloc(e,esz*sizeof(struct Emdisplay *));
  for (i=(esz-1); i>(ce+1); --i) e[i]=e[i-1];
  if (esz==1) ce=0; else ++ce;
  e[ce]=malloc(sizeof(struct Emdisplay));
  if (esz==1) e[ce]->d=ds_alloc(bf_alloc(),0,0,COLS,LINES-1);
  else
  {
    int old_x,old_y,old_maxx,old_maxy;

    old_x=e[ce-1]->d->orix;
    old_y=e[ce-1]->d->oriy;
    old_maxx=e[ce-1]->d->maxx;
    old_maxy=e[ce-1]->d->maxy;
    ds_reshape(old_x,old_y,old_maxx,old_maxy/2);
    e[ce]->d=ds_alloc(e[ce-1]->d->buffer,old_x,old_y+old_maxy/2,old_maxx,old_maxy-old_maxy/2);
    e[ce]->flags=e[ce-1]->flags;
    e[ce]->d->mode=e[ce-1]->d->mode;
    e[ce]->d->tabsize=e[ce-1]->d->tabsize;
  }
  ds_current(e[ce]->d);
  return 1;
}
/*}}}*/
/* em_pipe       -- process region and put result in new emdisplay */ /*{{{*/
static void em_pipe(void)
{
  /* variables */ /*{{{*/
  struct Buffer *old,*new=(struct Buffer*)0;
  char cmd[_POSIX_PATH_MAX];
  size_t x,offx;
  char *tmp;
  /*}}}*/

  assert(esz>=1);
  e[ce]->flags&=~(YANKMARK|APPENDKILL);
  x=offx=0;
  cmd[0]='\0';
  if (editline(cmd,sizeof(cmd),COMMAND,&x,&offx,cmdhist)==0) return;
  if (cmd[0])
  {
    if ((old=ds_covebuf(e[ce]->d,0,1))!=(struct Buffer*)0)
    {
      int ok=1;

      tmp=tmpnam((char*)0);
      if (bf_save(old,tmp,"{{{","}}}",(unsigned int*)0))
      {
        if ((new=bf_alloc())!=(struct Buffer*)0)
        {
          if (bf_load(new,cmd,tmp,-1,"{{{","}}}"))
          {
            if (e[ce]->d->maxy>=4)
            {
              int i;
              int old_x,old_y,old_maxx,old_maxy;

              ++esz;
              e=realloc(e,esz*sizeof(struct Emdisplay *));
              for (i=(esz-1); i>(ce+1); --i) e[i]=e[i-1];
              e[++ce]=malloc(sizeof(struct Emdisplay));
              old_x=e[ce-1]->d->orix;
              old_y=e[ce-1]->d->oriy;
              old_maxx=e[ce-1]->d->maxx;
              old_maxy=e[ce-1]->d->maxy;
              ds_reshape(old_x,old_y,old_maxx,old_maxy/2);
              e[ce]->d=ds_alloc(new,old_x,old_y+old_maxy/2,old_maxx,old_maxy-old_maxy/2);
              e[ce]->flags=e[ce-1]->flags;
              e[ce]->d->mode=e[ce-1]->d->mode;
              e[ce]->d->tabsize=e[ce-1]->d->tabsize;
              ds_current(e[ce]->d);
              ds_begin();
            }
            else
            {
              ok=0;
              showmsg(TOOSMALL);
            }
          }
          else
          {
            showmsg(NOTFILTERED3);
            ok=0;
          }
        }
        else
        {
          showmsg(NOTFILTERED3);
          ok=0;
        }
      }
      else
      {
        showmsg(NOTFILTERED4);
        ok=0;
      }
      unlink(tmp);
      if (!ok) bf_free(new);
      bf_free(old);
    }
    else showmsg(NOTFILTERED1);
  }
}
/*}}}*/
/* em_free       -- free emdisplay */ /*{{{*/
static void em_free(void)
{
  int i;

  if (ce==0)
  {
    if (esz>1)
    {
      ds_current(e[1]->d);
      ds_reshape(e[0]->d->orix,e[0]->d->oriy,e[0]->d->maxx,e[0]->d->maxy+e[1]->d->maxy);
    }
  }
  else
  {
    ds_current(e[ce-1]->d);
    ds_reshape(e[ce-1]->d->orix,e[ce-1]->d->oriy,e[ce-1]->d->maxx,e[ce-1]->d->maxy+e[ce]->d->maxy);
  }
  ds_free(e[ce]->d);
  free(e[ce]);
  for (i=ce; i<(esz-1); ++i) e[i]=e[i+1];
  --esz;
  if (esz) e=realloc(e,esz*sizeof(struct Emdisplay*));
  else free(e);
  if (ce>0) --ce;
}
/*}}}*/
/* em_setlang    -- set language */ /*{{{*/
static void em_setlang(struct Emdisplay *e)
{
  MenuChoice menu[LANGUAGES];
  int i;

  for (i=0; i<LANGUAGES-1; ++i)
  {
    menu[i].str=malloc(strlen(language[i].name)+2);
    strcpy(menu[i].str+1,language[i].name);
    menu[i].str[0]=tolower(language[i].name[0]);
    menu[i].c='\0';
  }
  menu[LANGUAGES-1].str=(char*)0;
  i=menuline(LANGUAGE,menu,e->d->buffer->lang-language);
  if (i>=0) bf_lang(e->d->buffer,&(language[i]));
  for (i=0; i<LANGUAGES-1; ++i) free(menu[i].str);
}
/*}}}*/
/* em_reshape    -- reshape display */ /*{{{*/
static void em_reshape(void)
{
  MenuChoice menu[4];
  int i;

  menu[0].str=mystrmalloc(SHRINK); menu[0].c='-';
  menu[1].str=mystrmalloc(GROW);   menu[1].c='+';
  menu[2].str=mystrmalloc(QUIT);   menu[2].c='\0';
  menu[3].str=(char*)0;
  i=0;
  do
  {
    ds_refresh();
    switch (i=menuline(RESHAPE,menu,i))
    {
      case 0: /* shrink */ /*{{{*/
      {
        if ((ce+1)<esz && e[ce]->d->maxy>2)
        {
          ds_reshape(e[ce]->d->orix,e[ce]->d->oriy,e[ce]->d->maxx,e[ce]->d->maxy-1);
          ds_current(e[ce+1]->d);
          ds_reshape(e[ce+1]->d->orix,e[ce+1]->d->oriy-1,e[ce+1]->d->maxx,e[ce+1]->d->maxy+1);
          ds_current(e[ce]->d);
        }
        else if ((ce+1)==esz && e[ce]->d->maxy>2)
        {
          ds_reshape(e[ce]->d->orix,e[ce]->d->oriy+1,e[ce]->d->maxx,e[ce]->d->maxy-1);
          ds_current(e[ce-1]->d);
          ds_reshape(e[ce-1]->d->orix,e[ce-1]->d->oriy,e[ce-1]->d->maxx,e[ce-1]->d->maxy+1);
          ds_current(e[ce]->d);
        }
        break;
      }
      /*}}}*/
      case 1: /* grow */ /*{{{*/
      {
        if ((ce+1)<esz && e[ce+1]->d->maxy>2)
        {
          ds_reshape(e[ce]->d->orix,e[ce]->d->oriy,e[ce]->d->maxx,e[ce]->d->maxy+1);
          ds_current(e[ce+1]->d);
          ds_reshape(e[ce+1]->d->orix,e[ce+1]->d->oriy+1,e[ce+1]->d->maxx,e[ce+1]->d->maxy-1);
          ds_current(e[ce]->d);
        }
        else
        if ((ce+1)==esz && e[ce-1]->d->maxy>2)
        {
          ds_current(e[ce-1]->d);
          ds_reshape(e[ce-1]->d->orix,e[ce-1]->d->oriy,e[ce-1]->d->maxx,e[ce-1]->d->maxy-1);
          ds_current(e[ce]->d);
          ds_reshape(e[ce]->d->orix,e[ce]->d->oriy-1,e[ce]->d->maxx,e[ce]->d->maxy+1);
        }
        break;
      }
      /*}}}*/
      case 2: break;
      case -1: break;
      default: assert(0);
    }
  } while (i!=2 && i!=-1);
  free(menu[0].str);
  free(menu[1].str);
  free(menu[2].str);
}
/*}}}*/

int main(int argc, char *argv[]) /*{{{*/
{
  /* variables *//*{{{*/
  chtype ch;
  int o;
  int argument,argsign,clear_argument,line;
  char *options;
  struct Macro curmac;
  struct Buffer *b;
  /*}}}*/

  /* init buffer and parse options *//*{{{*/
  line=0;
#ifdef LC_MESSAGES
  setlocale(LC_MESSAGES,"");
#endif
  setlocale(LC_CTYPE,"");
  while ((o=getopt(argc,argv,"l:h?"))!=EOF) switch (o)
  {
    case 'l': line=strtol(optarg,(char**)0,10); break;
    default: fputs(USAGE,stderr); exit(1);
  }
  ds_init();
  em_new();
  mc_null(&curmac);
  if ((options=getenv("FE"))) while (*options) 
  {
    switch (*options)
    {
      case 'b': ds_addmode(SHOWBEGEND); break;
      case 'i': e[0]->flags|=AUTOINDENT; break;
      case 'o': ds_addmode(OVERWRITE); break;
      case 'p': ds_addmode(POSITION); break;
      case 'l': ds_addmode(LINENO); break;
      case 'R': typeahead(-1); break;
      case 't': ds_addmode(TIME); break;
      case 's': ds_addmode(SIDESCROLL); break;
      case 'm': ds_addmode(MAGIC); break;
      case 'B': ds_addmode(BOLDMARKS); break;
    }
    ++options;
  }
  /* init history buffers *//*{{{*/
  for (o=0; o<HISTORY_LEN; ++o)
  {
    filehist[o]=(optind+o<argc ? mystrmalloc(argv[optind+o]) : (char*)0);
    gotohist[o]=(char*)0;
    searchhist[o]=(char*)0;
    cmdhist[o]=(char*)0;
  }
  /*}}}*/
  if (optind<argc)
  {
    bf_name(e[0]->d->buffer,argv[optind]);
    if (bf_load(e[0]->d->buffer,(const char*)0,argv[optind],-1,"{{{","}}}")==0) showmsg(LOADFAILED,strerror(errno));
    e[0]->d->buffer->mode&=~CHANGED;
    bf_begin(e[0]->d->buffer);
    if (line) ds_gotoline(line-1);
    ++optind;
    if (optind<argc && em_new())
    {
      ds_realloc(e[1]->d,bf_alloc());
      bf_name(e[1]->d->buffer,argv[optind]);
      if (bf_load(e[1]->d->buffer,(const char*)0,argv[optind],-1,"{{{","}}}")==0) showmsg(LOADFAILED,strerror(errno));
      e[1]->d->buffer->mode&=~CHANGED;
      bf_begin(e[1]->d->buffer);
    }
  }
  else showmsg(MENUKEY);
  /*}}}*/
  /* init kill ring buffers *//*{{{*/
  for (o=0; o<KILL_LEN; ++o)
  {
    if ((killbf[o]=bf_alloc())==(struct Buffer*)0)
    {
      showmsg(KILLINIT);
      echo();
      noraw();
      endwin();
      write(1,"\n",1);
    }
    kill_len=0;
    kill_cur=-1;
  }
  /*}}}*/
  argument=0;
  argsign=0;
  clear_argument=0;
  do
  {
    /* do argument display and management *//*{{{*/
    if (clear_argument)
    {
      argument=0;
      argsign=0;
    }
    else if (argument) showmsg("M-%d",argument);
    else if (argsign) showmsg("M--");
    clear_argument=1;
    /*}}}*/
    ds_refresh();
    ch=mc_get();
    showmsg((const char*)0);
    /* process ch *//*{{{*/
    switch (ch)
    {
      case '\000':     /*{{{*/ /* C-@       -- set mark                           */
      {
        ds_mark();
        showmsg(MARKSET);
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        break;
      }
      /*}}}*/
      case '\001':     /*{{{*/ /* C-a       -- beginning of line                  */
      case KEY_BEG:
      {
        ds_beginline();
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        break;
      }
      /*}}}*/
      case '\002':     /*{{{*/ /* C-b       -- backward character                 */
      case KEY_LEFT:
      {
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_backward()==0)
          {
            showmsg(ATBEGIN);
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_forward()==0)
          {
            showmsg(ATEND);
            break;
          } while (++argument);
        }
        else if (ds_backward()==0) showmsg(ATBEGIN);
        break;
      }
      /*}}}*/
      case '\003':     /*{{{*/ /* C-c       -- close fold                         */
      {
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_closefold()==0)
          {
            showmsg(NOTONFOLD);
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_openfold()==0)
          {
            showmsg(NOTONFOLD);
            break;
          } while (++argument);
        }
        else if (ds_closefold()==0) showmsg(NOTONFOLD);
        break;
      }
      /*}}}*/
      case '\004':     /*{{{*/ /* C-d       -- delete character under cursor      */
      case '\177':
      case KEY_DC:
      {
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_deletech(0)==0)
          {
            showmsg(UNDELETABLE);
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_deletech(1)==0)
          {
            showmsg(UNDELETABLE);
            break;
          } while (++argument);
        }
        else if (ds_deletech(0)==0) showmsg(UNDELETABLE);
        break;
      }
      /*}}}*/
      case '\005':     /*{{{*/ /* C-e       -- end of line                        */
      {
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        ds_endline();
        break;
      }
      /*}}}*/
      case '\006':     /*{{{*/ /* C-f       -- forward character                  */
      case KEY_RIGHT:
      {
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_forward()==0)
          {
            showmsg(ATEND);
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_backward()==0)
          {
            showmsg(ATBEGIN);
            break;
          } while (++argument);
        }
        else if (ds_forward()==0) showmsg(ATEND);
        break;
      }
      /*}}}*/
      case '\007':     /*{{{*/ /* C-g       -- abort character                    */
      {
        showmsg(ABORTED);
        argument=0;
        break;
      }
      /*}}}*/
      case '\010':     /*{{{*/ /* C-h       -- delete previous character          */
      case KEY_BACKSPACE:
      {
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_deletech(1)==0)
          {
            showmsg(UNDELETABLE);
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_deletech(0)==0)
          {
            showmsg(UNDELETABLE);
            break;
          } while (++argument);
        }
        else if (ds_deletech(1)==0) showmsg(UNDELETABLE);
        break;
      }
      /*}}}*/
      case '\013':     /*{{{*/ /* C-k       -- kill to end of line                */
      {
        e[ce]->flags&=~YANKMARK;
        if ((b=ds_moveline()))
        {
          if (em_kill(b,e[ce]->flags&APPENDKILL)==0)
          {
            ds_insertbuf(b,1);
            showmsg(KENOUGH);
          }
          else e[ce]->flags|=APPENDKILL;
        }
        break;
      }
      /*}}}*/
      case '\014':     /*{{{*/ /* C-l       -- center display                     */
      {
        ds_torow(e[ce]->d,(e[ce]->d->maxy-1)/2);
        break;
      }
      /*}}}*/
      case '\016':     /*{{{*/ /* C-n       -- next line                          */
      case KEY_DOWN:
      {
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_nextline()==0)
          {
            showmsg(ATBOTTOM);
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_prevline()==0)
          {
            showmsg(ATTOP);
            break;
          } while (++argument);
        }
        else if (ds_nextline()==0) showmsg(ATBOTTOM);
        break;
      }
      /*}}}*/
      case '\017':     /*{{{*/ /* C-o       -- open fold                          */
      {
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_openfold()==0)
          {
            showmsg(NOTONFOLD);
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_closefold()==0)
          {
            showmsg(NOTINFOLD);
            break;
          } while (++argument);
        }
        else if (ds_openfold()==0) showmsg(NOTONFOLD);
        break;
      }
      /*}}}*/
      case '\020':     /*{{{*/ /* C-p       -- previous line                      */
      case KEY_UP:
      {
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_prevline()==0)
          {
            showmsg(ATTOP);
            break;
          } while (--argument);
        }
        else if (argument<0)
        {
          do if (ds_nextline()==0)
          {
            showmsg(ATBOTTOM);
            break;
          } while (++argument);
        }
        else if (ds_prevline()==0) showmsg(ATTOP);
        break;
      }
      /*}}}*/
      case '\021':     /*{{{*/ /* C-q       -- quote character                    */
      {
        chtype ch2;

        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        showmsg(QUOTE);
        ch=mc_get();
        if (ch!=(ch&0xff))
        {
          showmsg(ONLYCHAR);
          break;
        }
        if (ds_compose(ch,0))
        {
          ch2=mc_get();
          if (ch2!=(ch2&0xff))
          {
            showmsg(ONLYCHAR);
            break;
          }
          if ((ch=ds_compose(ch,ch2))==0)
          {
            showmsg(UNCOMPOSABLE);
            break;
          }
        }
        if (argument>0)
        {
          do if (ds_insertch((char)ch)==0) break; while (--argument);
        }
        else if (ds_insertch((char)ch)) showmsg((const char*)0);
        break;
      }
      /*}}}*/
      case '\022':     /*{{{*/ /* C-r       -- reverse incremental search         */
      {
        em_itssearch(e[ce],0,searchhist);
        break;
      }
      /*}}}*/
      case '\023':     /*{{{*/ /* C-s       -- incremental search                 */
      {
        em_itssearch(e[ce],1,searchhist);
        break;
      }
      /*}}}*/
      case '\024':     /*{{{*/ /* C-t       -- transpose character                */
      {
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        ds_transpose();
        break;
      }
      /*}}}*/
      case '\025':     /*{{{*/ /* C-u       -- unfold                             */
      {
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        if (argument>0)
        {
          do if (ds_unfold()==0)
          {
            showmsg(NOTONFOLD);
            break;
          } while (--argument);
        }
        else if (ds_unfold()==0) showmsg(NOTONFOLD);
        break;
      }
      /*}}}*/
      case '\026':     /*{{{*/ /* C-v       -- page down                          */
      case KEY_NPAGE:
      {
        em_page(argument,1);
        break;
      }
      /*}}}*/
      case KEY_PPAGE:  /*{{{*/ /* KEY_PPAGE -- page up                            */
      {
        em_page(argument,0);
        break;
      }
      /*}}}*/
      case '\027':     /*{{{*/ /* C-w       -- wipe out region                    */
      {
        struct Buffer *b;
        
        e[ce]->flags&=~YANKMARK;
        b=ds_covebuf(e[ce]->d,1,1);
        if (b==(struct Buffer*)0) showmsg(NOTWIPED);
        else if (em_kill(b,e[ce]->flags&APPENDKILL)==0)
        {
          ds_insertbuf(b,1);
          showmsg(KENOUGH);
        }
        e[ce]->flags|=APPENDKILL;
        break;
      }
      /*}}}*/
      case '\030':     /*{{{*/ /* C-x       -- C-x prefix                         */
      {
        e[ce]->flags&=~(YANKMARK|APPENDKILL);
        showmsg("C-x ");
        ds_refresh();
        ch=mc_get();
        showmsg((const char*)0);
        switch (ch)
        {
          case '\002': /*{{{*/ /* C-x C-b -- name buffer                          */
          {
            if (em_name(e[ce],filehist)==0) showmsg(NAMEFAILED);
            break;
          }
          /*}}}*/
          case '\003': /*{{{*/ /* C-x C-c -- exit                                 */
          {
            int i;

            for (i=0; i<MAXDISPLAYS; ++i) if (buffers[i] && buffers[i]->mode&CHANGED)
            {
              if (ok(QUITOK,0)==1) goto exit;
              else break;
            }
            if (i==MAXDISPLAYS) goto exit;
            break;
          }
          /*}}}*/
          case '\006': /*{{{*/ /* C-x C-f -- filter region                        */
          em_filter(e[ce]); break;
          /*}}}*/
          case '|':    /*{{{*/ /* C-x C-| -- pipe region                          */
          em_pipe(); break;
          /*}}}*/
          case '\011': /*{{{*/ /* C-x C-i -- insert file                          */
          em_insert(e[ce]); break;
          /*}}}*/
          case '\015': /*{{{*/ /* C-x C-m -- delete mode                          */
          {
            showmsg("C-x C-m ");
            ds_refresh();
            ch=mc_get();
            showmsg((const char*)0);
            switch (ch)
            {
              /* b         -- begin/end marks *//*{{{*/
              case 'b': ds_delmode(SHOWBEGEND); break;
              /*}}}*/
              /* i         -- auto indent *//*{{{*/
              case 'i':
              {
                e[ce]->flags&=~AUTOINDENT;
                break;
              }
              /*}}}*/
              /* m         -- magic *//*{{{*/
              case 'm': ds_delmode(MAGIC); break;
              /*}}}*/
              /* o         -- overwrite *//*{{{*/
              case 'o': ds_delmode(OVERWRITE); break;
              /*}}}*/
              /* p         -- position *//*{{{*/
              case 'p': ds_delmode(POSITION); break;
              /*}}}*/
              /* r         -- read-only */ /*{{{*/
              case 'r':
              {
                switch (ok(CHECKOUTFILE,1))
                {
                  case 1:ds_checkout(); break;
                  case 0: e[ce]->d->buffer->mode&=~READONLY; break;
                }
                break;
              }
              /*}}}*/
              /* R         -- redraw *//*{{{*/
              case 'R': typeahead(0); showmsg(TYPEAHEAD); break;
              /*}}}*/
              /* s         -- scroll sideways *//*{{{*/
              case 's': ds_delmode(SIDESCROLL); break;
              /*}}}*/
              /* l         -- display line numbers *//*{{{*/
              case 'l': ds_delmode(LINENO); break;
              /*}}}*/
              /* t         -- time *//*{{{*/
              case 't': ds_delmode(TIME); break;
              /*}}}*/
              /* B         -- bold marks *//*{{{*/
              case 'B': ds_delmode(BOLDMARKS); break;
              /*}}}*/
              /* default   -- key not bound *//*{{{*/
              default: showmsg(KEYNOTBOUND,ch,ch);
              /*}}}*/
            }
            break;
          }
          /*}}}*/
          case '\022': /*{{{*/ /* C-x C-r -- load buffer from file                */
          {
            em_load();
            break;
          }
          /*}}}*/
          case '\023': /*{{{*/ /* C-x C-s -- save buffer to file                  */
          {
            (void)em_save(e[ce]);
            break;
          }
          /*}}}*/
          case '\027': /*{{{*/ /* C-x C-w -- save buffer to new file              */
          {
            em_saveas(e[ce]);
            break;
          }
          /*}}}*/
          case '\030': /*{{{*/ /* C-x C-x -- swap mark and cursor                 */
          {
            ds_swapmark();
            break;
          }
          /*}}}*/
          case '0':    /*{{{*/ /* C-x 0   -- remove display                       */
          {
            if (esz>1 && (e[ce]->d->buffer->refcnt>1 || !(e[ce]->d->buffer->mode&CHANGED) || ok(LEAVEOK,0)==1))
            {
              em_free();
            }
            break;
          }
          /*}}}*/
          case '2':    /*{{{*/ /* C-x 2   -- split display                        */
          {
            if (em_new()==0) showmsg(TOOSMALL);
            break;
          }
          /*}}}*/
          case '=':    /*{{{*/ /* C-x =   -- describe line                        */
          {
            char c,mark;

            if (bf_rchar(e[ce]->d->buffer,&c,&mark))
            {
              if (c=='\0') switch (mark)
              {
                case NUL: showmsg(DESCRIBELN3,e[ce]->d->buffer->cury+1,e[ce]->d->x,"^@"); break;
                case FOLDSTART: showmsg(DESCRIBELN3,e[ce]->d->buffer->cury+1,e[ce]->d->x,OPENFOLD); break;
                case FOLDEND: showmsg(DESCRIBELN3,e[ce]->d->buffer->cury+1,e[ce]->d->x,CLOSEFOLD); break;
                default: assert(0);
              }
              else
              {
                if (c>=' ' && (c&0x80)=='\0') showmsg(DESCRIBELN2,e[ce]->d->buffer->cury+1,e[ce]->d->x,c);
                else showmsg(DESCRIBELN4,e[ce]->d->buffer->cury+1,e[ce]->d->x,(unsigned char)c,(unsigned char)c);
              }
            }
            else showmsg(DESCRIBELN1,e[ce]->d->buffer->cury+1,e[ce]->d->x);
            break;
          }
          /*}}}*/
          case ':':    /*{{{*/ /* C-x :   -- set variable                         */
          {
            showmsg("C-x : ");
            ds_refresh();
            ch=mc_get();
            showmsg((const char*)0);
            switch (ch)
            {
              /* d         -- set display size */ /*{{{*/
              case 'd':
              {
                em_reshape();
                break;
              }
              /*}}}*/
              /* t         -- set tab size */ /*{{{*/
              case 't':
              {
                char buf[8],*end;
                size_t x,offx;
                int tabsize;

                sprintf(buf,"%d",ds_tabsize(-1));
                x=offx=0;
                while (editline(buf,sizeof(buf),TABWIDTH,&x,&offx,(char**)0)!=0)
                {
                  tabsize=strtol(buf,&end,0);
                  if (end!=buf && *end=='\0' && tabsize>=0 && tabsize<=MAXTAB)
                  {
                    ds_tabsize(tabsize);
                    break;
                  }
                }
                break;
              }
              /*}}}*/
              /* l         -- set language */ /*{{{*/
              case 'l': em_setlang(e[ce]); break;
              /*}}}*/
              /* #         -- set counter and increment */ /*{{{*/
              case '#':
              {
                char buf[8],*end;
                size_t x,offx;
                int c,i;

                sprintf(buf,"%d",counter);
                x=offx=0;
                while (editline(buf,sizeof(buf),COUNTER,&x,&offx,(char**)0)!=0)
                {
                  c=strtol(buf,&end,0);
                  if (end!=buf && *end=='\0')
                  {
                    sprintf(buf,"%d",increment);
                    x=offx=0;
                    while (editline(buf,sizeof(buf),INCREMENT,&x,&offx,(char**)0)!=0)
                    {
                      i=strtol(buf,&end,0);
                      if (end!=buf && *end=='\0')
                      {
                        counter=c;
                        increment=i;
                        break;
                      }
                    }
                    break;
                  }
                }
                break;
              }
              /*}}}*/
              /* default   -- key not bound */ /*{{{*/
              default: showmsg(KEYNOTBOUND,ch,ch);
              /*}}}*/
            }
            break;
          }
          /*}}}*/
          case '(':    /*{{{*/ /* C-x (   -- start recording macro                */
          {
            if (mc_defstart()) showmsg(STARTDEF);
            else showmsg(ALREADYDEF);
            break;
          }
          /*}}}*/
          case ')':    /*{{{*/ /* C-x )   -- end recording macro                  */
          {
            if (mc_defend(&curmac,2)==0) showmsg(NOTDEF);
            break;
          }
          /*}}}*/
          case '!':    /*{{{*/ /* C-x !   -- start shell command                  */
          {
            const char *msg;
            char cmd[255];
            size_t x,offx;
    
            x=offx=0;
            cmd[0]='\0';
            if (editline(cmd,sizeof(cmd),COMMAND,&x,&offx,cmdhist)==0) break;
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            if ((msg=ds_shell((const char*)0,cmd))) showmsg(SUBSHELLBACK,msg);
            break;
          }
          /*}}}*/
          case 'c':    /*{{{*/ /* C-x c   -- start sub shell                      */
          {
            const char *msg;
    
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            if ((msg=ds_shell(SUBSHELLGO,(const char*)0))) showmsg(SUBSHELLBACK,msg);
            break;
          }
          /*}}}*/
          case 'e':    /*{{{*/ /* C-x e   -- execute macro                        */
          {
            if (mc_start(&curmac,argument>0 ? argument : 1)==-1) showmsg(ALREADYDEF);
            break;
          }
          /*}}}*/
          case 'm':    /*{{{*/ /* C-x m   -- add mode                             */
          {
            showmsg("C-x m ");
            ds_refresh();
            ch=mc_get();
            showmsg((const char*)0);
            switch (ch)
            {
              /* b         -- begin/end marks *//*{{{*/
              case 'b': ds_addmode(SHOWBEGEND); break;
              /*}}}*/
              /* i         -- auto indent *//*{{{*/
              case 'i':
              {
                e[ce]->flags|=AUTOINDENT;
                break;
              }
              /*}}}*/
              /* m         -- magic *//*{{{*/
              case 'm': ds_addmode(MAGIC); break;
              /*}}}*/
              /* o         -- overwrite *//*{{{*/
              case 'o': ds_addmode(OVERWRITE); break;
              /*}}}*/
              /* p         -- position *//*{{{*/
              case 'p': ds_addmode(POSITION); break;
              /*}}}*/
              /* r         -- read-only */ /*{{{*/
              case 'r':
              {
                switch (ok(CHECKINFILE,1))
                {
                  case 1: ds_checkin(); break;
                  case 0: e[ce]->d->buffer->mode|=READONLY; break;
                }
                break;
              }
              /*}}}*/
              /* R         -- redraw *//*{{{*/
              case 'R': typeahead(-1); showmsg(NOTYPEAHEAD); break;
              /*}}}*/
              /* s         -- scroll sideways *//*{{{*/
              case 's': ds_addmode(SIDESCROLL); break;
              /*}}}*/
              /* l         -- display line numbers *//*{{{*/
              case 'l': ds_addmode(LINENO); break;
              /*}}}*/
              /* t         -- time *//*{{{*/
              case 't': ds_addmode(TIME); break;
              /*}}}*/
              /* B         -- bold marks *//*{{{*/
              case 'B': ds_addmode(BOLDMARKS); break;
              /*}}}*/
              /* default   -- key not bound *//*{{{*/
              default: showmsg(KEYNOTBOUND,ch,ch);
              /*}}}*/
            }
            break;
          }
          /*}}}*/
          case 'o':    /*{{{*/ /* C-x o   -- other window                         */
          {
            if (++ce==esz) ce=0;
            ds_current(e[ce]->d);
            break;
          }
          /*}}}*/
          case '#':    /*{{{*/ /* C-x #   -- insert counter                       */
          {
            char buf[10],*s;

            sprintf(s=buf,"%d",counter);
            while (*s)
            {
              if (ds_insertch(*s)==0) break;
              ++s;
            }
            counter+=increment;
            break;
          }
          /*}}}*/
          default:     /*{{{*/ /* default -- key not bound                        */
          showmsg(KEYNOTBOUND,ch,ch);
          /*}}}*/
        }
        break;
      }
      /*}}}*/
      case '\031':     /*{{{*/ /* C-y       -- yank current kill ring buffer back */
      {
        if (kill_len)
        {
          if (ds_insertbuf(killbf[kill_cur],1)) e[ce]->flags|=(YANKMARK|APPENDKILL);
        }
        else showmsg(KILLEMPTY);
        break;
      }
      /*}}}*/
      case '\032':     /*{{{*/ /* C-z       -- suspend                            */
      case KEY_SUSPEND:
      ds_suspend(SUSPENDED); break;
      /*}}}*/
      case '\033':     /*{{{*/ /* M-        -- M- prefix                          */
      {
        showmsg("M-");
        ds_refresh();
        ch=mc_get();
        showmsg((const char*)0);
        switch (ch)
        {
          case '\000': /*{{{*/ /* M-C-@     -- fold region                        */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (ds_fold()==0) showmsg(CANTFOLD);
            break;
          }
          /*}}}*/
          case '\004': /*{{{*/ /* M-C-d     -- delete word                        */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument>0)
            {
              do if (ds_forword(DELETE)==0)
              {
                showmsg(ATEND);
                break;
              } while (--argument);
            }
            else if (argument<0)
            {
              do if (ds_backword(DELETE)==0)
              {
                showmsg(ATBEGIN);
                break;
              } while (++argument);
            }
            else if (ds_forword(DELETE)==0) showmsg(ATEND);
            break;
          }
          /*}}}*/
          case '\010': /*{{{*/ /* M-C-h     -- delete word backward               */
          case KEY_BACKSPACE:
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument<0)
            {
              do if (ds_forword(DELETE)==0)
              {
                showmsg(ATEND);
                break;
              } while (++argument);
            }
            else if (argument>0)
            {
              do if (ds_backword(DELETE)==0)
              {
                showmsg(ATBEGIN);
                break;
              } while (--argument);
            }
            else if (ds_backword(DELETE)==0) showmsg(ATBEGIN);
            break;
          }
          /*}}}*/
          case '\014': /*{{{*/ /* M-C-l     -- redraw screen                      */
          {
            ds_redraw();
            break;
          }
          /*}}}*/
          case '\016': /*{{{*/ /* M-C-n     -- go to end of fold                  */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (ds_endfold()==0) showmsg(NOTINFOLD);
            break;
          }
          /*}}}*/
          case '\020': /*{{{*/ /* M-C-p     -- go to beginning of fold            */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (ds_beginfold()==0) showmsg(NOTINFOLD);
            break;
          }
          /*}}}*/
          case '\022': /*{{{*/ /* M-C-r     -- search backward                    */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            em_search(e[ce],0);
            break;
          }
          /*}}}*/
          case '\023': /*{{{*/ /* M-C-s     -- search forward                     */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            em_search(e[ce],1);
            break;
          }
          /*}}}*/
          case '.':    /*{{{*/ /* M-.       -- set mark                           */
          {
            ds_mark();
            showmsg(MARKSET);
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            break;
          }
          /*}}}*/
          case '#':    /*{{{*/ /* M-#       -- go to matching fence               */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            ds_gotofence();
            break;
          }
          /*}}}*/
          case '<':    /*{{{*/ /* M-<       -- beginning of buffer                */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            ds_begin();
            break;
          }
          /*}}}*/
          case '>':    /*{{{*/ /* M->       -- end of buffer                      */
          { 
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            ds_end();
            break;
          }
          /*}}}*/
          case '-':    /*{{{*/ /* M--       -- set argument                       */
          {
            argsign=1;
            clear_argument=0;
            break;
          }
          /*}}}*/
          /* digit */  /*{{{*/ /* M-digit   -- set argument                       */
          case '0':
          case '1':
          case '2':
          case '3':
          case '4':
          case '5':
          case '6':
          case '7':
          case '8':
          case '9': argument=(argsign ? argument*10-(ch-'0') : argument*10+(ch-'0')); clear_argument=0; break;
          /*}}}*/
          case 'b':    /*{{{*/ /* M-b       -- backward word                      */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument>0)
            {
              do if (ds_backword(NOTHING)==0)
              {
                showmsg(ATBEGIN);
                break;
              } while (--argument);
            }
            else if (argument<0)
            {
              do if (ds_forword(NOTHING)==0)
              {
                showmsg(ATBEGIN);
                break;
              } while (++argument);
            }
            else if (ds_backword(NOTHING)==0) showmsg(ATBEGIN);
            break;
          }
          /*}}}*/
          case 'c':    /*{{{*/ /* M-c       -- capitalise word                    */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument>0)
            {
              do if (ds_forword(CAPITALISE)==0)
              {
                showmsg(ATEND);
                break;
              } while (--argument);
            }
            else if (ds_forword(CAPITALISE)==0) showmsg(ATEND);
            break;
          }
          /*}}}*/
          case 'f':    /*{{{*/ /* M-f       -- forward word                       */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument>0)
            {
              do if (ds_forword(NOTHING)==0)
              {
                showmsg(ATEND);
                break;
              } while (--argument);
            }
            else if (argument<0)
            {
              do if (ds_backword(NOTHING)==0)
              {
                showmsg(ATEND);
                break;
              } while (++argument);
            }
            else if (ds_forword(NOTHING)==0) showmsg(ATEND);
            break;
          }
          /*}}}*/
          case 'g':    /*{{{*/ /* M-g       -- go to line                         */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            em_gotoline(e[ce],argument);
            break;
          }
          /*}}}*/
          case 'l':    /*{{{*/ /* M-l       -- lower case word                    */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument>0)
            {
              do if (ds_forword(TOLOWER)==0)
              {
                showmsg(ATEND);
                break;
              } while (--argument);
            }
            else if (ds_forword(TOLOWER)==0) showmsg(ATEND);
            break;
          }
          /*}}}*/
          case 'R':    /*{{{*/ /* M-R       -- replace                            */
          {
            em_replace(e[ce],argument,0);
            break;
          }
          /*}}}*/
          case 'r':    /*{{{*/ /* M-r       -- query replace                      */
          {
            em_replace(e[ce],argument,1);
            break;
          }
          /*}}}*/
          case 'u':    /*{{{*/ /* M-u       -- upper case word                    */
          {
            e[ce]->flags&=~(YANKMARK|APPENDKILL);
            if (argument>0)
            {
              do if (ds_forword(TOUPPER)==0)
              {
                showmsg(ATEND);
                break;
              } while (--argument);
            }
            else if (ds_forword(TOUPPER)==0) showmsg(ATEND);
            break;
          }
          /*}}}*/
          case 'v':    /*{{{*/ /* M-v       -- previous page                      */
          case KEY_PPAGE:
          {
            em_page(argument,0);
            break;
          }
          /*}}}*/
          case 'w':    /*{{{*/ /* M-w       -- copy region to kill ring element   */
          {
            struct Buffer *b;
            
            e[ce]->flags&=~YANKMARK;
            b=ds_covebuf(e[ce]->d,0,1);
            if (b==(struct Buffer*)0) showmsg(NOTCOPIED);
            else if (em_kill(b,e[ce]->flags&APPENDKILL)==0)
            {
              ds_insertbuf(b,1);
              showmsg(KENOUGH);
            }
            e[ce]->flags|=APPENDKILL;
            break;
          }
          /*}}}*/
          case 'x':    /*{{{*/ /* M-x       -- main menu                          */
          {
            MenuChoice menu[6];
            
            menu[0].str=mystrmalloc(LOAD); menu[0].c='\0';
            menu[1].str=mystrmalloc(SAVE); menu[1].c='\0';
            menu[2].str=mystrmalloc(MOVE); menu[2].c='\0';
            menu[3].str=mystrmalloc(MODE); menu[3].c='\0';
            menu[4].str=mystrmalloc(QUIT); menu[4].c='\0';
            menu[5].str=(char*)0;
            switch (menuline(MENU,menu,0))
            {
              /*      -1 -- abort menu *//*{{{*/
              case -1: break;
              /*}}}*/
              /*       0 -- load *//*{{{*/
              case 0:
              {
                em_load();
                break;
              }
              /*}}}*/
              /*       1 -- save *//*{{{*/
              case 1: (void)em_save(e[ce]); break;
              /*}}}*/
              /*       2 -- move menu *//*{{{*/
              case 2:
              {
                MenuChoice movemenu[5];

                movemenu[0].str=mystrmalloc(BEGIN);   movemenu[0].c='\0';
                movemenu[1].str=mystrmalloc(TOEND);   movemenu[1].c='\0';
                movemenu[2].str=mystrmalloc(TOLINE);  movemenu[2].c='\0';
                movemenu[3].str=mystrmalloc(TOSTR);   movemenu[3].c='\0';
                movemenu[4].str=(char*)0;
                switch (menuline(MOVEMENU,movemenu,0))
                {
                  case -1: break;
                  case 0: e[ce]->flags&=~(YANKMARK|APPENDKILL); ds_begin(); break;
                  case 1: e[ce]->flags&=~(YANKMARK|APPENDKILL); ds_end(); break;
                  case 2: em_gotoline(e[ce],0); break;
                  case 3: em_search(e[ce],1); break;
                  default: assert(0);
                }
                free(movemenu[0].str);
                free(movemenu[1].str);
                free(movemenu[2].str);
                free(movemenu[3].str);
                break;
              }
              /*}}}*/
              /*       3 -- mode *//*{{{*/
              case 3:
              {
                MenuChoice modemenu[5];
                int add;
                
                modemenu[0].str=mystrmalloc(DELMODE);     modemenu[0].c='\0';
                modemenu[1].str=mystrmalloc(ADDMODE);     modemenu[1].c='\0';
                modemenu[2].str=(char*)0;
                add=menuline(MMODE,modemenu,0);
                free(modemenu[0].str);
                free(modemenu[1].str);
                if (add==-1) break;
                modemenu[0].str=mystrmalloc(MAUTOINDENT); modemenu[0].c='\0';
                modemenu[1].str=mystrmalloc(MSIDESCROLL); modemenu[1].c='\0';
                modemenu[2].str=mystrmalloc(MTIME);       modemenu[2].c='\0';
                modemenu[3].str=mystrmalloc(MPOSITION);   modemenu[3].c='\0';
                modemenu[4].str=(char*)0;
                switch (menuline(add ? MADDMODE : MDELMODE,modemenu,0))
                {
                  case -1: break;
                  case 0: e[ce]->flags&=~(YANKMARK|APPENDKILL); if (add) e[ce]->flags|=AUTOINDENT; else e[ce]->flags&=~AUTOINDENT; break;
                  case 1: e[ce]->flags&=~(YANKMARK|APPENDKILL); (add ? ds_addmode : ds_delmode)(SIDESCROLL); break;
                  case 2: e[ce]->flags&=~(YANKMARK|APPENDKILL); (add ? ds_addmode : ds_delmode)(TIME); break;
                  case 3: e[ce]->flags&=~(YANKMARK|APPENDKILL); (add ? ds_addmode : ds_delmode)(POSITION); break;
                  default: assert(0);
                }
                free(modemenu[0].str);
                free(modemenu[1].str);
                free(modemenu[2].str);
                free(modemenu[3].str);
                break;
              }
              /*}}}*/
              /*       4 -- quit *//*{{{*/
              case 4: if (!(e[ce]->d->buffer->mode&CHANGED) || ok(QUITOK,0)==1) goto exit; break;
              /*}}}*/
              /* default -- should not happen *//*{{{*/
              default: assert(0);
              /*}}}*/
            }
            free(menu[0].str);
            free(menu[1].str);
            free(menu[2].str);
            free(menu[3].str);
            free(menu[4].str);
            break;
          }
          /*}}}*/
          case 'y':    /*{{{*/ /* M-y       -- move backward in kill ring         */
          {
            e[ce]->flags&=~(APPENDKILL); 
            if (kill_len)
            {
              if (e[ce]->flags&YANKMARK)
              {
                b=ds_covebuf(e[ce]->d,1,1);
                if (argument>=0) do
                {
                  if (kill_cur==0) kill_cur=kill_len-1;
                  else --kill_cur;
                  if (argument==0) argument=1;
                } while (--argument);
                else if (argument<0) do
                {
                  if (kill_cur==(kill_len-1)) kill_cur=0;
                  else ++kill_cur;
                } while (++argument>0);
                if (ds_insertbuf(killbf[kill_cur],1)==0)
                {
                  ds_insertbuf(b,1);
                  showmsg(NOTYANKED);
                }
                bf_free(b);
              }
              else showmsg(NOYANKMARK);
            }
            else showmsg(KILLEMPTY);
            break;
          }
          /*}}}*/
          case 'z':    /*{{{*/ /* M-z       -- save and exit                      */
          {
            int i;

            for (i=0; i<esz; ++i) if (e[i]->d->buffer->mode&CHANGED)
            if (em_save(e[i])) ds_refresh();
            else break;
            if (i==esz) goto exit;
            break;
          }
          /*}}}*/
          default:     /*{{{*/ /* default   -- key not bound                      */
          showmsg(KEYNOTBOUND,ch,ch);
          /*}}}*/
        }
        break;
      }
      /*}}}*/
      case KEY_IC:     /*{{{*/ /* KEY_IC    -- toggle insert mode                 */
      {
        if (e[ce]->d->mode&OVERWRITE) ds_delmode(OVERWRITE);
        else ds_addmode(OVERWRITE);
        break;
      }
      /*}}}*/
      default:         /*{{{*/ /* default   -- insert all text characters         */
      {
        if (ch==KEY_ENTER) ch='\r';
        if ((argument || argsign) && isdigit(ch))
        {
          argument=(argsign ? argument*10-(ch-'0') : argument*10+(ch-'0'));
          clear_argument=0;
        }
        else if (ch==(ch&0xff))
        {
          struct Buffer *b;

          if (!(e[ce]->d->mode&OVERWRITE) && ((e[ce]->flags&AUTOINDENT && ch=='\15') || (!(e[ce]->flags&AUTOINDENT) && ch=='\n')))
          {
            if ((b=ds_indent())==(struct Buffer*)0)
            {
              showmsg(NOMEM);
              break;
            }
          }
          else b=(struct Buffer*)0;
          if (ch=='\015') ch='\n';
          e[ce]->flags&=~(YANKMARK|APPENDKILL); 
          if (argument<=0) argument=1;
          while (argument--)
          {
            if (ds_insertch((char)ch)==0) break;
            else if (b && ds_insertbuf(b,0)==0) break;
          }
          if (b) bf_free(b);
        }
        else showmsg(KEYNOTBOUND,ch,ch);
      }
      /*}}}*/
    }
    /*}}}*/
  } while (1);
  exit:
  /* land nicely *//*{{{*/
  ds_exit(1);
  while (esz) em_free();
  for (o=0; o<HISTORY_LEN; ++o)
  {
    if (filehist[o]) free(filehist[o]);
    if (gotohist[o]) free(gotohist[o]);
    if (searchhist[o]) free(searchhist[o]);
    if (cmdhist[o]) free(cmdhist[o]);
  }
  for (o=0; o<KILL_LEN; ++o) bf_free(killbf[o]);
  /*}}}*/
  return 0;
}
/*}}}*/
