/*
 *  $Id: sbcnst.c,v 1.7 1994/06/07 21:30:17 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   sbcnst.c                                                           *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#define _SBCNSTDEF
#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop
#include <stdio.h>
#include <memory.h>
#include <malloc.h>

#ifndef lint
static char SCCSid[] = "%W% %G%";
#endif

#ifdef MPIR_DEBUG_MEM
#undef MPIR_SBinit
#undef MPIR_SBalloc
#undef MPIR_SBfree
#undef MPIR_SBdestroy
#endif

/*
   This file contains routines for allocating a number of fixed-sized blocks.
   This is often a good way to improve the performance of dynamic-memory
   codes, at the expense of some additional space.  However, unlike a purely
   static allocation (a fixed maximum), this mechanism allows space to grow.

   The basic interface is

  sb = MPIR_SBinit( blocksize, initialnumber, incrementnumber );
  ptr = MPIR_SBalloc( sb );
  ...
  MPIR_SBfree( sb, ptr );
  ...
  MPIR_SBdestroy( sb );
 */


// void MPIR_SBiAllocate(MPIR_SBHeader far *sb, Int bsize, Int nb);

MPIR_SBHeader far *MPIR_SBinit( Int bsize, Int nb, Int nbincr)
{
MPIR_SBHeader far *head;

/* Make sure that the blocksizes are multiples of pointer size */
if (bsize < sizeof(char far *)) 
	bsize = sizeof(char far *);

head           = MPI_NEW(MPIR_SBHeader);  
if (!head) {
   MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_EXHAUSTED, "Not enough space" );
   return 0;
   }
head->nbfree   = 0;
head->nballoc  = 0;
head->sizeb    = bsize;
head->sizeincr = nbincr;
head->avail    = 0;
head->blocks   = 0;
if (nb > 0) {
    MPIR_SBiAllocate( head, bsize, nb );
    if (!head->avail) {
	MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_EXHAUSTED, 
		    "Failed to allocate space" );
	head = 0;
	}
    }

return head;
}

/* 
    MPIR_SBfree - return a fixed-sized block to the allocator
 */    
void MPIR_SBfree( MPIR_SBHeader far *sb, void far *ptr)
{
	((MPIR_SBblock far *)ptr)->next = (char far *)(sb->avail);
	sb->avail              = (MPIR_SBblock far *)ptr;
	sb->nbfree++;
	sb->nballoc--;
}

/*
    Internal routine to allocate space
 */
void MPIR_SBiAllocate( MPIR_SBHeader far *sb, Int bsize, Int nb)
{
char     far *p, far *p2;
Int      i, headeroffset, n;
MPIR_SBiAlloc far *header;

/* printf( "Allocating %d blocks of size %d\n", nb, bsize ); */
/* Double-align block */
headeroffset    = (sizeof(MPIR_SBiAlloc) + sizeof(double) - 1) / sizeof(double);
headeroffset    *= sizeof(double);

sb->avail       = 0;
p               = (char far *) MPI_MALLOC( bsize * nb + headeroffset );
if (!p) {
   MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_EXHAUSTED, "Not enough space" );
   return;
   }
/* zero the data FOR DEBUGGING
   LATER CHANGE THIS TO SOME INVALID POINTER VALUE */
n = bsize * nb + headeroffset;
for (i=0; i<n; i++) 
    p[i] = 0;

header          = (MPIR_SBiAlloc far *)p;
/* Place at header for list of allocated blocks */
header->next    = sb->blocks;
sb->blocks      = header;
header->nbytes  = bsize * nb;
header->nballoc = nb;
header->nbinuse = nb;

/* Link the list together */
p2 = p + headeroffset;
for (i=0; i<nb-1; i++) {
    ((MPIR_SBblock far *)p2)->next = p2 + bsize;
    p2 += bsize;
    }
((MPIR_SBblock far *)p2)->next = (char far *)sb->avail;
sb->avail  = (MPIR_SBblock far *)(p + headeroffset);
sb->nbfree += nb;
}

/* 
    MPIR_SBalloc - Gets a block from the fixed-block allocator.

    Input Parameter:
.   sb - Block context (from MPIR_SBinit)

    Returns:
    Address of new block.  Allocates more blocks if required.
 */
void far *MPIR_SBalloc( MPIR_SBHeader far *sb)
{
	MPIR_SBblock far *p;
	
	if (!sb->avail) 
	{
	    MPIR_SBiAllocate( sb, sb->sizeb, sb->sizeincr );   /* nbincr instead ? */
	    if (!sb->avail) 
	    {
			MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_EXHAUSTED, "Not enough space" );
			return 0;
		}
	}
	p         = sb->avail;
	sb->avail = (MPIR_SBblock far *)(p->next);
	sb->nballoc++;
	sb->nbfree--;
	/* printf( "Allocating a block at address %x\n", (char *)p ); */
	return (void far *)p;
}	

/* 
    MPIR_SBPrealloc - Insure that at least nb blocks are available

    Input Parameters:
.   sb - Block header
.   nb - Number of blocks that should be preallocated

    Notes:
    This routine insures that nb blocks are available, not that an
    additional nb blocks are allocated.  This is appropriate for the common
    case where the preallocation is being used to insure that enough space
    is available for a new object (e.g., a sparse matrix), reusing any
    available blocks.
 */
void MPIR_SBPrealloc( MPIR_SBHeader far *sb, Int nb)
{
	if (sb->nbfree < nb) 
	{
	    MPIR_SBiAllocate( sb, sb->sizeb, nb - sb->nbfree );
	}	
}

/* 
    MPIR_SBdestroy - Destroy a fixed-block allocation context

 */
void MPIR_SBdestroy( MPIR_SBHeader far *sb)
{
	MPIR_SBiAlloc far *p, far *pn;
	
    p = sb->blocks;
    while (p) 
    {
	pn = p->next;
	MPI_FREE( p );
	p = pn;
    }
    MPI_FREE( sb );
}

/* Decrement the use count for the block containing p */
void MPIR_SBrelease( MPIR_SBHeader far *sb, void far *ptr)
{
	char far *p = (char far *)ptr;
	MPIR_SBiAlloc far *b = sb->blocks;
	char far *first, far *last;
	
	/* printf( "Releasing a block at address %x\n", (char *)ptr ); */
	while (b) 
	{
	    first = ((char far *)b) + sizeof(MPIR_SBiAlloc) - 1;
	    last  = first + b->nbytes - 1;
	    if (p >= first && p <= last) 
	    {
			b->nbinuse--;
			break;
		}
	    b = b->next;
    }
}

/* Release any unused chuncks */
void MPIR_SBFlush( MPIR_SBHeader far *sb)
{
	MPIR_SBiAlloc far *b, far *bnext, far *bprev = 0;
	
	b = sb->blocks;
	while (b) 
	{
	    bnext = b->next;
	    if (b->nbinuse == 0) 
	    {
			if (bprev) 
				bprev->next = bnext;
			else       
				sb->blocks  = bnext;
			sb->nballoc -= b->nballoc;
			MPI_FREE( b );
		}
	    else 
		bprev = b;
	    b = bnext;
    }
}

/* Print the allocated blocks */
void MPIR_SBDump( FILE *fp, MPIR_SBHeader far *sb)
{
	MPIR_SBiAlloc far *b = sb->blocks;
	
	while (b) 
	{
	    fprintf( fp, "Block %fp of %ld bytes and %ld chuncks in use\n", 
		     (char far *)b, b->nbytes, b->nbinuse );
	    b = b->next;
    }
}

void MPIR_SBReleaseAvail( MPIR_SBHeader far *sb)
{
MPIR_SBblock far *p, far *pnext;
	
	p         = sb->avail;
	while (p) 
	{
	    pnext = (MPIR_SBblock far *)(p->next);
	    sb->avail = pnext;
	    sb->nbfree--;
	    MPIR_SBrelease( sb, p );
	    p     = pnext;
    }
}

