/*
 *  $Id: graph_create.c,v 1.8 1994/06/14 16:04:52 bridges Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   gph_cre.c                                                          *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop

/*@

MPI_Graph_create - Makes a new communicator to which topology information
                 has been attached

Input Parameters:
. comm_old - input communicator without topology (handle) 
. nnodes - number of nodes in graph (integer) 
. index - array of integers describing node degrees (see below) 
. edges - array of integers describing graph edges (see below) 
. reorder - ranking may be reordered (true ) or not (false ) (logical) 

Output Parameter:
. comm_graph - communicator with graph topology added (handle) 

Algorithm:
We ignore the periods and reorder info currently.

@*/
Int MPI_Graph_create ( MPI_Comm comm_old, Int nnodes, Int far *index, 
		       Int far *edges, Int reorder, MPI_Comm far *comm_graph)
{
  Int range[1][3];
  MPI_Group group_old, group;
  Int i, num_ranks = 1;
  Int errno = MPI_SUCCESS;
  Int flag, size;

  reorder = reorder;
  /* Check validity of arguments */
  if (MPIR_TEST_COMM(comm_old,comm_old) || 
      MPIR_TEST_ARG(comm_graph) || MPIR_TEST_ARG(index) || 
      MPIR_TEST_ARG(edges) || 
	  ((nnodes     <  1)             && ((errno = MPI_ERR_ARG) != 0)) )
    return MPIR_ERROR( comm_old, errno, "Error in MPI_GRAPH_CREATE" );

  /* Check for Intra-communicator */
  MPI_Comm_test_inter ( comm_old, &flag );
  if (flag)
    return MPIR_ERROR(comm_old, MPI_ERR_COMM,
                      "Inter-communicator invalid in MPI_GRAPH_CREATE");
  
  /* Determine number of ranks in topology */
  num_ranks = nnodes;
  if ( num_ranks < 1 ) {
    (*comm_graph)  = MPI_COMM_NULL;
    return MPIR_ERROR( comm_old, MPI_ERR_TOPOLOGY, 
		       "Error in MPI_GRAPH_CREATE" );
  }

  /* Is the old communicator big enough? */
  MPI_Comm_size (comm_old, &size);
  if (num_ranks > size) 
	return MPIR_ERROR(comm_old, MPI_ERR_ARG, 
			         "Topology size too big in MPI_GRAPH_CREATE");

  /* Make new communicator */
  range[0][0] = 0; range[0][1] = num_ranks - 1; range[0][2] = 1;
  MPI_Comm_group ( comm_old, &group_old );
  MPI_Group_range_incl ( group_old, 1, range, &group );
  MPI_Comm_create  ( comm_old, group, comm_graph );

  /* Store topology information in new communicator */
  if ( (*comm_graph) != MPI_COMM_NULL ) {
    (*comm_graph)->topology.type                 = MPI_GRAPH;
    (*comm_graph)->topology.tag.graph.nnodes     = nnodes;
    (*comm_graph)->topology.tag.graph.nedges     = index[nnodes-1];
    (*comm_graph)->topology.tag.graph.index      = 
      (Int far *)MPI_MALLOC( sizeof(Int) * (nnodes + index[nnodes-1]) );
    (*comm_graph)->topology.tag.graph.edges      = 
      (*comm_graph)->topology.tag.graph.index + nnodes;
    for ( i=0; i<nnodes; i++ )
      (*comm_graph)->topology.tag.graph.index[i] = index[i];
    for ( i=0; i<index[nnodes-1]; i++ )
      (*comm_graph)->topology.tag.graph.edges[i] = edges[i];
  }
  return (errno);
}
