/*
 *  $Id: cart_shift.c,v 1.10 1994/06/07 21:23:59 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   crt_shf.c                                                          *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop

/*@

MPI_Cart_shift - Returns the shifted source and destination ranks, given a 
                 shift direction and amount

Input Parameters:
. comm - communicator with cartesian structure (handle) 
. direction - coordinate dimension of shift (integer) 
. disp - displacement ($> 0$: upwards shift, $< 0$: downwards shift) (integer) 

Output Parameters:
. rank_source - rank of source process (integer) 
. rank_dest - rank of destination process (integer) 

@*/
Int MPI_Cart_shift ( MPI_Comm  comm, Int direction, Int displ, 
		     Int far *source, Int far *dest)
{
  Int rank, size;
  Int source_position, dest_position, save_position, periodic;
  Int errno = MPI_SUCCESS;

  /* Check for valid arguments */
  if ( MPIR_TEST_COMM(comm,comm) ||
	   ((comm->topology.type!= MPI_CART)      && ((errno = MPI_ERR_TOPOLOGY)) != 0)||
	   ((direction          <  0)             && ((errno = MPI_ERR_ARG)) != 0)     ||
	   ((direction>=(comm->topology.tag.cart.ndims))&&((errno=MPI_ERR_ARG)) != 0)   ||
       MPIR_TEST_ARG(dest) || MPIR_TEST_ARG(source)) 
    return MPIR_ERROR( comm, errno, "Error in MPI_CART_SHIFT" );

  /* Check the case for a 0 displacement */
  MPI_Comm_rank (comm, &rank);
  if (displ == 0) {
    (*source) = (*dest) = rank;
    return (errno);
  }

  /* Get ready for shifting */
  MPI_Comm_size (comm, &size);
  periodic = comm->topology.tag.cart.periods[direction];
  save_position = source_position = dest_position = 
    comm->topology.tag.cart.position[direction];
  
  /* Shift for the destination */
  dest_position += displ;
  if ( dest_position >= comm->topology.tag.cart.dims[direction] ) {
    if ( periodic )
      dest_position %= comm->topology.tag.cart.dims[direction];
    else
      dest_position = MPI_PROC_NULL;
  }
  else if ( dest_position < 0 ) {
    if ( periodic )
      dest_position += comm->topology.tag.cart.dims[direction];
    else
      dest_position = MPI_PROC_NULL;
  }
  comm->topology.tag.cart.position[direction] = dest_position;
  MPI_Cart_rank ( comm, comm->topology.tag.cart.position, dest );

  /* Shift for the source */
  source_position -= displ;
  if ( source_position >= comm->topology.tag.cart.dims[direction] ) {
    if ( periodic )
      source_position %= comm->topology.tag.cart.dims[direction];
    else
      source_position = MPI_PROC_NULL;
  }
  else if ( source_position < 0 ) {
    if ( periodic )
      source_position += comm->topology.tag.cart.dims[direction];
    else
      source_position = MPI_PROC_NULL;
  }
  comm->topology.tag.cart.position[direction] = source_position;
  MPI_Cart_rank ( comm, comm->topology.tag.cart.position, source );

  /* Restore my position */
  comm->topology.tag.cart.position[direction] = save_position;

  return (errno);
}


