/*
 *  $Id: type_struct.c,v 1.9 1994/06/07 21:22:39 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   type_st.c                                                          *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop

#ifndef lint
static char vcid[] = "$Id: type_struct.c,v 1.9 1994/06/07 21:22:39 gropp Exp $";
#endif /* lint */

#include <malloc.h>

/*@
    MPI_Type_struct - Creates a struct datatype

Input Parameters:
. count - number of blocks (integer) -- also number of 
entries in arrays array_of_types ,
array_of_displacements  and array_of_blocklengths  
. array_of_blocklength - number of elements in each block 
(array of integer) 
. array_of_displacements - byte displacement of each block 
(array of integer) 
. array_of_types - type of elements in each block (array 
of handles to datatype objects) 

Output Parameter:
. newtype - new datatype (handle) 
@*/
Int MPI_Type_struct( Int count, Int far *blocklens, MPI_Aint far *indices, 
			MPI_Datatype far *old_types, MPI_Datatype far *newtype)
{
  MPI_Datatype    dteptr;
  MPI_Aint        ub, lb, high, low;
  MPIR_BOOL       high_init = MPIR_NO, low_init = MPIR_NO;
  Int             i, errno = MPI_SUCCESS;
  Int             ub_marker, lb_marker;
  MPIR_BOOL       ub_found = MPIR_NO, lb_found = MPIR_NO;
  Int pad, size;

  /* Check for bad arguments */
  if ( count < 0 )
	return MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_COUNT,
					  "Negative count in MPI_TYPE_STRUCT" );
    
  /* Check blocklens and old_types arrays and find number of bound */
  /* markers */
  for (i=0; i<count; i++) {
    if ( blocklens[i] < 0)
      return MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_OTHER,
                        "Negative block length in MPI_TYPE_STRUCT");
    if ( old_types[i] == MPI_DATATYPE_NULL )
      return MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_TYPE,
                        "Null type in MPI_TYPE_STRUCT");
  }
    
  /* Create and fill in the datatype */
  dteptr = (*newtype) = (MPI_Datatype) MPIR_SBalloc( MPIR_dtes );
  if (!dteptr) 
      return MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_EXHAUSTED, 
			 "Out of space in MPI_TYPE_HVECTOR" );
  dteptr->dte_type    = MPIR_STRUCT;
  dteptr->committed   = MPIR_FALSE;
  dteptr->basic       = MPIR_FALSE;
  dteptr->permanent   = MPIR_FALSE;
  dteptr->is_contig   = MPIR_FALSE;
  dteptr->ref_count   = 1;
  dteptr->align       = old_types[0]->align;
  dteptr->count       = count;
  dteptr->elements    = 0;
  dteptr->size        = 0;

  /* Create indices and blocklens arrays and fill them */
  dteptr->indices     = ( Int far * ) MPI_MALLOC( count     * sizeof( Int ) );
  dteptr->blocklens   = ( Int far * ) MPI_MALLOC( count     * sizeof( Int ) );
  dteptr->pads        = ( Int far * ) MPI_MALLOC( (count-1) * sizeof( Int ) );
  dteptr->old_types   = ( MPI_Datatype far * ) MPI_MALLOC(count*sizeof(MPI_Datatype));
  if (!dteptr->indices || !dteptr->blocklens || ! dteptr->pads || 
      !dteptr->old_types) 
      return MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_EXHAUSTED, 
			 "Out of space in MPI_TYPE_HINDEXED" );
  high = low = ub = lb = 0;
  for (i = 0; i < count; i++)  {
	dteptr->old_types[i]  = (MPI_Datatype)MPIR_Type_dup (old_types[i]);
	dteptr->indices[i]    = indices[i];
	dteptr->blocklens[i]  = blocklens[i];
    if ( old_types[i]->dte_type == MPIR_UB ) {
      if (ub_found) {
        if (indices[i] > ub_marker)
          ub_marker = indices[i];
      }
      else {
        ub_marker = indices[i];
        ub_found  = MPIR_YES;
      }
    }
    else if ( old_types[i]->dte_type == MPIR_LB ) {
      if (lb_found) {
        if ( indices[i] < lb_marker )
          lb_marker = indices[i];
      }
      else {
        lb_marker = indices[i];
        lb_found  = MPIR_YES;
      }
    }
    else {
      ub = indices[i] + (blocklens[i] * old_types[i]->extent) ;
      lb = indices[i];
      if (!high_init) { high = ub; high_init = MPIR_YES; }
      if (!low_init ) { low  = lb; low_init  = MPIR_YES; }
      if (ub > lb) {
        if ( high < ub ) high = ub;
        if ( low  > lb ) low  = lb;
      }
      else {
        if ( high < lb ) high = lb;
        if ( low  > ub ) low  = ub;
      }
      dteptr->elements += (blocklens[i] * old_types[i]->elements);
    }
  }
  
  for (i=0; i<(count-1); i++) {
    size = old_types[i]->size * blocklens[i];
    if ( blocklens[i] > 1 ) {
      pad = (old_types[i]->align - (old_types[i]->size %
             old_types[i]->align)) % old_types[i]->align;
      size += ((blocklens[i]-1) * pad );
    }
	dteptr->pads[i] = ((old_types[i+1]->align - 
        (size % old_types[i+1]->align)) % old_types[i+1]->align);
	dteptr->size   += (size + dteptr->pads[i]); 
  }
  dteptr->size     += (blocklens[i] * old_types[i]->size);
  if ( blocklens[i] > 1 ) {
    pad = (old_types[i]->align - (old_types[i]->size %
                                  old_types[i]->align)) % old_types[i]->align;
    dteptr->size += ((blocklens[i]-1) * pad );
  }
  
  /* Set the upper/lower bounds and the extent and size */
  dteptr->lb          = lb_found ? lb_marker : (low_init  ? low : 0);
  dteptr->ub          = ub_found ? ub_marker : (high_init ? high: 0);
  dteptr->extent      = dteptr->ub - dteptr->lb;
  
  return (errno);
}
