/*
 *  $Id: type_hvec.c,v 1.7 1994/06/13 18:08:44 doss Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   type_hv.c                                                          *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop

#ifndef lint
static char vcid[] = "$Id: type_hvec.c,v 1.7 1994/06/13 18:08:44 doss Exp $";
#endif /* lint */

/*@
    MPI_Type_hvector - Creates a vector (strided) datatype with offset in bytes

Input Parameters:
. count - number of blocks (nonnegative integer) 
. blocklength - number of elements in each block 
(nonnegative integer) 
. stride - number of bytes between start of each block (integer) 
. oldtype - old datatype (handle) 

Output Parameter:
. newtype - new datatype (handle) 
@*/
Int MPI_Type_hvector( Int count, Int blocklen, MPI_Aint stride, 
			MPI_Datatype old_type, MPI_Datatype far *newtype )
{
  MPI_Datatype  dteptr;
  Int           errno = MPI_SUCCESS;

  /* Check for bad arguments */
  if ( MPIR_TEST_DATATYPE(MPI_COMM_WORLD,old_type) ||
   ( (old_type == (*newtype))        && ((errno = MPI_ERR_TYPE) != 0) )  ||
   ( (count   <= 0)                  && ((errno = MPI_ERR_COUNT) != 0) ) ||
   ( (blocklen <= 0)                 && ((errno = MPI_ERR_ARG) != 0) )   ||
   ( (old_type->dte_type == MPIR_UB) && ((errno = MPI_ERR_TYPE) != 0) )  ||
   ( (old_type->dte_type == MPIR_LB) && ((errno = MPI_ERR_TYPE) != 0) ) )
	return MPIR_ERROR( MPI_COMM_WORLD, errno,
					  "Error in MPI_TYPE_HVECTOR" );
	
  /* Handle the case where blocklen & stride make a contiguous type */
  if ( ((blocklen * old_type->extent) == stride) ||
	   (count                         == 1) )
	return MPI_Type_contiguous ( count * blocklen, old_type, newtype );

  /* Create and fill in the datatype */
  dteptr = (*newtype) = (MPI_Datatype) MPIR_SBalloc( MPIR_dtes );
  if (!dteptr) 
      return MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_EXHAUSTED, 
			 "Out of space in MPI_TYPE_HVECTOR" );
  dteptr->dte_type    = MPIR_HVECTOR;
  dteptr->committed   = MPIR_FALSE;
  dteptr->basic       = MPIR_FALSE;
  dteptr->permanent   = MPIR_FALSE;
  dteptr->is_contig   = MPIR_FALSE;
  dteptr->ref_count   = 1;
  dteptr->align       = old_type->align;
  dteptr->elements    = count * blocklen * old_type->elements;
  dteptr->stride      = stride;
  dteptr->blocklen    = blocklen;
  dteptr->old_type    = (MPI_Datatype)MPIR_Type_dup (old_type);
  dteptr->count       = count;
  dteptr->pad         = ((old_type->align -
                        (old_type->size % old_type->align)) % old_type->align);

  /* Set the upper/lower bounds and the extent and size */
  dteptr->extent      = ((count-1) * stride) + (blocklen * old_type->extent);
  if (dteptr->extent < 0) {
	dteptr->ub     = old_type->lb;
	dteptr->lb     = dteptr->ub + dteptr->extent;
	dteptr->extent = -dteptr->extent;
  }
  else {
	dteptr->lb     = old_type->lb;
	dteptr->lb     = dteptr->lb + dteptr->extent;
  }
  dteptr->size        = (count * blocklen * dteptr->old_type->size) +
	                    (((count * blocklen) - 1) * dteptr->pad);
  
  return (errno);
}
