/*
 *  $Id: pack.c,v 1.3 1994/06/07 21:22:17 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   pack.c                                                             *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop
#include <memory.h>

#ifndef lint
static char vcid[] = "$Id: pack.c,v 1.3 1994/06/07 21:22:17 gropp Exp $";
#endif /* lint */

#ifndef MPIR_TRUE
#define MPIR_TRUE  1
#define MPIR_FALSE 0
#endif


/*@
    MPI_Pack - Packs a datatype into contiguous memory

Input Parameters:
. inbuf - input buffer start (choice) 
. incount - number of input data items (integer) 
. datatype - datatype of each input data item (handle) 
. outcount - output buffer size, in bytes (integer) 
. position - current position in buffer, in bytes (integer) 
. comm - communicator for packed message (handle) 

Output Parameter:
. outbuf - output buffer start (choice) 

@*/ 

Int MPIR_Pack (void far *buf, Int count, MPI_Datatype type, void far *dest);

Int MPI_Pack ( 	void far *inbuf, Int incount, MPI_Datatype type, 
				void far *outbuf, Int outcount, Int far *position, MPI_Comm comm)
{
	Int size, pad, errno = MPI_SUCCESS;
	
	/* NOT ENOUGH ERROR CHECKING AT PRESENT */
	if (MPIR_TEST_COMM(comm,comm) || MPIR_TEST_DATATYPE(comm,type) ||
		MPIR_TEST_COUNT(comm,incount) || MPIR_TEST_ARG(position) ||
		( (*position < 0 ) && ((errno = MPI_ERR_ARG) != 0) ) )
	{ 
		return MPIR_ERROR(comm,errno,"Error in MPI_PACK" );
	}
	  
	/* What kind of padding is necessary? */
	pad = (type->align - ((*position) % type->align)) % type->align;
	
	/* Is there enough room to finish packing the type? */
	MPIR_Pack_size ( incount, type, comm, &size );
	if (((*position) + pad + size) > outcount)
		return MPIR_ERROR(comm, MPI_ERR_LIMIT, "Buffer too small in MPI_PACK");
	
	/* Figure the pad and adjust position */
	(*position) += pad;
//	errno = MPIR_Pack(inbuf, incount, type, (Int)outbuf + (*position));
	errno = MPIR_Pack(inbuf, incount, type, ((char far *)outbuf) + (*position));
	(*position) += size;
	return (errno);
}


/*
   This code assumes that we can use char * pointers (previous code 
   incremented pointers by considering them integers, which is even 
   less portable).  Systems that, for example, use word-oriented pointers
   may want to use different code.

   This code is used in dmpi/dmpipk.c to pack data for a device that
   only supports contiguous messages.
 */
Int MPIR_Pack ( void far *buf, Int count, MPI_Datatype type, void far *dest)
{
	Int i, j;
	Int pad = 0;
	char far *tmp_buf;
	char far *lbuf = (char far *)buf, far *ldest = (char far *)dest;
	
	/* Pack contiguous data */
	/* At this point, if the type is contiguous, it should be
	 a basic type, so we could pack it with something other
	 than memcpy */
	if (type->is_contig) 
	{
		_fmemcpy ( dest, buf, (int)(type->size * count) );
		return (MPI_SUCCESS);
	}
	
	/* For each of the count arguments, pack data */
	switch (type->dte_type) 
	{
	/* Contiguous types */
		case MPIR_CONTIG:
			MPIR_Pack ( buf, count * type->count, type->old_type, dest );
			break;
	
	/* Vector types */
		case MPIR_VECTOR:
		case MPIR_HVECTOR:
			if (count > 1)
				pad = (type->align - (type->size % type->align)) % type->align;
			tmp_buf = lbuf;
			for (i=0; i<count; i++) 
			{
				lbuf = tmp_buf;
				for (j=0; j<type->count; j++) 
				{
					MPIR_Pack (lbuf, type->blocklen, type->old_type, ldest);
					lbuf  += (type->stride);
					if ((j+1) != type->count)
						ldest += ((type->blocklen * type->old_type->size) + type->pad);
				}
				ldest += ((type->blocklen * type->old_type->size) + pad);
				tmp_buf += type->extent;
			}
			break;
	
	/* Indexed types */
		case MPIR_INDEXED:
		case MPIR_HINDEXED:
			if (count > 1)
				pad = (type->align - (type->size % type->align)) % type->align;
			for (i=0; i<count; i++) 
			{
				for (j=0;j<type->count; j++) 
				{
					tmp_buf  = lbuf + type->indices[j];
					MPIR_Pack (tmp_buf, type->blocklens[j], type->old_type, ldest);
					if ((j+1) != type->count)
						ldest += ((type->blocklens[j]*type->old_type->size)+type->pad);
				}
				ldest += ((type->blocklens[j]*type->old_type->size) + pad);
				lbuf += type->extent;
			}
			break;
	
	/* Struct type */
		case MPIR_STRUCT:
			if (count > 1)
				pad = (type->align - (type->size % type->align)) % type->align;
			for (i=0; i<count; i++) 
			{
				for (j=0;j<type->count; j++) 
				{
					tmp_buf  = lbuf + type->indices[j];
					MPIR_Pack(tmp_buf,type->blocklens[j],type->old_types[j],ldest);
					if ((j+1) != type->count)
						ldest += ((type->blocklens[j] * type->old_types[j]->size) +
									 type->pads[j]);
				}
				ldest+=((type->blocklens[type->count-1]*
						   type->old_types[type->count-1]->size)+pad);
				lbuf +=type->extent;
			}
			break;
	
		default:
			return MPIR_ERROR (MPI_COMM_WORLD, MPI_ERR_TYPE, "Error in MPI_PACK");
//			break;
	}
	
	/* Everything fell through, must have been successful */
	return (MPI_SUCCESS);
}
