/*
 *  $Id: dmpi.c,v 1.14 1994/06/07 21:30:12 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/*  dmpi.c - routines in mpir that are called by the device */

/***********************************************************************
*                                                                      *
*   dmpi.c                                                             *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop
#include <memory.h>

#ifndef lint
static char vcid[] = "$Id: dmpi.c,v 1.14 1994/06/07 21:30:12 gropp Exp $";
#endif /* lint */

#define MPIR_MIN(a,b) (a) < (b) ? (a) : (b)
#define DEVICE_PREFERS_MEMCPY 1

/* called by device when a message arrives.  Returns 1 if there is posted
 * receive, 0 otherwise.
 *
 * This puts the responsibility for searching the unexpected queue and
 * posted-receive queues on the device.  If it is operating asynchronously
 * with the user code, the device must provide the necessary locking mechanism.
 */

void DMPI_msg_arrived (Int src, Int tag, MPIR_CONTEXT context_id, 
			LPPMPIR_RHANDLE dmpi_recv_handle, Int far *foundflag)
{
    Int          found;
    MPIR_RHANDLE far * handleptr;

    MPIR_search_posted_queue( src, tag, context_id, &found, 1, dmpi_recv_handle);
    if ( found )
    {
		*foundflag = 1;	
		/* note this overwrites any wild-card values in the posted handle */
		handleptr         	= *dmpi_recv_handle;
		handleptr->source 	= src;
		handleptr->tag  	= tag;
		/* count is set in the put and get routines */
    }
    else
    {
		/* allocate handle and put in unexpected queue */
		*dmpi_recv_handle       = 
		    ( MPIR_RHANDLE far * ) MPIR_SBalloc ( MPIR_rhandles );
		handleptr         	= *dmpi_recv_handle;
		handleptr->handle_type  = MPIR_RECV;
		handleptr->source 	= src;
		handleptr->tag  	= tag;
		handleptr->contextid    = context_id;
		handleptr->datatype     = MPI_BYTE;
		handleptr->completed    = MPIR_NO;
		MPID_alloc_recv_handle( &(handleptr->dev_rhandle) );
		
		MPIR_enqueue( &MPIR_unexpected_recvs, (void far * ) *dmpi_recv_handle,
	                      MPIR_QSHANDLE );
		*foundflag = 0;
    }
}

/*
 *  This code implements one of the interfaces between the device and the
 *  MPI code for moving data between the two systems.  This code delivers
 *  to the device contiguous chuncks of data, packing or unpacking as
 *  required.
 */

/* get length of data to be sent
 */
void DMPI_get_totallen(MPIR_SHANDLE far *handle, Int far *len)
{
    *len = (handle->count)*(handle->datatype->size);
				/* correct for contig messages, at least*/
    handle->totallen       = *len;
    handle->bufpos         = handle->bufadd;
    handle->transfer_count = 0;
}

/* tell mpi how many bytes have been read 
 */
void DMPI_put_totallen(MPIR_RHANDLE far *handle, Int len)
{
    handle->totallen       = len;
    handle->bufpos         = handle->bufadd;
    handle->transfer_count = 0;
}

/* SENDS */
/* device has send buffer ready, wants mpi to copy data to it
 */
void DMPI_get_into_contig (MPIR_SHANDLE far *handle, void far *addr, 
				Int maxlen, Int far *actlen)
{
	Int len_left = (handle->count) * (handle->datatype->size) - 
	               handle->transfer_count;

	*actlen = MPIR_MIN(len_left,maxlen);
#ifdef DEVICE_PREFERS_MEMCPY
    _fmemcpy( addr, handle->bufpos, (int)*actlen );
#else
#pragma message ("Provide code ")
    /* device-defined macro for data movement, for example load-store through
       vector registers */
#endif
    handle->bufpos         += *actlen;
    handle->transfer_count += *actlen;
}

/* device will copy mpi's contiguous buffer, wants to know where it is
 */
void DMPI_get_from_contig( MPIR_SHANDLE far *handle, LPPVOID addr, 
							Int maxlen, Int far *actlen)
{
    
    *addr   = handle->bufpos;
    *actlen = (handle->count)*(handle->datatype->size);
    if (*actlen > maxlen) *actlen = maxlen;
    handle->bufpos         += *actlen;
    handle->transfer_count += *actlen;
}

/* RECEIVES */
/* device will put data into mpi's contiguous buffer, wants to know where it is
 */
void DMPI_put_into_contig (MPIR_RHANDLE far *handle, LPPVOID addr, 
				Int maxlen, Int far *actlen)
{
    *addr                  = handle->bufpos;
    *actlen                = (handle->actcount)*(handle->datatype->size);
    if (*actlen > maxlen) *actlen = maxlen;
    handle->bufpos         += *actlen;
    handle->transfer_count += *actlen;
}

/* device has contiguous buffer ready, wants mpi to copy data from it
 */
void DMPI_put_from_contig (MPIR_RHANDLE far *handle, void far *addr, Int maxlen)
{
    Int actlen;

    actlen = MPIR_MIN( maxlen, handle->totallen - handle->transfer_count);

#ifdef DEVICE_PREFERS_MEMCPY
    _fmemcpy( handle->bufpos, addr, (int)actlen );
#else
#pragma message ("Provide code ")
    /* device-defined macro for data movement, for example load-store through
       vector registers */
#endif
    handle->bufpos         += actlen;
    handle->transfer_count += actlen;
    handle->actcount        =  handle->transfer_count / 
										(handle->datatype->size);
}

/*
   Let the device tell the API that a handle can be freed (this handle
   was generated by an unexpected receive and inserted by DMPI_msg_arrived.
 */
void DMPI_free_unexpected (MPIR_RHANDLE far *dmpi_recv_handle)
{
	MPIR_SBfree( MPIR_rhandles, dmpi_recv_handle );
}
