/*
 *  $Id: group_rincl.c,v 1.12 1994/06/07 21:24:39 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */


/***********************************************************************
*                                                                      *
*   grp_rin.c                                                          *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop
#include <malloc.h>
#include <memory.h>

/*@

MPI_Group_range_incl - Creates a new group from ranges of ranks in an 
        existing group

Input Parameters:
. group - group (handle) 
. n - number of triplets in array  ranges (integer) 
. ranges - a one-dimensional array of integer triplets, of the 
form (first rank, last rank, stride) indicating ranks in
group  or processes to be included in newgroup  

Output Parameter:
. newgroup - new group derived from above, in the 
order defined by  ranges  (handle)  

Note:
This implementation does not currently check to see that the list of
ranges to include are valid ranks in the group.

@*/
Int MPI_Group_range_incl ( MPI_Group group, Int n, Int far ranges[][3], 
				MPI_Group far *newgroup )
{
  Int i, j, k, ranks, first, last, stride;
  Int np = 0;
  MPI_Group new_group;
  Int errno = MPI_SUCCESS;

  if ( MPIR_TEST_GROUP(MPI_COMM_WORLD,group) )
    return MPIR_ERROR( MPI_COMM_WORLD, errno, 
					  "Error in MPI_GROUP_RANGE_INCL" );

  /* Check for a EMPTY input group or EMPTY sized new group */
  if ( (group == MPI_GROUP_EMPTY) || (n <= 0) ) {
    MPIR_Group_dup ( MPI_GROUP_EMPTY, newgroup );
    return (errno);
  }
  
  /* Determine the number of ranks that will be included */
  for (i=0; i<n; i++)
    if (ranges[i][2] != 0)
      if ( (ranks=((ranges[i][1]-ranges[i][0])/ranges[i][2])+1) > 0 )
        np += ranks;

  /* Check for np == 0 ranks to include */
  if ( np <=0 ) {
    MPIR_Group_dup ( MPI_GROUP_EMPTY, newgroup );
    return (errno);
  }

  /* Create the new group */
  new_group = (*newgroup)   = MPI_NEW(struct MPIR_GROUP);
  if (!new_group) 
	return MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_EXHAUSTED, 
				  "Out of space in MPI_GROUP_RANGE_INCL" );
  new_group->ref_count      = 1;
  new_group->permanent      = 0;
  new_group->local_rank     = MPI_UNDEFINED;
  new_group->set_mark       = NULL;
  new_group->np             = np;
  new_group->lrank_to_grank = (Int far *) MPI_MALLOC( np * sizeof(Int) );
  if (!new_group->lrank_to_grank) {
	return MPIR_ERROR( MPI_COMM_WORLD, MPI_ERR_EXHAUSTED, 
				  "Out of space in MPI_GROUP_RANGE_INCL" );
  }

  /* Fill in the lrank_to_grank list */
  k = 0;
  for (i=0; i<n; i++) {
    first = ranges[i][0]; last = ranges[i][1]; stride = ranges[i][2];
    if (stride != 0)
      for ( j=first; j*stride <= last*stride; j += stride ) {
        if ( (j < group->np) && (j >= 0) ) {
          if ( group->local_rank == j )  
            new_group->local_rank = k;
          new_group->lrank_to_grank[k++] = group->lrank_to_grank[j];
        }
        else
          new_group->lrank_to_grank[k++] = MPI_UNDEFINED;
      }
  }

  /* Determine the previous and next powers of 2 */
  MPIR_Powers_of_2 ( new_group->np, &(new_group->N2_next), 
		     &(new_group->N2_prev) );

  return (errno);
}

