/*
 *  $Id: barrier.c,v 1.13 1994/06/07 21:29:42 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   barrier.c                                                          *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop

#ifndef lint
static char vcid[] = "$Id: barrier.c,v 1.13 1994/06/07 21:29:42 gropp Exp $";
#endif /* lint */

#ifdef __BORLANDC__
/* BC cant find local header files - BUG ??? */
#include "..\src\coll\coll.h"
#else
#include "coll.h"
#endif /* __BORLANDC__ */

/*@

MPI_Barrier - Blocks until all process have reached this routine.

Input Parameters:
. comm - communicator (handle) 

Notes:
Blocks the caller until all group members have called it; 
the call returns at any process only after all group members
have entered the call.

Algorithm:  
A tree-like or combine algorithm is used to broadcast a message 
to all members of the communicator.  We can modifiy this to
use "blocks" at a later time (see MPI_Bcast).

@*/
Int MPI_Barrier ( MPI_Comm comm )
{
  Int        rank, size, N2_prev, surfeit;
  Int        d, dst, src;
  Int        flag;
  Int        errno = MPI_SUCCESS;
  MPI_Status status;

  /* Check for valid communicator */
  if ( MPIR_TEST_COMM(comm,comm) )
    return MPIR_ERROR(comm, errno, "Error in MPI_BARRIER");

  /* Check for intra-communicator */
  MPI_Comm_test_inter ( comm, &flag );
  if (flag) 
    return MPIR_ERROR(comm, MPI_ERR_COMM,
                      "Inter-communicator invalid in MPI_BARRIER");

  /* Intialize communicator size */
  (void) MPI_Comm_size ( comm, &size );

  /* If there's only one member, this is trivial */
  if ( size > 1 ) {

    /* Initialize collective communicator */
    comm = comm->comm_coll;
    (void) MPI_Comm_rank ( comm, &rank );
    (void) MPIR_Comm_N2_prev ( comm, &N2_prev );
    surfeit = size - N2_prev;

    /* Lock for collective operation */
    MPID_THREAD_LOCK(comm);

    /* Perform a combine-like operation */
    if ( rank < N2_prev ) {
      if( rank < surfeit ) {

        /* get the fanin letter from the upper "half" process: */
        dst = N2_prev + rank;

        MPI_Recv((void far *)0,0,MPI_INT,dst,MPIR_BARRIER_TAG, comm, &status);
      }

      /* combine on embedded N2_prev power-of-two processes */
      for (d = 1; d < N2_prev; d <<= 1) {
        dst = (rank ^ d);

        MPI_Sendrecv( (void far *)0,0,MPI_INT,dst, MPIR_BARRIER_TAG,
                     (void far *)0,0,MPI_INT,dst, MPIR_BARRIER_TAG, 
                     comm, &status);
      }

      /* fanout data to nodes above N2_prev... */
      if ( rank < surfeit ) {
        dst = N2_prev + rank;
        MPI_Send( (void far *)0, 0, MPI_INT, dst, MPIR_BARRIER_TAG, comm);
      }
    } 
    else {
      /* fanin data to power of 2 subset */
      src = rank - N2_prev;
      MPI_Sendrecv( (void far *)0, 0, MPI_INT, src, MPIR_BARRIER_TAG,
                   (void far *)0, 0, MPI_INT, src, MPIR_BARRIER_TAG, 
                   comm, &status);
    }

    /* Unlock for collective operation */
    MPID_THREAD_UNLOCK(comm);

  } 
  return(MPI_SUCCESS); 
}




