/*
 *  $Id: chrecv.c,v 1.23 1994/05/28 19:43:51 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   p4recv.c                                                           *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop

#ifndef lint
static char vcid[] = "$Id: chrecv.c,v 1.23 1994/05/28 19:43:51 gropp Exp $";
#endif /* lint */

#include <memory.h>
#include <malloc.h>
//#include "mpid.h"

#ifndef MPI_ANY_TAG
#define MPI_ANY_SOURCE 	(-1)
#define MPI_ANY_TAG	(-1)
#endif

Int MPID_P4_check_incoming( MPID_BLOCKING_TYPE is_blocking);
/* 
  This file contains the routines to handle receiving a message

  Because we don't know the length of messages (at least the long ones with
  tag MPID_PT2PT2_TAG(src) tags), we never post a receive.  Rather, we have
  a MPID_ch_check_incoming routine that looks for headers.  Note that
  messages sent from a source to destination with the MPID_PT2PT2_TAG(src)
  are ordered (we assume that the message-passing system preserves order).
 */


/***************************************************************************/
/* These routines enable the debugging output                              */
/***************************************************************************/
static Int DebugFlag = 0;
                      
void MPID_SetDebugFlag(Int f)
{
	DebugFlag = f;
}
                          
void MPID_SetRecvDebugFlag(Int f)
{
	DebugFlag = f;
	MPID_SetSyncDebugFlag( f );
} 

/***************************************************************************/

/***************************************************************************/
/* These are used to keep track of the number and kinds of messages that   */
/* are received                                                            */
/***************************************************************************/
static Int n_short       = 0,         /* short messages */
           n_long        = 0,         /* long messages */
           n_unexpected  = 0,         /* unexpected messages */
           n_syncack     = 0;         /* Syncronization acknowledgments */
static Int DebugMsgFlag = 0;

void MPID_SetMsgDebugFlag(Int f)
{
	DebugMsgFlag = f;
} 

Int MPID_GetMsgDebugFlag(void)
{
	return DebugMsgFlag;
}     

void MPID_PrintMsgDebug(void)
{
//	fprintf( stdout, "[%ld] short = %ld, long = %ld, unexpected = %ld, ack = %ld\n",
//		 __MYPROCID, n_short, n_long, n_unexpected, n_syncack );
	printf("[%ld] short = %ld, long = %ld, unexpected = %ld, ack = %ld\n",
			__MYPROCID, n_short, n_long, n_unexpected, n_syncack );
}
/***************************************************************************/

/* Unpack a short message from a packet.
   If address is provided, use that for the data.  Should be
   mpid_recv_handle->start or an allocated location */
void MPID_P4_extract_short(void far *address, MPIR_RHANDLE far *dmpi_recv_handle, 
				Int len, void far *inbuf)
{
	n_short++;
	if (address) 
	    _fmemcpy( address, inbuf, (int)len );
	else 
	{
	    DMPI_put_totallen( dmpi_recv_handle, len );
	    DMPI_put_from_contig( dmpi_recv_handle, inbuf, len );
	}
	if (DebugFlag) 
	{
	    printf( "[%ld]R Got data from packet\n", __MYPROCID );
	    if (len < 78 && address) 
	    {
			Int i; 
			char far *aa = (char far *)address;
			
			for (i=0; i<len; i++) 
			    printf( "%lx", aa[i] );
			printf( "\n" );
		}
//	    fflush( stdout );
	}
}

/* Extract a long message from a packet
   If address is provided, use that for the data.  Should be
   mpid_recv_handle->start or an allocated location.

   inbuf is provided in case the chosen protocol places some data in the
   initial packet. 
 */
void MPID_P4_extract_long(void far *address, MPIR_RHANDLE far *dmpi_recv_handle, 
				Int len, void far *inbuf, Int from)
{
	Int  actual_len;
	void far *laddress;
	
	inbuf = inbuf;


	n_long++;             
	if (address) 
	{
	    char far *__p4lbuf=0;
	    
	    __P4LEN=len;
	    __P4FROM=-1;
	    __P4TYPE=MPID_PT2PT2_TAG(from);
		p4_recv(&__P4TYPE, &__P4FROM, &__p4lbuf, &__P4LEN);
		_fmemcpy((char far *)address,__p4lbuf, (int)__P4LEN);
	}
	else 
	{
	    DMPI_put_totallen( dmpi_recv_handle, len );
	    actual_len = len;
	    DMPI_put_into_contig( dmpi_recv_handle, (LPPVOID)(&laddress), 
				  len, &actual_len );
	    {
	    char far *__p4lbuf=0;
	    __P4LEN=actual_len;
	    __P4FROM=-1;
	    __P4TYPE=MPID_PT2PT2_TAG(from);
		p4_recv(&__P4TYPE, &__P4FROM, &__p4lbuf, &__P4LEN);
		_fmemcpy(laddress, __p4lbuf, (int)__P4LEN);
		}
	}
	if (DebugFlag) 
	{
	    printf( "[%ld]R Got data from packet\n", __MYPROCID );
	    printf( "[%ld]R Receiving a long message from %ld\n", __MYPROCID, from );
//	    fflush( stdout );
	}
}

/*
   This code is called when a receive finds that the message has already 
   arrived and has been placed in the unexpected queue.  This code
   stores the information about the message (source, tag, length),
   copies the message into the receiver's buffer, and generates a
   acknowledgement if the message has mode SYNC.

   dmpi_recv_handle is the API's receive handle that is to receive the
   data.

   dmpi_unexpected is the handle of the data found in the unexpected queue.
 */
Int MPID_P4_Process_unexpected(MPIR_RHANDLE far *dmpi_recv_handle, 
				   MPIR_RHANDLE far *dmpi_unexpected)
{
	MPID_RHANDLE far *mpid_recv_handle;
	MPID_RHANDLE far *mpid_recv_handle_unex;
	Int errno = MPI_SUCCESS;
	
	n_unexpected++;
	
	if (DebugFlag) 
	{
	    printf( "[%ld]R Found message in unexpected queue\n", __MYPROCID );
//	    fflush( stdout );
	}
	/* Copy relevant data to recv_handle */
	mpid_recv_handle	   = &dmpi_recv_handle->dev_rhandle;
	mpid_recv_handle_unex	   = &dmpi_unexpected->dev_rhandle;
	dmpi_recv_handle->source   = dmpi_unexpected->source;
	dmpi_recv_handle->tag	   = dmpi_unexpected->tag;
	dmpi_recv_handle->totallen = mpid_recv_handle_unex->bytes_as_contig;
	if (DebugFlag) 
	{
	    printf( "[%ld]R Found message in temp area of %ld bytes...\n", __MYPROCID,
		    mpid_recv_handle_unex->bytes_as_contig );
//	    fflush( stdout );
	}
	/* Error test on length of message */
	if (mpid_recv_handle->bytes_as_contig < dmpi_recv_handle->totallen) 
	{
	    mpid_recv_handle_unex->bytes_as_contig = mpid_recv_handle->bytes_as_contig;
	    errno = MPI_ERR_TRUNCATE;
	    (*MPID_ErrorHandler)( 1, "Truncated message"  );
	}
	if (mpid_recv_handle_unex->bytes_as_contig > 0) 
	{
	    if (mpid_recv_handle->start) 
	    {
			if (DebugFlag) 
			{
			    printf( "[%ld]R About to copy to %lx from %lx...\n", __MYPROCID,
				   mpid_recv_handle_unex->start, mpid_recv_handle_unex->temp );
//			    fflush( stdout );
			}
			_fmemcpy( mpid_recv_handle->start, mpid_recv_handle_unex->temp,
			       (int)mpid_recv_handle_unex->bytes_as_contig );
		}
	    else 
	    {
			if (DebugFlag) 
			{
			    printf( "[%ld]R About to use DMPI to copy from %lx...\n", __MYPROCID,
				   mpid_recv_handle_unex->temp );
//			    fflush( stdout );
		    }
			DMPI_put_totallen( dmpi_recv_handle, 
					  mpid_recv_handle_unex->bytes_as_contig );
			DMPI_put_from_contig( dmpi_recv_handle, 
					     (void far *)(mpid_recv_handle_unex->temp),
					     mpid_recv_handle_unex->bytes_as_contig );
		}
	}
	if (DebugFlag) 
	{
	    printf( "[%ld]R Copied message out of temp area; send mode is %lx ...\n", 
		    __MYPROCID, mpid_recv_handle_unex->mode );
//	    fflush( stdout );
	}
	
	if (mpid_recv_handle_unex->temp) 
	    MPI_FREE(mpid_recv_handle_unex->temp );
	mpid_recv_handle_unex->temp = 0;      /* In case of a cancel */
	
	/* Return the synchronization message */
	if ((mpid_recv_handle_unex->mode & MPID_MODE_MASK) == MPIR_MODE_SYNCHRONOUS) 
	{
	    if (DebugFlag) 
	    {
			printf( "[%ld]SYNC Returning sync for %lx to %ld\n", __MYPROCID,
			        mpid_recv_handle_unex->mode, mpid_recv_handle_unex->from );
//			fflush( stdout );
		}
	    n_syncack++;
	    MPID_SyncReturnAck( mpid_recv_handle_unex->mode, 
			        mpid_recv_handle_unex->from );
	}
	
	mpid_recv_handle->done = MPIR_YES;
	DMPI_mark_recv_completed(dmpi_recv_handle);
	
	/* Recover dmpi_unexpected */
	DMPI_free_unexpected( dmpi_unexpected );
	if (DebugFlag) 
	{
	    printf( "[%ld]R Leaving 'process unexpected' ...\n", __MYPROCID );
//	    fflush( stdout );
	}
	return errno;
}


/*
   Post a receive.

   Since the Chameleon implementation lets the underlying message transport
   layer handle this, there isn't much to do.  We DO handle the case of
   posting a BLOCKING receive by doing BLOCKING waits on incoming messages.

   Otherwise, we simply try to handle any receives that are ready for
   processing.
 */
Int MPID_P4_post_recv(MPIR_RHANDLE far *dmpi_recv_handle, Int far *is_available)
{
    MPID_RHANDLE far *mpid_recv_handle;
    // MPID_RHANDLE *mpid_recv_handle_unex;
    MPIR_RHANDLE far *dmpi_unexpected;
    Int          found;
	
    mpid_recv_handle = &dmpi_recv_handle->dev_rhandle;
    *is_available    = 0;
	
    if (DebugFlag) 
    {
	printf( "[%ld]R starting recv for tag = %ld, source = %ld, ctx = %ld\n", 
	__MYPROCID, dmpi_recv_handle->tag, dmpi_recv_handle->source,
	dmpi_recv_handle->contextid );
//	fflush( stdout );
    }
    /* At this time, we check to see if the message has already been received.
       (this is a macro that checks first to see if the queue is empty) */
    DMPI_search_unexpected_queue( dmpi_recv_handle->source, 
			   dmpi_recv_handle->tag, dmpi_recv_handle->contextid, 
			   &found, 1, &dmpi_unexpected );
    if (found) 
    {
	if (DebugFlag) 
	{
	    printf( "[%ld]R found in unexpected queue\n", __MYPROCID );
//	    fflush( stdout );
	}
	MPID_P4_Process_unexpected( dmpi_recv_handle, dmpi_unexpected );
	*is_available = 1;
	return MPI_SUCCESS;
    }
	
	/* Add to the posted receive queue */
	MPIR_enqueue( &MPIR_posted_recvs, dmpi_recv_handle, MPIR_QRHANDLE );
	
	/* If we got here, the message is not yet available */
	if (!mpid_recv_handle->is_non_blocking) 
	{
	    if (DebugFlag) 
	    {
			printf( "[%ld]R Blocking recv; starting wait loop\n", __MYPROCID );
//			fflush( stdout );
		}
	    /* Continue to process incoming messages until the one we want 
	       arrives */
	    while (!dmpi_recv_handle->completed) {
		(void)MPID_P4_check_incoming( MPID_BLOCKING );
		}
	}
	else 
	{
	    if (DebugFlag) 
	    {
			printf( 
			       "[%ld]R About to do a non-blocking check of incoming messages\n",
			        __MYPROCID );
//			fflush( stdout );
		}
	    /* Process all pending messages until there are none left */
	    while (MPID_P4_check_incoming( MPID_NOTBLOCKING ) != -1) 
	    	;
	}
	
	if (DebugFlag) 
	{
	    printf( "[%ld]R Exiting post receive\n", __MYPROCID );
//	    fflush( stdout );
	}
	
	return MPI_SUCCESS;
}

/*
   Copy the body of a message into the destination buffer for a posted
   receive 
 */
Int MPID_P4_Copy_body(MPIR_RHANDLE far *dmpi_recv_handle, MPID_PACKET far *pkt, Int from)
{
	MPID_RHANDLE far *mpid_recv_handle;
	
	mpid_recv_handle = &dmpi_recv_handle->dev_rhandle;
	
	dmpi_recv_handle->totallen = pkt->len;
	if (pkt->len <= MPID_PACKET_SIZE) 
	    MPID_P4_extract_short( mpid_recv_handle->start, 
				  dmpi_recv_handle, pkt->len, pkt->buffer );
	else 
	    MPID_P4_extract_long( mpid_recv_handle->start, 
				 dmpi_recv_handle, pkt->len, pkt->buffer, from );
	
	if ((pkt->mode & MPID_MODE_MASK) == MPIR_MODE_SYNCHRONOUS) 
	{
	    if (DebugFlag) 
	    {
			printf( "[%ld]SYNC Returning sync for %lx to %ld\n", __MYPROCID,
			       pkt->mode, from );
//			fflush( stdout );
		}
	    n_syncack++;
	    MPID_SyncReturnAck( pkt->mode, from );
	}
	DMPI_mark_recv_completed(dmpi_recv_handle);
	mpid_recv_handle->done = MPIR_YES;
	
	return MPI_SUCCESS;
}

/*
   Copy the body of a message into the destination buffer for an
   unexpected message.
 */
Int MPID_P4_Copy_body_unex(MPIR_RHANDLE far *dmpi_recv_handle, MPID_PACKET far *pkt, Int from)
{
	MPID_RHANDLE far *mpid_recv_handle;
	char far *address;
	
	mpid_recv_handle = &dmpi_recv_handle->dev_rhandle;
	/* We buffer the message on the receiver.   There are, of course, other
	   strategies */
	/* We want to avoid allocating storage of size zero */
	if (pkt->len > 0) 
	{ 
	    address = (char far *)MPI_MALLOC((size_t)pkt->len ); 
	    if (!address) 
	    {
			(*MPID_ErrorHandler)( 1, 
					 "No more memory for storing unexpected messages"  );
			return MPI_ERR_EXHAUSTED;
		}
	    if (pkt->len <= MPID_PACKET_SIZE) 
			MPID_P4_extract_short( address, NULL, 
					      pkt->len, pkt->buffer );
	    else 
			MPID_P4_extract_long( address, NULL, 
					     pkt->len, pkt->buffer, from );
	}
	else 
	    address = (void far *)0;
	
	if (DebugFlag) 
	{
	    printf( "[%ld]SYNC Returning sync for %lx to %ld\n", __MYPROCID,
		   pkt->mode, from );
//	    fflush( stdout );
	}
	mpid_recv_handle->temp            = address;
	mpid_recv_handle->bytes_as_contig = pkt->len;
	
	/* We also need to remember that the mode of this message incase
	   it was a Synchronous message */
	mpid_recv_handle->mode = pkt->mode;
	mpid_recv_handle->from = from;
	
#ifdef DEBUG_READY
	if ((pkt->mode & MPID_MODE_MASK) == MPIR_MODE_READY) 
	{
	    (*MPID_ErrorHandler)( 1, "Received ready message without matching receive");
	    return MPI_ERR_NOMATCH;
	}
#endif
	return MPI_SUCCESS;
}

/* Check for incoming messages.
    Input Parameter:
.   is_blocking - true if this routine should block until a message is
    available

    Returns -1 if nonblocking and no messages pending
 */    
Int MPID_P4_check_incoming(MPID_BLOCKING_TYPE is_blocking)
{
	MPID_PACKET  pkt;	
	Int          from;
	MPIR_RHANDLE far *dmpi_recv_handle;
	// Int          actual_len;
	MPID_RHANDLE far *mpid_recv_handle;
	Int          is_posted;
	
	/* If nonblocking and no headers available, exit */
#ifndef pvm3
	if (is_blocking == MPID_NOTBLOCKING) 
	{
	    if (!(__P4TYPE=MPID_PT2PT_TAG ,__P4FROM=-1,
						p4_messages_available(&__P4TYPE,&__P4FROM))) 
	    return -1;
	}
	{
		char far *__p4lbuf = 0;
		__P4LEN = sizeof(MPID_PACKET);
		__P4FROM = -1;
		__P4TYPE = MPID_PT2PT_TAG;
		p4_recv(&__P4TYPE,&__P4FROM,&__p4lbuf,&__P4LEN);
		_fmemcpy(&pkt,__p4lbuf, (int)__P4LEN);
	};
#else
	/* pvm3.0 doesn't have a real probe, but what they do have meets the 
	   semantics that we need here, though it is somewhat painful... 
	   All this to save  the user a single routine call in the case where
	   a probe is immediately followed by a recv.  Heaven help you if you
	   use the probe to decide to call some other code to process the 
	   message... 
	*/
	{
		Int bufid, bytes, msgtype; 
		if (is_blocking == MPID_NOTBLOCKING) 
		{
		    if ((bufid = pvm_nrecv( -1, MPID_PT2PT_TAG )) <= 0) 
		        return -1;
		    /* If we found a message, we now have to receive it */
		    pvm_bufinfo( bufid, &bytes, &msgtype, &__PVMFROMTID );
		    pvm_upkint( (Int *)&pkt, bytes / sizeof(Int), 1 );
		    __PVMFROM = -1;
		}
		else 
		{
		    /* For the blocking case, we can use the existing code ... */
		    {
		    char *__p4lbuf=0;
		    __P4LEN=sizeof(MPID_PACKET);
		    __P4FROM=-1;
		    __P4TYPE=MPID_PT2PT_TAG;
		    p4_recv(&__P4TYPE,&__P4FROM,&__p4lbuf,&__P4LEN);
			_fmemcpy(&pkt,__p4lbuf,__P4LEN);
			};
		}
	}
#endif
	
	from = __P4FROM;
	/* We should check the size here for internal errors .... */
	
	if ((pkt.mode & MPID_MODE_MASK) == MPIR_MODE_SYNC_ACK) 
	{
	    if (DebugFlag) 
	    {
			printf( "[%ld]SYNC received sync ack message for mode=%lx from %ld\n",  
			    __MYPROCID, pkt.mode, from );
//			fflush( stdout );
		}
	    MPID_SyncAck( pkt.mode );
	    return MPI_SUCCESS;    
	}
	
	if (DebugFlag) 
	{
	    printf( 
	   "[%ld]R received message for tag = %ld, source = %ld, ctx = %ld, mode = %lx\n", 
		    __MYPROCID, pkt.tag, from, pkt.context_id, pkt.mode );
//	    fflush( stdout );
	}
	/* Is the message expected or not? 
	   This routine RETURNS a dmpi_recv_handle, creating one if the message 
	   is unexpected (is_posted == 0) */
	DMPI_msg_arrived( pkt.lrank, pkt.tag, pkt.context_id, 
	                  &dmpi_recv_handle, &is_posted );
	mpid_recv_handle = &dmpi_recv_handle->dev_rhandle;

	mpid_recv_handle = mpid_recv_handle;

	if (DebugFlag) 
	{
	    printf( "[%ld]R msg was %s\n", __MYPROCID, 
		    is_posted ? "posted" : "unexpected" );
	}
	if (is_posted) 
	{
	    MPID_P4_Copy_body( dmpi_recv_handle, &pkt, from );
	}
	else 
	{
	/* Note that DMPI_msg_arrived has already added this message to the 
	   unexpected queue. */
	    MPID_P4_Copy_body_unex( dmpi_recv_handle, &pkt, from );
	}
	return MPI_SUCCESS;    
}

/*
    This routine completes a particular receive.  It does this by processing
    incoming messages until the indicated message is received.
 */
Int MPID_P4_complete_recv(MPIR_RHANDLE far *dmpi_recv_handle)
{
	MPID_RHANDLE far *mpid_recv_handle = &dmpi_recv_handle->dev_rhandle;
	
	mpid_recv_handle = mpid_recv_handle;

	while (!mpid_recv_handle->done) 
	    (void)MPID_P4_check_incoming( MPID_BLOCKING );
	    
	return MPI_SUCCESS;
}

/*
   Special case code for blocking receive.  The "common" case is handled with
   straight-through code; uncommon cases call routines.
 */
Int MPID_P4_blocking_recv(MPIR_RHANDLE far *dmpi_recv_handle)
{
	MPID_RHANDLE far *mpid_recv_handle;
	// MPID_RHANDLE *mpid_recv_handle_unex;
	MPIR_RHANDLE far *dmpi_unexpected, far *dmpi_save_recv_handle;
	MPID_PACKET  pkt;	
	Int          found, from, is_posted, tag, source, context_id;
	// Int			actual_len;
	
	mpid_recv_handle = &dmpi_recv_handle->dev_rhandle;

	mpid_recv_handle = mpid_recv_handle;
	
	if (DebugFlag) 
	{
	    printf( 
		"[%ld]R starting blocking recv for tag = %ld, source = %ld, ctx = %ld\n", 
		    __MYPROCID, dmpi_recv_handle->tag, dmpi_recv_handle->source,
		    dmpi_recv_handle->contextid );
//	    fflush( stdout );
	}
	/* At this time, we check to see if the message has already been received */
	DMPI_search_unexpected_queue( dmpi_recv_handle->source, 
			   dmpi_recv_handle->tag, dmpi_recv_handle->contextid, 
			   &found, 1, &dmpi_unexpected );
	if (found) 
	{
	    MPID_P4_Process_unexpected( dmpi_recv_handle, dmpi_unexpected );
	    return MPI_SUCCESS;
	}
	
	dmpi_save_recv_handle = dmpi_recv_handle;
	/* If we got here, the message is not yet available */
	if (DebugFlag) 
	{
	    printf( "[%ld]R Blocking recv; starting wait loop\n", __MYPROCID );
//	    fflush( stdout );
	}
	tag	   = dmpi_recv_handle->tag;
	context_id = dmpi_recv_handle->contextid;
	source	   = dmpi_recv_handle->source;
	while (!dmpi_save_recv_handle->completed) 
	{
	    {
		    char far *__p4lbuf = NULL;
		    __P4LEN=sizeof(MPID_PACKET);
		    __P4FROM=-1;
		    __P4TYPE=MPID_PT2PT_TAG;
			p4_recv(&__P4TYPE,&__P4FROM,&__p4lbuf,&__P4LEN);
			_fmemcpy(&pkt,__p4lbuf, (int)__P4LEN);
		};
	    from = __P4FROM;
	    /* We should check the size here for internal errors .... */
	
	    if ((pkt.mode & MPID_MODE_MASK) == MPIR_MODE_SYNC_ACK) 
	    {
			MPID_SyncAck( pkt.mode );
			continue;
		}
	    if (DebugFlag) 
	    {
			printf( 
			       "[%ld]R received message for tag = %ld, source = %ld, ctx = %ld\n", 
			       __MYPROCID, pkt.tag, from, pkt.context_id );
//			fflush( stdout );
		}
	    if (pkt.context_id == context_id && 
		(pkt.tag   == tag    || tag == MPI_ANY_TAG) &&
		(pkt.lrank == source || source == MPI_ANY_SOURCE)) 
		{
		/* Found the message that I'm waiting for (it was never queued) */
			is_posted = 1;
			dmpi_recv_handle	 = dmpi_save_recv_handle;
			dmpi_recv_handle->tag	 = pkt.tag;
			dmpi_recv_handle->source = pkt.lrank;
		}
	    else 
	    {
		/* Unexpected message */
			DMPI_msg_arrived( pkt.lrank, pkt.tag, pkt.context_id, 
					  &dmpi_recv_handle, &is_posted );
		}
	    mpid_recv_handle = &dmpi_recv_handle->dev_rhandle;
	    if (DebugFlag) 
	    {
			printf( "[%ld]R msg was %s\n", __MYPROCID, 
			        is_posted ? "posted" : "unexpected" );
		}
	    if (is_posted) 
			MPID_P4_Copy_body( dmpi_recv_handle, &pkt, from );
	    else 
	    	MPID_P4_Copy_body_unex( dmpi_recv_handle, &pkt, from );
	}
	return MPI_SUCCESS;

}


