
/***********************************************************************
*                                                                      *
*   p4_tsr.c                                                           *
*   p4 1.4 for MS-Windows 3.1                                          *
*   current version: 0.99b          07/16/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is the WIN31 version of the p4 Parallel Programming System    *
*   developed at Argonne National Laboratory.  Note their COPYRIGHT.   *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/p4 )                          *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <memory.h>
#include "p4.h"

#include "p4_sys.h"

/*
 * search_p4_queue tries to locate a message of the desired type in the
 * local queue of messages already received.  If it finds one, it dequeues it 
 * if deq is true, and returns its address; otherwise it returns NULL.
 */

struct p4_msg far *search_p4_queue(Int req_type, Int req_from, P4BOOL deq)
{
    struct local_data *p4_local;
    struct p4_queued_msg far *qpp, far *qp;
    struct p4_msg far *tqp;
    P4BOOL found;

    tqp = NULL;
    qpp = NULL;
    found = FALSE;

	p4_local = get_my_p4_local ();                                
    qp = p4_local->queued_messages->first_msg;
    while (qp)
    {
		if (qp->qmsg->ack_req & P4_BROADCAST_MASK)
		{
		    /* RMB p4_dprintfl(00, "subbrdcst type = %ld, sender = %ld\n",
		    						req_type,req_from); */
		    if (subtree_broadcast_p4(qp->qmsg->type, qp->qmsg->from,
		    							(char far *) &qp->qmsg->msg,
					     				qp->qmsg->len, qp->qmsg->data_type))
		    {
				p4_dprintf("search_p4_queue: failed\n");
				return(NULL);
		    }
		    qp->qmsg->ack_req &= ~P4_BROADCAST_MASK;	/* Unset broadcast bit */
		}
		qp = qp->next;
    }

    qp = p4_local->queued_messages->first_msg;
    while (qp && !found)
    {
		if (((qp->qmsg->type == req_type) || (req_type == -1)) &&
		    ((qp->qmsg->from == req_from) || (req_from == -1)))
		{
		    found = TRUE;
		    if (deq)
		    {
				if (p4_local->queued_messages->first_msg ==
				    p4_local->queued_messages->last_msg)
				{
				    p4_local->queued_messages->first_msg = NULL;
				    p4_local->queued_messages->last_msg = NULL;
				}
				else
				{
				    if (qp == p4_local->queued_messages->first_msg)
				    {
						p4_local->queued_messages->first_msg = qp->next;
				    }
				    else if (qp == p4_local->queued_messages->last_msg)
				    {
						p4_local->queued_messages->last_msg = qpp;
						qpp->next = NULL;
				    }
				    else
				    {
						qpp->next = qp->next;
				    }
				}
		    }
		}
		else
		{
		    qpp = qp;
		    qp = qp->next;
		}
    }
    if (found)
    {
		p4_dprintfl(30,"extracted queued msg of type %ld from %ld\n",
			    qp->qmsg->type,qp->qmsg->from);
		tqp = qp->qmsg;
		if (deq)
		{
		    free_quel(qp);
		}
    }
    return (tqp);
}

/*
 * This is the top-level receive routine, called by the user.
 *   req_type is either a desired type or -1.  In the -1 case it will be
 *        modified  by p4_recv to indicate the type actually received.
 *   req_from is either a desired source or -1.  In the -1 case it will be
 *        modified by p4_recv to the source of the message actually received.
 *   msg will be set by p4_recv to point to a buffer containing the message.
 *   len_rcvd will be set by p4_recv to contain the length of the message.
 *
 *   returns 0 if successful; non-zero if error
 */                         
/* function: 	checks in local queue if something appropriate there
 *				if yes, returns it; if no, tries to receive something
 *				from other sources (here only shmem queues)
 *				if this is appropriate, returns it, otherwise stores it in 
 *				local queue and continues receive attempts 
 */				
Int FAR PASCAL _export p4_recv (Int far *req_type, Int far *req_from, 
								LPPSTR msg, Int far *len_rcvd)
{
    struct local_data *p4_local;
    struct p4_msg far *tmsg;
    struct p4_msg far *tempmsg;
    P4BOOL good;

    p4_dprintfl(20, "receiving for type = %ld, sender = %ld\n",
		*req_type, *req_from);
	
	p4_local = get_my_p4_local ();                                
    ALOG_LOG(p4_local->my_id,END_USER,0,"");
    ALOG_LOG(p4_local->my_id,BEGIN_RECV,*req_from,"");

    for (good = FALSE; !good;)
    {
		ALOG_LOG(p4_local->my_id,END_RECV,0,"");
		ALOG_LOG(p4_local->my_id,BEGIN_WAIT,0,"");
		if (!(tmsg = search_p4_queue (*req_type, *req_from, TRUE)))
		{   /* if nothing there already, wait for something that might be interesting */
		    tmsg = recv_message(req_type, req_from);
		    /*****
		    if (tmsg)
			p4_dprintfl(00, "received type = %ld, sender = %ld\n",
			            tmsg->type,tmsg->from);
		    *****/
		}
		ALOG_LOG(p4_local->my_id,END_WAIT,0,"");
		ALOG_LOG(p4_local->my_id,BEGIN_RECV,0,"");
		if (tmsg == NULL)
		{
		    p4_dprintf("p4_recv: could not receive a message\n");
		    return (-1);
		}
		if (((tmsg->type == *req_type) || (*req_type == -1)) &&
		    ((tmsg->from == *req_from) || (*req_from == -1)))
		{
		    good = TRUE;
		}
		if (tmsg->ack_req & P4_BROADCAST_MASK)
		{
		    /* RMB p4_dprintfl(00, "subbrdcst type = %ld, sender = %ld\n",*req_type,*req_from); */
		    if (subtree_broadcast_p4(tmsg->type, tmsg->from, (char far *) &tmsg->msg,
					     tmsg->len, tmsg->data_type))
		    {
				p4_dprintf("p4_recv: subtree_brdcst failed\n");
				return(-1);
		    }
		    tmsg->ack_req &= ~P4_BROADCAST_MASK;	/* Unset broadcast bit */
		}
		if (!good)
		    queue_p4_message(tmsg, p4_local->queued_messages);
    }

    *req_type = tmsg->type;		// return type and sender of message
    *req_from = tmsg->from;

    p4_dprintfl(10, "received type=%ld, from=%ld\n", *req_type, *req_from);

    if (*msg == NULL)
    {
		*msg = (char far *) &(tmsg->msg);
		*len_rcvd = tmsg->len;
    }
    else
    {
		tempmsg = (struct p4_msg far *)
		    ((*msg) - sizeof(struct p4_msg) + sizeof(char far *));
		/* copy into the user's buffer area, truncating if necessary */
		if (tmsg->len < tempmsg->orig_len)
		    *len_rcvd = tmsg->len;
		else
		    *len_rcvd = tempmsg->orig_len;
//		bcopy((char *) &(tmsg->msg),*msg,*len_rcvd);
		_fmemcpy(*msg, &(tmsg->msg), (size_t) *len_rcvd); // Int -> unsigned int
		tmsg->msg_id = (-1);
		free_p4_msg(tmsg);
    }
    ALOG_LOG(p4_local->my_id,END_RECV,*req_from,"");
    ALOG_LOG(p4_local->my_id,BEGIN_USER,0,"");

    return (0);
}

struct p4_msg far *recv_message(Int far *req_type, Int far *req_from)
{
    while (TRUE)
    {
		if (shmem_msgs_available())
		{
		    return (shmem_recv());
		}
/* WIN receiving all other types of messages removed	*/
/* if no msg available, let others produce it			*/
		P4WinMessageLoop ();
    }
    return 0; // dummy
}

static P4BOOL first_call = TRUE;
static struct p4_msg_queue far *local_mq;
static struct p4_msg_queue far *local_qp;

P4BOOL FAR PASCAL _export p4_any_messages_available(void)
/* sometimes want a simple call with little overhead  - SRM 
   did not make much difference - the main overhead is in the select within
   sock_msg_on_fd() */
{    
    struct local_data *p4_local;
	Int  qidx;

	p4_local = get_my_p4_local ();                                
	if(first_call)
    {
//#pragma message ("Check this fix")
// WIN: looks good to me !!!
//		qidx = p4_local->my_id - p4_global->low_cluster_id;
		qidx = p4_local->my_id;
		local_mq = &(p4_global->shmem_msg_queues[qidx]);
		local_qp = p4_local->queued_messages;
		first_call = FALSE;
	}

//#if defined(CAN_DO_SOCKET_MSGS)
//    return((local_qp->first_msg) || (local_mq->first_msg) || socket_msgs_available());
//#else
    return((local_qp->first_msg) || (local_mq->first_msg));
//#endif
}

                            
P4BOOL FAR PASCAL _export p4_messages_available(Int far *req_type, Int far *req_from)
{
    struct local_data *p4_local;
    P4BOOL found;
//    Int len;
    struct p4_msg far *tmsg;

	p4_local = get_my_p4_local ();                                
    ALOG_LOG(p4_local->my_id,END_USER,0,"");
    ALOG_LOG(p4_local->my_id,BEGIN_WAIT,1,"");

    found = FALSE;
    if (tmsg = search_p4_queue(*req_type, *req_from, 0))
    {
		found = TRUE;
		*req_type = tmsg->type;
		*req_from = tmsg->from;
    }

/* while checking for messages, shmem messages are received, too
   and stored in local queue */
    while (!found && shmem_msgs_available())
    {
		tmsg = shmem_recv();
		if (((tmsg->type == *req_type) || (*req_type == -1)) &&
		    ((tmsg->from == *req_from) || (*req_from == -1)))
		{
		    found = TRUE;
		    *req_type = tmsg->type;
		    *req_from = tmsg->from;
		}
		queue_p4_message(tmsg, p4_local->queued_messages);
    }

    ALOG_LOG(p4_local->my_id,END_WAIT,1,"");
    ALOG_LOG(p4_local->my_id,BEGIN_USER,0,"");
/* all other types of message receives removed */

    return (found);
}				/* p4_messages_available */

P4VOID queue_p4_message (struct p4_msg far *msg, struct p4_msg_queue far *hdr)
{
    struct p4_queued_msg far *q;

    q = alloc_quel();
    q->qmsg = msg;
    q->next = NULL;

    if (hdr->first_msg == NULL)	// first message in queue
		hdr->first_msg = q;
    else
		hdr->last_msg->next = q;
    hdr->last_msg = q;
}

Int	FAR PASCAL _export send_message (Int type, Int from, Int to, 
									char far *msg, Int len, Int data_type, 
									P4BOOL ack_req, P4BOOL p4_buff_ind)
{
    struct local_data *p4_local;
    struct p4_msg far *tmsg;
    int conntype;

	p4_local = get_my_p4_local ();                                
//    if (to == 0xffff)		/* NCUBE broadcast */
//		conntype = CONN_LOCAL;
//    else
	conntype = p4_local->conntypetab[to];

    p4_dprintfl(90, "send_message: to = %ld, conntype=%d conntype=%s\n",
		to, conntype, print_conn_type(conntype));
    ALOG_LOG(p4_local->my_id,END_USER,0,"");
    ALOG_LOG(p4_local->my_id,BEGIN_SEND,to,"");

    switch (conntype)
    {
		case CONN_ME:
			tmsg = get_tmsg(type, from, to, msg, len, data_type, ack_req, p4_buff_ind);
			p4_dprintfl(20, "sending msg of type %ld to myself\n",type);
			queue_p4_message(tmsg, p4_local->queued_messages);
			p4_dprintfl(10, "sent msg of type %ld to myself\n", type);
			break;

		case CONN_SHMEM:
			tmsg = get_tmsg(type, from, to, msg, len, data_type, ack_req, p4_buff_ind);
			shmem_send(tmsg);
			break;

		default:
			p4_dprintf("send_message: to=%ld; invalid conn type=%ld\n",to,conntype);
			ALOG_LOG(p4_local->my_id,END_SEND,to,"");
			ALOG_LOG(p4_local->my_id,BEGIN_USER,0,"");
			return (-1);
    }

    ALOG_LOG(p4_local->my_id,END_SEND,to,"");
    ALOG_LOG(p4_local->my_id,BEGIN_USER,0,"");
    return (0);
}				/* send_message */

struct p4_msg far *get_tmsg (Int type, Int from, Int to, char far *msg, Int len, 
							 Int data_type, P4BOOL ack_req, P4BOOL p4_buff_ind)
{
//    Int i;
    struct p4_msg far *tmsg;                

    if (p4_buff_ind)
    {
		tmsg = (struct p4_msg far *) (msg - sizeof(struct p4_msg) + sizeof(char far *));
    }
    else
    {
        tmsg = alloc_p4_msg(len);
		if (tmsg == NULL)
		{
		    p4_dprintf("OOPS! get_tmsg: could not alloc buff **\n");
		    return (NULL);
		}
//		bcopy(msg, (char *) &(tmsg->msg), len);
		_fmemcpy( &(tmsg->msg), msg, (size_t) len); // Int -> unsigned int
    }
    tmsg->type = type;
    tmsg->from = from;
    tmsg->to = to;
    tmsg->len = len;
    tmsg->ack_req = ack_req;
    tmsg->data_type = data_type;
    return (tmsg);
}

/* allocates message and returns pointer to data buffer of message */
char far * FAR PASCAL _export p4_msg_alloc(Int msglen)
{
    char far *t;

    t = (char far *) alloc_p4_msg(msglen);
    ((struct p4_msg far *) t)->msg_id = -1;	/* msg not in use by IPSC isend */
    t = t + sizeof(struct p4_msg) - sizeof(char far *); // Do not need pointer to message ?
    return(t);
}

P4VOID FAR PASCAL _export p4_msg_free(char far *m)
{
    char far *t;

    t = m - sizeof(struct p4_msg) + sizeof(char far *);
    ((struct p4_msg far *) t)->msg_id = -1;	/* msg not in use by IPSC isend */
    free_p4_msg((struct p4_msg far *) t);
}


P4VOID initialize_msg_queue(struct p4_msg_queue far *mq)
{
    mq->first_msg = NULL;
    mq->last_msg = NULL;
    (P4VOID) p4_moninit(&(mq->m), 1);
    p4_lock_init(&(mq->ack_lock));   
    p4_lock(&(mq->ack_lock)); 
}


struct p4_queued_msg far *alloc_quel (void)
{
    struct p4_queued_msg far *q;

    p4_lock(&p4_global->avail_quel_lock);
    if (p4_global->avail_quel == NULL)
    {
		q = (struct p4_queued_msg far *) 
						p4_shmalloc(sizeof(struct p4_queued_msg));
		if (!q)
		    p4_error("alloc_quel:  could not allocate queue element",
			     sizeof(struct p4_queued_msg));
    }
    else
    {
		q = p4_global->avail_quel;
		p4_global->avail_quel = q->next;
    }
    p4_unlock(&p4_global->avail_quel_lock);
    return (q);
}

P4VOID free_quel(struct p4_queued_msg far *q)
{
    p4_lock(&p4_global->avail_quel_lock);
    q->next = p4_global->avail_quel;
    p4_global->avail_quel = q;
    p4_unlock(&p4_global->avail_quel_lock);
}
