
/***********************************************************************
*                                                                      *
*   p4_broad.c                                                         *
*   p4 1.4 for MS-Windows 3.1                                          *
*   current version: 0.99b          07/16/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is the WIN31 version of the p4 Parallel Programming System    *
*   developed at Argonne National Laboratory.  Note their COPYRIGHT.   *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/p4 )                          *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <string.h>
#include "p4.h"
#include "p4_sys.h"

#define MAXVAL(a,b) (((a)>(b)) ? (a) : (b))
#define MINVAL(a,b) (((a)<(b)) ? (a) : (b))
#define ABSVAL(a)   (((a)>=0 ) ? (a) : -(a))

static P4VOID init_p4_brdcst_info();

Int FAR PASCAL _export p4_broadcastx(Int type, char far *data, Int data_len, Int data_type)
/*
  Broadcast my data to all other processes.
  Other processes call p4_recv() in the normal fashion, specifying
  the node (if desired) that originated the broadcast.
*/
{
    Int status = 0;
    struct local_data *p4_local= get_my_p4_local ();
    Int my_id = p4_local->my_id;

    init_p4_brdcst_info();

    /* send to my subtree */
    status = subtree_broadcast_p4(type, my_id, data, data_len, data_type);
    if (my_id != 0)
    {
	/* send to node 0 for rest of tree */
		status = send_message(type, my_id, 0, data, data_len,
				      data_type, P4_BROADCAST_MASK, FALSE);
    }
    if (status && !(SOFTERR))
		p4_error("p4_broadcast failed, type=", type);

    return status;
}

Int subtree_broadcast_p4(Int type, Int from, char far *data, Int data_len, Int data_type)
/* Broadcast message to processes in my subtree */
{
    struct local_data *p4_local;
	Int status = 0;
    Int nodes[2], i;

    init_p4_brdcst_info();

    p4_dprintfl(90, "subtree_broadcast_p4: type=%ld, len=%ld\n", type, data_len);

    p4_local = get_my_p4_local ();

    nodes[0] = p4_local->brdcst_left_slave;
    nodes[1] = p4_local->brdcst_right_slave;

    for (i = 0; i < 2; i++)
    {
		if (nodes[i] > 0  &&  nodes[i] != from)
		{
		    if (send_message(type, from, nodes[i], data, data_len,
				     data_type, P4_BROADCAST_MASK, FALSE))
		    {
				status = -1;
				break;
		    }
		}
    }

    if (status && !SOFTERR)
		p4_error("subtree_broadcast_p4 failed, type=", type);

    p4_dprintfl(90, "subtree_broadcast_p4: exit status=%ld\n", status);
    return status;
}


static P4BOOL p4_brdcst_initialized = FALSE;

static P4VOID init_p4_brdcst_info()
/*
  Construct tree connections for cluster-master and slave
  processes and insert into global structure
*/
/*
 *	WIN: This function computed, when called first, the parent & children in the 
 *	broadcast tree FOR THIS PROCESS and was called in each process 
 *  In WIN we call it only once for the whole application, it has to compute the 
 *	broadcast tree for all processes (updating all entries in p4_localtab)
 */
{
    int i;
    struct local_data *p4_local;
    Int nprocs, node;

    if (p4_brdcst_initialized == TRUE)		/* Only need to do this once */
		return;

    p4_brdcst_initialized = TRUE;
    p4_local = get_my_p4_local ();
    nprocs = p4_num_total_ids();

    for (i = 0; i < P4_MAXPROCS; i++)
    {
        p4_local = p4_localtab + i;
        if (p4_local->hTask == 0)
        	break;		/* no more entries */

		p4_local->brdcst_up = -1;	/* -1 means no one there */
		p4_local->brdcst_left_slave = -1;
		p4_local->brdcst_right_slave = -1;

    /* Assign connections with subtree */
	    node = 2 * i + 1;
	    if (node < nprocs)
			p4_local->brdcst_left_slave = node;
	
	    node = 2 * i + 2;
	    if (node < nprocs)
			p4_local->brdcst_right_slave = node;
	
	    if (i != 0)	/* not root */
			p4_local->brdcst_up = (i - 1) / 2;

    p4_dprintfl(90, "brdcst_info: me=%d up=%ld slaves(%ld, %ld)\n",
				i,
				p4_local->brdcst_up,
				p4_local->brdcst_left_slave,
				p4_local->brdcst_right_slave);

    /* Sanity check */

    if (p4_local->brdcst_up != -1)
		if (CHECKNODE(p4_local->brdcst_up))
		    p4_error("init_p4_brdcst_info: up node is invalid", p4_local->brdcst_up);
    if (p4_local->brdcst_left_slave != -1)
		if (CHECKNODE(p4_local->brdcst_left_slave))
		    p4_error("init_p4_brdcst_info: left_slave node is invalid",
			     p4_local->brdcst_left_slave);
    if (p4_local->brdcst_right_slave != -1)
		if (CHECKNODE(p4_local->brdcst_right_slave))
		    p4_error("init_p4_brdcst_info: right_slave node is invalid",
			     p4_local->brdcst_right_slave);
	}
}


Int FAR PASCAL _export p4_global_op (Int type, LPSTR x, Int nelem, Int size, 
									 FPF (*op), Int data_type)
/* see userman for more details */
{
    struct local_data *p4_local;
    Int me = p4_get_my_id();
    Int status = 0;
    Int zero = 0;
    Int msg_len;
    LPSTR msg;

    init_p4_brdcst_info();

    p4_local = get_my_p4_local ();
    /* Accumulate up broadcast tree ... mess is due to return of soft errors */

    if (!status)
		if (p4_local->brdcst_left_slave > 0)
		{
		    msg = NULL;
		    status = p4_recv(&type, &p4_local->brdcst_left_slave, (LPPSTR)&msg, &msg_len);
		    if (!status)
		    {
				op(x, msg, msg_len / size);
				p4_msg_free(msg);
		    }
		}
    
    if (!status)
		if (p4_local->brdcst_right_slave > 0)
		{
		    msg = NULL;
		    status = p4_recv(&type, &p4_local->brdcst_right_slave, &msg, &msg_len);
		    if (!status)
		    {
				op(x, msg, msg_len / size);
				p4_msg_free(msg);
		    }
		}

    if ((!status) && me)
		status = p4_sendx(type, p4_local->brdcst_up, x, nelem * size, data_type);

    /* Broadcast the result back */

    if (!status)
    {
		if (me == 0)
		    status = p4_broadcastx(type, x, nelem * size, data_type);
		else
		{
		    msg = NULL;
		    status = p4_recv(&type, &zero, &msg, &msg_len);
		    if (!status)
		    {
				_fmemcpy (x, msg, (int)msg_len);    // hope the int cast never crashes
				p4_msg_free(msg);
		    }
		}
    }

    if (status && !SOFTERR)
	p4_error("p4_global_op failed, type=", type);

    return status;
}

/* Standard operations on doubles */

P4VOID FAR PASCAL _export p4_dbl_sum_op(char far *x, char far *y, Int nelem)
{
    double far *a = (double far *) x;
    double far *b = (double far *) y;

    while (nelem--)
		*a++ += *b++;
}

P4VOID FAR PASCAL _export p4_dbl_mult_op(char far *x, char far *y, Int nelem)
{
    double far *a = (double far *) x;
    double far *b = (double far *) y;

    while (nelem--)
		*a++ *= *b++;
}

P4VOID FAR PASCAL _export p4_dbl_max_op(char far *x, char far *y, Int nelem)
{
    double far *a = (double far *) x;
    double far *b = (double far *) y;

    while (nelem--)
    {
		*a = MAXVAL(*a, *b);
		a++;
		b++;
    }
}

P4VOID FAR PASCAL _export p4_dbl_min_op(char far *x, char far *y, Int nelem)
{
    double far *a = (double far *) x;
    double far *b = (double far *) y;

    while (nelem--)
    {
		*a = MINVAL(*a, *b);
		a++;
		b++;
    }
}

P4VOID FAR PASCAL _export p4_dbl_absmax_op(char far *x, char far *y, Int nelem)
{
    double far *a = (double far *) x;
    double far *b = (double far *) y;

    while (nelem--)
    {
		*a = MAXVAL(ABSVAL(*a), ABSVAL(*b));
		a++;
		b++;
    }
}

P4VOID FAR PASCAL _export p4_dbl_absmin_op(char far *x, char far *y, Int nelem)
{
    double far *a = (double far *) x;
    double far *b = (double far *) y;

    while (nelem--)
    {
		*a = MINVAL(ABSVAL(*a), ABSVAL(*b));
		a++;
		b++;
    }
}

/* Standard operations on floats */

P4VOID FAR PASCAL _export p4_flt_sum_op(char far *x, char far *y, Int nelem)
{
    float far *a = (float far *) x;
    float far *b = (float far *) y;

    while (nelem--)
		*a++ += *b++;
}

P4VOID FAR PASCAL _export p4_flt_mult_op(char far *x, char far *y, Int nelem)
{
    float far *a = (float far *) x;
    float far *b = (float far *) y;

    while (nelem--)
		*a++ *= *b++;
}

P4VOID FAR PASCAL _export p4_flt_max_op(char far *x, char far *y, Int nelem)
{
    float far *a = (float far *) x;
    float far *b = (float far *) y;

    while (nelem--)
    {
		*a = MAXVAL(*a, *b);
		a++;
		b++;
    }
}

P4VOID FAR PASCAL _export p4_flt_min_op(char far *x, char far *y, Int nelem)
{
    float far *a = (float far *) x;
    float far *b = (float far *) y;

    while (nelem--)
    {
		*a = MINVAL(*a, *b);
		a++;
		b++;
    }
}

P4VOID FAR PASCAL _export p4_flt_absmax_op(char far *x, char far *y, Int nelem)
{
    float far *a = (float far *) x;
    float far *b = (float far *) y;

    while (nelem--)
    {
		*a = MAXVAL(ABSVAL(*a), ABSVAL(*b));
		a++;
		b++;
    }
}

P4VOID FAR PASCAL _export p4_flt_absmin_op(char far *x, char far *y, Int nelem)
{
    float far *a = (float far *) x;
    float far *b = (float far *) y;

    while (nelem--)
    {
		*a = MINVAL(ABSVAL(*a), ABSVAL(*b));
		a++;
		b++;
    }
}


/* Standard operations on integers */

P4VOID FAR PASCAL _export p4_int_sum_op(char far *x, char far *y, Int nelem)
{
    Int far *a = (Int far *) x;
    Int far *b = (Int far *) y;

    while (nelem--)
		*a++ += *b++;
}

P4VOID FAR PASCAL _export p4_int_mult_op(char far *x, char far *y, Int nelem)
{
    Int far *a = (Int far *) x;
    Int far *b = (Int far *) y;

    while (nelem--)
		*a++ *= *b++;
}

P4VOID FAR PASCAL _export p4_int_max_op(char far *x, char far *y, Int nelem)
{
    Int far *a = (Int far *) x;
    Int far *b = (Int far *) y;

    while (nelem--)
    {
		*a = MAXVAL(*a, *b);
		a++;
		b++;
    }
}

P4VOID FAR PASCAL _export p4_int_min_op(char far *x, char far *y, Int nelem)
{
    Int far *a = (Int far *) x;
    Int far *b = (Int far *) y;

    while (nelem--)
    {
		*a = MINVAL(*a, *b);
		a++;
		b++;
    }
}

P4VOID FAR PASCAL _export p4_int_absmax_op(char far *x, char far *y, Int nelem)
{
    Int far *a = (Int far *) x;
    Int far *b = (Int far *) y;

    while (nelem--)
    {
		*a = MAXVAL(ABSVAL(*a), ABSVAL(*b));
		a++;
		b++;
    }
}

P4VOID FAR PASCAL _export p4_int_absmin_op(char far *x, char far *y, Int nelem)
{
    Int far *a = (Int far *) x;
    Int far *b = (Int far *) y;

    while (nelem--)
    {
		*a = MINVAL(ABSVAL(*a), ABSVAL(*b));
		a++;
		b++;
    }
}
