
/***********************************************************************
*                                                                      *
*   p4_alloc.c                                                         *
*   p4 1.4 for MS-Windows 3.1                                          *
*   current version: 0.99b          07/16/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is the WIN31 version of the p4 Parallel Programming System    *
*   developed at Argonne National Laboratory.  Note their COPYRIGHT.   *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/p4 )                          *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/


#include "p4.h"
#include "p4_sys.h"
#include <stdlib.h>
#include <memory.h>
#include <string.h>

struct local_data * FAR PASCAL _export get_new_p4_local (void)
{
    HTASK	hTask;
    int		i;
    
    hTask = GetCurrentTask ();
    for (i = 0; i < P4_MAXPROCS; i++)
    {
        if (p4_localtab[i].hTask == hTask)
        {
            p4_error ("get_new_p4_local: task %u has already p4_local\n", hTask);
        }
    
        if (p4_localtab[i].hTask == NULL)
            break;
    }
    if (i < P4_MAXPROCS)
    {
    	p4_localtab[i].hTask = hTask;
    	p4_localtab[i].my_id = i;		// my_id pretty trivial now
        return (p4_localtab + i);
    }  
    p4_error ("get_new_p4_local: p4_localtab FULL !!!\n", 0);
    return (NULL);
}

struct local_data *get_my_p4_local (void)
{
    HTASK	hTask;
    int		i;
    
    hTask = GetCurrentTask ();
    for (i = 0; i < P4_MAXPROCS; i++)
    {
        if (p4_localtab[i].hTask == hTask)
            return (p4_localtab + i);
    }
    p4_error ("get_my_p4_local: p4_local for %u not found\n", hTask);
}

struct local_data *alloc_local (void)
{
    struct local_data *l;

    l = get_my_p4_local (); // real allocation already done in WinMain
    if (l == NULL)
    {
		p4_dprintf("OOPS: alloc_local: get_new_p4_local failed \n");
		return (l);
    }

#pragma message ("Could allocate globally to empty local heap ")
    l->queued_messages = (struct p4_msg_queue far *)
				malloc (sizeof(struct p4_msg_queue));
    if (l == NULL)
    {
		p4_dprintf("OOPS: alloc_local: malloc failed \n");
		return (l);
    }
    initialize_msg_queue(l->queued_messages);
    l->soft_errors = 0;

    return (l);
}

struct local_data *alloc_local_bm(void)
{
    struct local_data *l;

    l = alloc_local ();

    l->am_bm = TRUE;

    return (l);
}

struct local_data *alloc_local_slave(void)
{
    struct local_data *l;

    l = alloc_local ();

    l->am_bm = FALSE;

    return (l);
}


/*
    This routine should be called before any sends and receives are done by
    the user.  If not, some buffers may be lost.
*/                                          
/* WIN is not called */
P4VOID p4_set_avail_buff (Int bufidx, Int size)
{
    p4_global->avail_buffs[bufidx].size = size;
    p4_global->avail_buffs[bufidx].buff = NULL;
}


P4VOID init_avail_buffs(void)
{
    static Int sizes[NUMAVAILS] = 
//            {64,256,1024,4096,16384,65536,262144,1048576};
            {64,256,1024,4096,16384};
    Int i;

    for (i = 0; i < NUMAVAILS; i++)
    {
        p4_global->avail_buffs[i].size = sizes[i];
        p4_global->avail_buffs[i].buff = NULL;
    }
}

P4VOID FAR PASCAL _export p4_print_avail_buffs (void)
{
    int i, count;
    struct p4_msg far * next;

    p4_dprintf("avail lists for message buffers:\n");
    p4_lock(&p4_global->avail_buffs_lock);
    for (i = 0; i < NUMAVAILS; i++)
    {
		count = 0;
		for (next = p4_global->avail_buffs[i].buff; next; next = next->link)
		     count++;
		p4_dprintf("%d buffers of size %ld\n",
			   count, p4_global->avail_buffs[i].size);
    }
    p4_unlock(&p4_global->avail_buffs_lock);
}

struct p4_msg far *alloc_p4_msg(Int msglen)
{
    struct p4_msg far *rmsg = NULL;
    Int i, rounded, buff_len;
    struct local_data *p4_local = get_my_p4_local ();

    p4_dprintfl(40, "allocating a buffer for message of size %ld\n", msglen);

    i = 0;
    while ((i < NUMAVAILS) && (msglen > p4_global->avail_buffs[i].size))
		i++;

    if (i == NUMAVAILS)		/* didn't find a big enough avail buffer */
    {
        buff_len = sizeof(struct p4_msg) + msglen - sizeof(char far *);
		rmsg = (struct p4_msg far *) p4_shmalloc(buff_len);
		p4_dprintfl(40, "allocated new buffer at %Fp for message size %ld\n", 
			    rmsg, msglen);
    }
    else
    {
		rounded = p4_global->avail_buffs[i].size;
		buff_len = sizeof(struct p4_msg) + rounded - sizeof(char far *);
		p4_lock(&p4_global->avail_buffs_lock);
		if (p4_global->avail_buffs[i].buff)
		{
	
		    rmsg = p4_global->avail_buffs[i].buff;
		    p4_global->avail_buffs[i].buff = rmsg->link;   // not clear
		    p4_dprintfl(40, "reused a buffer of size %ld for message size %ld\n",
				rounded, msglen);
	
		    p4_unlock(&p4_global->avail_buffs_lock);
		}
		else
		{
		    p4_unlock(&p4_global->avail_buffs_lock);
		    rmsg = (struct p4_msg far *) p4_shmalloc(buff_len);
		    p4_dprintfl(40, "allocated new buffer at 0x%Fp of size %ld for message size %ld\n", 
		    			rmsg, rounded, msglen);
		}
    }

    if ((rmsg == NULL) && !SOFTERR)
		p4_error("alloc_p4_msg failed", 0);

    rmsg->len = msglen;
    rmsg->orig_len = msglen;
    return(rmsg);

}				/* alloc_p4_msg */

P4VOID free_p4_msg(struct p4_msg far *tmsg)
{
    int i;
    struct p4_msg far *p;

    p4_dprintfl(40, "freeing a buffer with bufflen=%ld msglen=%ld\n", 
		tmsg->orig_len,tmsg->len);

    /* Sanity check here as bad message pointer causes havoc */

    if ((tmsg->orig_len < 0) || (tmsg->orig_len > P4_MAX_MSGLEN))
		p4_error("free_p4_msg: bad hdr: msglen out of range", tmsg->len);

    i = 0;
    while ((i < NUMAVAILS) && (tmsg->orig_len > p4_global->avail_buffs[i].size))
		i++;

    if (i == NUMAVAILS)
    {
	/* buffer being freed is not a kept size */
		p4_shfree(tmsg);
		p4_dprintfl(40, "freeing a buffer at %Fp with bufflen=%ld msglen=%ld\n", 
			    tmsg,tmsg->orig_len,tmsg->len);
    }
    else
    {
	/* hook new buffer in at end of list */
		p4_lock(&p4_global->avail_buffs_lock);
		if ((p = p4_global->avail_buffs[i].buff) == NULL)
		    p4_global->avail_buffs[i].buff = tmsg;
		else
		{
		    while (p->link != NULL)
			p = p->link;
		    p->link = tmsg;
		}
		tmsg->link = NULL;
		p4_unlock(&p4_global->avail_buffs_lock);
		p4_dprintfl(40, "saved a buffer of size %ld in avail list for size %ld\n",
			    tmsg->orig_len, p4_global->avail_buffs[i].size);
    }

}				/* free_p4_msg */

P4VOID alloc_global(void)
{
    Int i;
    struct p4_global_data far *g;

    p4_global = (struct p4_global_data far *) p4_shmalloc(sizeof(struct p4_global_data));
    if (p4_global == NULL)
    {
		p4_error("alloc_global: alloc_global failed\n", 
								sizeof(struct p4_global_data));
    }

    g = p4_global;

    p4_lock_init(&g->slave_lock);

    g->procgroup = NULL;
    g->n_started_slaves = 0;

    for (i = 0; i < P4_MAX_MSG_QUEUES; i++)
    {
		initialize_msg_queue(&g->shmem_msg_queues[i]);
		g->dest_id[i] = -1;
    }

    p4_lock_init(&g->avail_buffs_lock);  
    init_avail_buffs();
    p4_lock_init(&g->avail_quel_lock);  
    g->avail_quel = NULL;

    g->num_in_proctable = 0;
    g->num_installed = 0;
    get_qualified_hostname(g->my_host_name);  // = win_node

    p4_barrier_init(&(g->global_barrier));

//    sprintf(g->application_id, "p4_%-8d", getpid());
//    sprintf(exportbuf, "p4_%-8d", GetCurrentTask());
}
