
*/*F
**************************************************
*                                                *
*  subroutine SELF_GRAVITY_3D                    *
*                                                *
*  Solve the Poisson equation and compute the    *
*  gravitational potential  PHI.                 *
*                                                * 
*  This particular version is restricted only to *
*  PERIODIC boundary conditions adn to           *
*  a UNIFORM GRID (i.e. DX = CONST)              *
*                                                * 
*  Here we use a MULTIGRID method                *
*                                                *
*   Lap(PHI) = 4PI rho                           *
*                                                *                              *                                                *
*   Author: Andrea Malagoli                      *
*   Date: 16 July 1995                           *
*                                                *
**************************************************
*F*/
*>>>>>>>>> The SELF-GRAVITY STEP>>>>>*
        subroutine Self_Gravity_3d(sz,mg_li,proc,pgm,
     &               rho, phi, r, x, y, z, wrk,  
     &               step_par, grid_par, fluid_par, do_par )
      include 'meshf.h'
      include 'param.h'
      parameter(ndim = 3)
      integer sz(sz_0:sz_1,ndim,1)
      integer mg_li(mg_0:mg_1,ndim,1)
      integer proc(p_0:p_1), pgm(*)

      dimension rho(1,1,1), phi(1,1,1),wrk(1)
      dimension x(1),y(1),z(1)

*.... NOTE: eventually this can be eliminated, since .*
*........   one can write a smarter Multigrid ........*
*........   and use only the wrk array ...............*
      dimension r(1,1,1)

*............. Local Variables ...............*
      parameter( maxit = 200 ) 
      data one /1.d0/
      integer gdim(3), lsize, iter
      logical check_conv, printinfo

      external bc_self_grav
*.............................................*
*.... Get the various array sizes ......*
      myid = proc(p_id)

      check_conv = do_par( 6)
      printinfo  = do_par( 4)

      call AL_get_global_dim(gdim,sz,3)
      nx = gdim(1)
      ny = gdim(2)
      nz = gdim(3)
      call AL_get_array_bounds(bx, ex, bxgp, exgp, 'x', sz)
      call AL_get_array_bounds(by, ey, bygp, eygp, 'y', sz)
      call AL_get_array_bounds(bz, ez, bzgp, ezgp, 'z', sz)
      call AL_get_array_tsize(lsize,sz,ndim)

      pi  = 4.d0*atan(one)
      pi2 = 2.d0*pi
      pi4 = 4.d0*pi

*.... if TOLERANCE is < 1, then we use it as an actual error control ...*
*.....if TOLERANCE is > 1, then we use it as an iteration counter and ..*
*.....                     we skip error control                   .....*
      TOLERANCE = step_par(20)
      if( TOLERANCE .gt. 1. ) then
          check_conv = .false.
          iter = TOLERANCE
      else
          check_conv = .true.
          iter = maxit
      endif

*.... Compute the Average density (Needed in periodic geometry) ..
      rho_av = AL_get_average(rho,sz,ndim,proc,MPI_COMM_WORLD)

*.... Compute the source term for the poisson equation ....*
      call self_grav_rhs(r,rho,rho_av,  bx,ex,bxgp,exgp,
     &                   by,ey,bygp,eygp, bz,ez,bzgp,ezgp )

*>>>>>>>>>>>>>>>> BEGIN ITERATION >>>>>>>>>>>>>>>>>>>>>>>>*
      Do iter = 1, iter

*.... This is done for the error check .....*
*.... Save the OLD PHI .....................*
      if( check_conv ) 
     &   call AL_copy_array(sz,ndim,phi,wrk(1+3*lsize))

*.... Solve the 3D poission equation .......*
      call poisson_3d(sz,mg_li,proc,pgm, bc_self_grav,
     &               phi, r, x, y, z, wrk,  
     &               step_par, grid_par, fluid_par, do_par )

*.... This is done for the error check .....*
      if( check_conv ) then
          err = poisson_3d_err(sz,proc, phi,wrk(1+3*lsize), 
     &      bx,ex,bxgp,exgp, by,ey,bygp,eygp, bz,ez,bzgp,ezgp )

        if( ERR .le. TOLERANCE ) goto 1000
      endif
*>>>>>>>>>>>>>>>> END ITERATION >>>>>>>>>>>>>>>>>>>>>>>>*
      End Do

1000  continue
      
      if( myid .eq. 0 .and. printinfo ) then      
          print*,'Poisson converged in ',iter,'   iterations'
      endif

      return
      end

*********************************************************
*
*   SUBROUTINE SELF_GRAV_RHS
*
*   Set up the RHS for Poisson's equation.
*
      subroutine self_grav_rhs(r,rho,rho_av,  
     &      bx,ex,bxgp,exgp, by,ey,bygp,eygp, bz,ez,bzgp,ezgp )

      include 'meshf.h'
      dimension rho(bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)
      dimension r  (bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)

*............. Local Variables ...............*
      data one /1.d0/
*.............................................*
      
      pi  = 4.d0*atan(one)
      pi4 = 4.d0*pi
            
           Do k = bz, ez
           Do j = by, ey
           Do i = bx, ex
                r(i,j,k) = pi4*(rho(i,j,k) - rho_av)
           End Do
           End Do
           End Do
 
      Return
      End

*********************************************************
*
*   FUNCTION POISSON_3D_ERR
*
*   Compute the relative error during one cycle
*
*********************************************************
      function poisson_3d_err(sz, proc, phi, phi_o,
     &    bx,ex,bxgp,exgp, by,ey,bygp,eygp, bz,ez,bzgp,ezgp )
      include 'meshf.h'
      parameter(ndim = 3)
      integer sz(sz_0:sz_1,ndim)
      integer proc(p_0:p_1), pgm(1)
      dimension phi(bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)
      dimension phi_o(bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)

      real*8 er(1), w(1)
           err = 0.d0
           Do k = bz, ez
           Do j = by, ey
           Do i = bx, ex
               err = max(err,abs(phi(i,j,k)-phi_o(i,j,k)))
           End Do
           End Do
           End Do

           er(1) = err
c          call PIgdmax(er(1),1,w(1),ALLPROC)
           call MPI_ALLREDUCE(er(1),w(1),1,MESH_PRECISION,MPI_MAX,
     &               MPI_COMM_WORLD,ierr)
           err = w(1)

           poisson_3d_err = err        
      Return
      End

*********************************************************
*
*   SUBROUTINE POISSON_3D
*
*   Solve a 3D Poisson equation on a grid.
*
*   We consider now only uniform grids, but this can
*   change easyliy in the future.
*
**********************************************************
      subroutine poisson_3d(sz,mg_li,proc,pgm, bc_poisson,
     &                      phi, r, x, y, z, w,  
     &                      step_par, grid_par, fluid_par, do_par )
      include 'meshf.h'
      include 'param.h'
      parameter(ndim = 3)

      integer sz(sz_0:sz_1,ndim,1)
      integer mg_li(mg_0:mg_1,ndim,1)
      integer proc(p_0:p_1), pgm(*)

      dimension phi(1,1,1),w(1)
      dimension x(1),y(1),z(1)

*.... NOTE: eventually this can be eliminated, since .*
*........   one can write a smarter Multigrid ........*
*........   and use only the wrk array ...............*
      dimension r(1,1,1)

*............. Local Variables ...............*
      integer nr(max_lev+1), ns(max_lev+1)
      integer bx1,ex1,bxgp1,exgp1, by1,ey1,bygp1,eygp1
      integer bz1,ez1,bzgp1,ezgp1
      integer gdim(3)

      logical do_collect, do_redistribute

      data one /1.d0/

      external bc_poisson

*.............................................*
      nproc = proc(p_np)
      n_levels  = mg_li(mg_nlev,1,1)
      n_plevels = mg_li(mg_plev,1,1)

*.... Check if this a single processor run ....*
      if( nproc .eq. 1 ) then
          nlev = n_levels
      else
          nlev = n_levels+1
      endif

c      nlev = 2

*.... This is only temporary: set the number of iterations .....*
      Do ilev = 1, nlev-1
         Do nd = 1, ndim
            mg_li(mg_iter,nd,ilev) = 2
         End Do
      End Do
      
      Do nd = 1, ndim
           mg_li(mg_iter,nd,nlev) = 50
      End Do


*.... Get the various array sizes ......*
      call AL_get_global_dim(gdim,sz,3)
      nx = gdim(1)
      ny = gdim(2)
      nz = gdim(3)
      
*.... Get offsets for the work array ..........*
*..... nphi() is the offset for the solution ..*
*..... nr()   is the offset for the residual...*
      call mg_get_wrk_offs(ns,sz,mg_li,ndim,MG_LINEAR)
      Do n = 1, nlev+1
           nr(n) = ns(n) + ns(nlev+1)
      End Do

*.... offset for the global buffer .....*
      ngbuf = nr(nlev+1)

*.... Clean the work array ..............* IMPORTANT !      
      call AL_zero_array(w,1,nr(nlev+1)-1)

*>>>>>>>>>>>>> GET DX DY DZ (NOT THE MOST REFINED, BUT IT WORKS)>>>>>>* 
*.... Array size at the upper level ......*
      ilev = 1
      call AL_get_array_bounds(bx, ex, bxgp, exgp, 'x', sz(sz_0,1,ilev))
      call AL_get_array_bounds(by, ey, bygp, eygp, 'y', sz(sz_0,1,ilev))
      call AL_get_array_bounds(bz, ez, bzgp, ezgp, 'z', sz(sz_0,1,ilev))

*.... This is only for DX, DY, DZ constant .....(THIS PATHETIC, I KNOW)...*
      dx = x((ex+exgp-bx+bxgp+1)*(gr_dx+1))
      dy = y((ey+eygp-by+bygp+1)*(gr_dx+1))
      dz = z((ez+ezgp-bz+bzgp+1)*(gr_dx+1))
*>>>>>>>>>>>>> GOT DX DY DZ (NOT THE MOST REFINED, BUT IT WORKS)>>>>>>* 

*>>>> Begin DESCENDING LOOP >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
      Do ilev = 1, nlev

*.... Find out if this is the end of the parallel levels
*.... Time to COLLECT !
      do_collect = (ilev .eq. n_plevels .and. nproc .ne. 1)

*.... Offsets for this level .....*
*.... The conventions are:  ns_  SOLUTION
*....                       nr_  RESIDUAL
*....                         _u UP   (i.e. upper level)
*....                         _d DOWN (i.e. lower level)
      ns_u = ns(ilev)
      ns_d = ns(ilev+1)
      nr_u = nr(ilev)
      nr_d = nr(ilev+1)

*.... Array size at the upper level ......*
      call AL_get_array_bounds(bx, ex, bxgp, exgp, 'x', sz(sz_0,1,ilev))
      call AL_get_array_bounds(by, ey, bygp, eygp, 'y', sz(sz_0,1,ilev))
      call AL_get_array_bounds(bz, ez, bzgp, ezgp, 'z', sz(sz_0,1,ilev))

*.... BEGIN if do_collect .....*
*.... We do the regular MG cycle if the following is true
      if( .not. do_collect ) then

*.... Gauss--Seidel Iteration .....................*
      if( ilev .eq. 1 ) then
      call poisson_3d_relax_gs(sz(sz_0,1,ilev),mg_li(mg_0,1,ilev),
     &                         proc, pgm(ilev), bc_poisson,
     &                         bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         phi, r, dx,dy,dz,
     &                         step_par, grid_par, fluid_par )
*..... GO BACK IF N_LEVELS = 1.....*
      if( nlev .eq. 1 ) return
      else

      call poisson_3d_relax_gs(sz(sz_0,1,ilev),mg_li(mg_0,1,ilev),
     &                         proc, pgm(ilev), bc_poisson,
     &                         bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         w(ns_u), w(nr_u),  dx,dy,dz,
     &                         step_par, grid_par, fluid_par )
      endif
*... End Gauss-Seidel Iteration .....................*

*.....Begin INJECTION: ONLY IF THIS IS NOT THE LOWEST LEVEL....*
      if( ilev .ne. nlev ) then
*.... Array size at the lower level ......*
      il1 = ilev+1
      call AL_get_array_bounds(bx1,ex1,bxgp1,exgp1,'x',sz(sz_0,1,il1))
      call AL_get_array_bounds(by1,ey1,bygp1,eygp1,'y',sz(sz_0,1,il1))
      call AL_get_array_bounds(bz1,ez1,bzgp1,ezgp1,'z',sz(sz_0,1,il1))
 
*.... Injection of the residual ..........*
      if( ilev .eq. 1) then
      call poisson_3d_inject_r(sz(sz_0,1,ilev),mg_li(mg_0,1,ilev),
     &                         proc, pgm(ilev), bc_poisson,
     &                         bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         bx1, ex1, bxgp1, exgp1,
     &                         by1, ey1, bygp1, eygp1,
     &                         bz1, ez1, bzgp1, ezgp1,
     &                         phi, r, w(nr_d),   dx,dy,dz,
     &                         step_par, grid_par, fluid_par )
      else
      call poisson_3d_inject_r(sz(sz_0,1,ilev),mg_li(mg_0,1,ilev),
     &                         proc, pgm(ilev), bc_poisson,
     &                         bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         bx1, ex1, bxgp1, exgp1,
     &                         by1, ey1, bygp1, eygp1,
     &                         bz1, ez1, bzgp1, ezgp1,
     &                         w(ns_u), w(nr_u), w(nr_d), dx,dy,dz,
     &                         step_par, grid_par, fluid_par )
      endif

      endif
*.....End INJECTION: ONLY IF THIS IS NOT THE LOWEST LEVEL....*

*.... else do_collect
*.... We do a COLLECT otherwise .............................*
      else

*.... Array size at the lower level ......*
      il1 = ilev+1
      call AL_get_array_bounds(bx1,ex1,bxgp1,exgp1,'x',sz(sz_0,1,il1))
      call AL_get_array_bounds(by1,ey1,bygp1,eygp1,'y',sz(sz_0,1,il1))
      call AL_get_array_bounds(bz1,ez1,bzgp1,ezgp1,'z',sz(sz_0,1,il1))
 
      call poisson_3d_collect(proc, mg_li(mg_0,1,ilev), 
     &                         bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         bx1, ex1, bxgp1, exgp1,
     &                         by1, ey1, bygp1, eygp1,
     &                         bz1, ez1, bzgp1, ezgp1,
     &                         w(ns_u),w(ns_d),
     &                         w(nr_u),w(nr_d),w(ns_d),w(ngbuf))

*.... END if do_collect ........................................*
      endif

      End Do
*>>>> END DESCENDING LOOP >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*

*>>>> Begin ASCENDING LOOP >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
      Do ilev = nlev-1, 1, -1

*.... Check if we have to do a redistribution from local to
*.... distributed array ....................................
      do_redistribute = ( ilev .eq. n_plevels .and. nproc .ne. 1 )

*.... Offsets for this level .....*
*.... Offsets for this level .....*
*.... The conventions are:  ns_  SOLUTION
*....                       nr_  RESIDUAL
*....                         _u UP   (i.e. upper level)
*....                         _d DOWN (i.e. lower level)
      ns_u = ns(ilev)
      ns_d = ns(ilev+1)
      nr_u = nr(ilev)
      nr_d = nr(ilev+1)

*.... Array size at the upper level ......*
      call AL_get_array_bounds(bx, ex, bxgp, exgp, 'x', sz(sz_0,1,ilev))
      call AL_get_array_bounds(by, ey, bygp, eygp, 'y', sz(sz_0,1,ilev))
      call AL_get_array_bounds(bz, ez, bzgp, ezgp, 'z', sz(sz_0,1,ilev))

*.... Array size at the lower level ......*
      il1 = ilev+1
      call AL_get_array_bounds(bx1,ex1,bxgp1,exgp1,'x',sz(sz_0,1,il1))
      call AL_get_array_bounds(by1,ey1,bygp1,eygp1,'y',sz(sz_0,1,il1))
      call AL_get_array_bounds(bz1,ez1,bzgp1,ezgp1,'z',sz(sz_0,1,il1))

*.... BEGIN if do_redistribute .....*
*.... We do the regular MG cycle if the following is true
      if( .not. do_redistribute ) then

*.....Begin Prolongation ............................................* 
*.... Injection of the residual .....*
      if(ilev .eq. 1 ) then
      call poisson_3d_prolong( bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         bx1, ex1, bxgp1, exgp1,
     &                         by1, ey1, bygp1, eygp1,
     &                         bz1, ez1, bzgp1, ezgp1,
     &                         phi, w(ns_d) )
      else
      call poisson_3d_prolong( bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         bx1, ex1, bxgp1, exgp1,
     &                         by1, ey1, bygp1, eygp1,
     &                         bz1, ez1, bzgp1, ezgp1,
     &                         w(ns_u), w(ns_d) )
      endif
*.....End Prolongation ............................................* 

*.... Gauss--Seidel Iteration .....*
      if( ilev .eq. 1 ) then
      call poisson_3d_relax_gs(sz(sz_0,1,ilev),mg_li(mg_0,1,ilev),
     &                         proc, pgm(ilev), bc_poisson,
     &                         bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         phi, r,  dx,dy,dz,
     &                         step_par, grid_par, fluid_par )
      else
      call poisson_3d_relax_gs(sz(sz_0,1,ilev),mg_li(mg_0,1,ilev),
     &                         proc, pgm(ilev), bc_poisson,
     &                         bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         w(ns_u), w(nr_u),  dx,dy,dz,
     &                         step_par, grid_par, fluid_par )
      endif

*.... else do_redistribute 
*.... We do a REDISTRIBUTE otherwise .............................*
      else
      call poisson_3d_redistribute( bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         bx1, ex1, bxgp1, exgp1,
     &                         by1, ey1, bygp1, eygp1,
     &                         bz1, ez1, bzgp1, ezgp1,
     &                         w(ns_u), w(ns_d),
     &                         w(nr_u), w(nr_d)  )

*.... END if do_redistribute ........................................*
      endif

      End Do
*>>>> END ASCENDING LOOP >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
 
      Return
      End

*********************************************************
*
*   SUBROUTINE POISSON_3D_RELAX_GS
*
*   Apply Gauss-Seidel relaxation with Red and Black 
*   ordering.
*
*   We consider now only uniform grids, but this can
*   change easyliy in the future.
*
**********************************************************

       subroutine poisson_3d_relax_gs(sz,mg,proc, pgm,bc_poisson, 
     &                         bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         phi,r, dx1,dy1,dz1,
     &                         step_par, grid_par, fluid_par )
      include 'meshf.h'
c      include 'grid.h'
      include 'param.h'
      parameter(ndim = 3)
      integer sz(sz_0:sz_1,ndim)
      integer mg(mg_0:mg_1,ndim)
      integer proc(p_0:p_1), pgm(1)
      integer gp

      dimension phi(bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)
      dimension r  (bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)

      external bc_poisson

*............. Local Variables ...............*
      data half, one /0.5d0, 1.d0/
      integer xof
*.............................................*

      niter = mg(mg_iter,1)

      ilev  = mg(mg_mlev,1)

*..... (just in case some machines do not behave) ...*

      if(ilev .eq. 1) then
           scale_lev = 1.d0
      else
           scale_lev  = 2.d0**(ilev-1)
      endif

*.... Here we only consider constant grids .....*

      dx = dx1*scale_lev
      dy = dy1*scale_lev
      dz = dz1*scale_lev

      dx2 = dx*dx
      dy2 = dy*dy
      dz2 = dz*dz

      a1 = 1.d0/(2.d0/dx2+2.d0/dy2+2.d0/dz2)

      ai = a1/dx2
      aj = a1/dy2
      ak = a1/dz2
   
      ar = -a1

*>>>>>> Begin GS iterations >>>>>>>>>>*
      Do ns = 1, niter

*>>>>>> Begin RED-BLACK iterations >>>>>>>>>>*
      Do irb = 1, 2

*.... Apply Boundary Conditions before relaxing .........*
       call bc_poisson(sz,mg,proc,pgm, phi, dx,dy,dz,
     &         bx,ex,bxgp,exgp, by,ey,bygp,eygp, bz,ez,bzgp,ezgp,
     &                 grid_par )

*>>>>>> Begin one sweep >>>>>>>>>>*
          Do k = bz, ez
               xof = mod(k+irb,2)
          Do j = by, ey

          Do i = bx+xof, ex, 2
               rr       = r(i,j,k)
               phi_i0jk = phi(i-1,j,k)
               phi_i1jk = phi(i+1,j,k)
               phi_ij0k = phi(i,j-1,k)
               phi_ij1k = phi(i,j+1,k)
               phi_ijk0 = phi(i,j,k-1)
               phi_ijk1 = phi(i,j,k+1)
               dphi_i   = (phi_i1jk + phi_i0jk)
               dphi_j   = (phi_ij1k + phi_ij0k)
               dphi_k   = (phi_ijk1 + phi_ijk0)

               phi(i,j,k) = ar*rr + ai*dphi_i + aj*dphi_j + ak*dphi_k

*>>>>>> End one sweep >>>>>>>>>>*
          End Do
          End Do
          End Do

*>>>>>> End RED-BLACK iterations >>>>>>>>>>*
      End Do

*>>>>>> End GS iterations >>>>>>>>>>*
      End Do

*>>>>>> Apply Boundary Conditions before exiting >>>>>>>>>*
       call bc_poisson(sz,mg,proc,pgm, phi, dx,dy,dz,
     &         bx,ex,bxgp,exgp, by,ey,bygp,eygp, bz,ez,bzgp,ezgp,
     &                   grid_par )

      Return
      End

*********************************************************

*/*F
*********************************************************
*
*   SUBROUTINE POISSON_3D_INJECT_R
*
*   Inject the residual at the lower level
*   We consider now only uniform grids, but this can
*   change easyliy in the future.
*
**********************************************************
*F*/
       subroutine poisson_3d_inject_r(sz,mg,proc, pgm,
     &                                bc_poisson, 
     &                                bx, ex, bxgp, exgp,
     &                                by, ey, bygp, eygp,
     &                                bz, ez, bzgp, ezgp,
     &                                bx1, ex1, bxgp1, exgp1,
     &                                by1, ey1, bygp1, eygp1,
     &                                bz1, ez1, bzgp1, ezgp1,
     &                                phi,r, rs,  dx1,dy1,dz1,
     &                                step_par, grid_par, fluid_par )
      include 'meshf.h'
c      include 'grid.h'
      include 'param.h'
      parameter(ndim = 3)
      integer sz(sz_0:sz_1,ndim)
      integer mg(mg_0:mg_1,ndim)
      integer proc(p_0:p_1), pgm(1)

      integer bx1,ex1,bxgp1,exgp1,by1,ey1,bygp1,eygp1
      integer bz1,ez1,bzgp1,ezgp1

      dimension phi(bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)
      dimension r  (bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)
      dimension 
     & rs(bx1-bxgp1:ex1+exgp1,by1-bygp1:ey1+eygp1,bz1-bzgp1:ez1+ezgp1)

      external bc_poisson

*............. Local Variables ...............*
      data half, one /0.5d0, 1.d0/
      integer xof

*.... These indexes define a cube around a point on the coarse mesh ..*
      integer io(8), jo(8), ko(8)
      data io /-1, 0, 0,-1,-1, 0, 0,-1/
      data jo /-1,-1, 0, 0,-1,-1, 0, 0/
      data ko /-1,-1,-1,-1, 0, 0, 0, 0/

*.............................................*

      ilev  = mg(mg_mlev,1)

*..... (just in case some machines do not behave) ...*

      if(ilev .eq. 1) then
           scale_lev = 1.d0
      else
           scale_lev  = 2.d0**(ilev-1)
      endif

*.... Here we only consider constant grids .....*

      dx = dx1*scale_lev
      dy = dy1*scale_lev
      dz = dz1*scale_lev

      dx2 = dx*dx
      dy2 = dy*dy
      dz2 = dz*dz

      a1 = (2.d0/dx2+2.d0/dy2+2.d0/dz2)

      ai = 1.d0/dx2
      aj = 1.d0/dy2
      ak = 1.d0/dz2
   
*.... This may not be optimal, but it saves my sanity .....*
      Do k1 = bz1, ez1
         kk = 2*k1
         Do j1 = by1, ey1
            jj = 2*j1
            Do i1 = bx1, ex1
               ii = 2*i1

               rstemp = 0.d0

               Do n = 1, 8

                  k = kk + ko(n)
                  j = jj + jo(n)
                  i = ii + io(n)
 
                  rr     = r(i,j,k)

                  p_i0jk = phi(i-1,j,k)
                  p_i1jk = phi(i+1,j,k)
                  p_ij0k = phi(i,j-1,k)
                  p_ij1k = phi(i,j+1,k)
                  p_ijk0 = phi(i,j,k-1)
                  p_ijk1 = phi(i,j,k+1)
 
                  p_i   = (p_i1jk + p_i0jk)
                  p_j   = (p_ij1k + p_ij0k)
                  p_k   = (p_ijk1 + p_ijk0)

                  rstemp = rstemp + rr
     &           - ai*p_i - aj*p_j - ak*p_k + a1*phi(i,j,k)

               End Do

               rs(i1,j1,k1) = rstemp*0.125d0

            End Do
         End Do
      End Do

      return
      end

*/*F
*********************************************************
*
*   SUBROUTINE POISSON_3D_PROLONG
*
*   Prolong from the coarse to the refined mesh
*   using simple bilinear interpolation.
*
*   We consider now only uniform grids, but this can
*   change in the future.
*
**********************************************************
*F*/
      subroutine poisson_3d_prolong( bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         bx1, ex1, bxgp1, exgp1,
     &                         by1, ey1, bygp1, eygp1,
     &                         bz1, ez1, bzgp1, ezgp1,
     &                         pu, pd)

      include 'meshf.h'

      integer bx1,ex1,bxgp1,exgp1,by1,ey1,bygp1,eygp1
      integer bz1,ez1,bzgp1,ezgp1

      dimension pu(bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)
      dimension 
     & pd(bx1-bxgp1:ex1+exgp1,by1-bygp1:ey1+eygp1,bz1-bzgp1:ez1+ezgp1)


*..... Example of convention .................*
*..... i1 j1 k1 => i*2-1  j*2-1 k*2-1
*..... i2 j1 k1 => i*2  j*2-1 k*2-1
*..............................................*
 
      c0 = 27.d0/64.d0
      c1 =  9.d0/64.d0
      c2 =  3.d0/64.d0
      c3 =  1.d0/64.d0

      Do k1 = bz1, ez1
         kk = 2*k1
         Do j1 = by1, ey1
            jj = 2*j1
            Do i1 = bx1, ex1
               ii = 2*i1

               a0 = c0*pd(i1,j1,k1)

*..............

               pu(ii-1,jj-1,kk-1) = pu(ii-1,jj-1,kk-1) + a0 
     &  +  c1*(pd(i1-1,j1  ,k1  )+pd(i1  ,j1-1,k1  )+pd(i1,j1  ,k1-1))
     &  +  c2*(pd(i1-1,j1-1,k1  )+pd(i1-1,j1  ,k1-1)+pd(i1,j1-1,k1-1))
     &  +  c3*(pd(i1-1,j1-1,k1-1))

*..............

               pu(ii-1,jj-1,kk  ) = pu(ii-1,jj-1,kk  ) + a0 
     &  +  c1*(pd(i1-1,j1  ,k1  )+pd(i1  ,j1-1,k1  )+pd(i1,j1  ,k1+1))
     &  +  c2*(pd(i1-1,j1-1,k1  )+pd(i1-1,j1  ,k1+1)+pd(i1,j1-1,k1+1))
     &  +  c3*(pd(i1-1,j1-1,k1+1))    

               pu(ii-1,jj  ,kk-1) = pu(ii-1,jj  ,kk-1) + a0 
     &  +  c1*(pd(i1-1,j1  ,k1  )+pd(i1  ,j1+1,k1  )+pd(i1,j1  ,k1-1))
     &  +  c2*(pd(i1-1,j1+1,k1  )+pd(i1-1,j1  ,k1-1)+pd(i1,j1+1,k1-1))
     &  +  c3*(pd(i1-1,j1+1,k1-1))

               pu(ii  ,jj-1,kk-1) = pu(ii  ,jj-1,kk-1) + a0 
     &  +  c1*(pd(i1+1,j1  ,k1  )+pd(i1  ,j1-1,k1  )+pd(i1,j1  ,k1-1))
     &  +  c2*(pd(i1+1,j1-1,k1  )+pd(i1+1,j1  ,k1-1)+pd(i1,j1-1,k1-1))
     &  +  c3*(pd(i1+1,j1-1,k1-1))    

*...............

               pu(ii  ,jj  ,kk-1) = pu(ii  ,jj  ,kk-1) + a0 
     &  +  c1*(pd(i1+1,j1  ,k1  )+pd(i1  ,j1+1,k1  )+pd(i1,j1  ,k1-1))
     &  +  c2*(pd(i1+1,j1+1,k1  )+pd(i1+1,j1  ,k1-1)+pd(i1,j1+1,k1-1))
     &  +  c3*(pd(i1+1,j1+1,k1-1))    
 
               pu(ii  ,jj-1,kk  ) = pu(ii  ,jj-1,kk  ) + a0 
     &  +  c1*(pd(i1+1,j1  ,k1  )+pd(i1  ,j1-1,k1  )+pd(i1,j1  ,k1+1))
     &  +  c2*(pd(i1+1,j1-1,k1  )+pd(i1+1,j1  ,k1+1)+pd(i1,j1-1,k1+1))
     &  +  c3*(pd(i1+1,j1-1,k1+1))    

               pu(ii-1,jj  ,kk  ) = pu(ii-1,jj  ,kk  ) + a0 
     &  +  c1*(pd(i1-1,j1  ,k1  )+pd(i1  ,j1+1,k1  )+pd(i1,j1  ,k1+1))
     &  +  c2*(pd(i1-1,j1+1,k1  )+pd(i1-1,j1  ,k1-1)+pd(i1,j1+1,k1+1))
     &  +  c3*(pd(i1-1,j1+1,k1+1))    
 
*..............

               pu(ii  ,jj  ,kk  ) = pu(ii  ,jj  ,kk  ) + a0 
     &  +  c1*(pd(i1+1,j1  ,k1  )+pd(i1  ,j1+1,k1  )+pd(i1,j1  ,k1+1))
     &  +  c2*(pd(i1+1,j1+1,k1  )+pd(i1+1,j1  ,k1-1)+pd(i1,j1+1,k1+1))
     &  +  c3*(pd(i1+1,j1+1,k1+1))    

*..............
 
            End Do
         End Do
      End Do
      


      return
      end

*/*F
*********************************************************
*
*   SUBROUTINE POISSON_3D_COLLECT
*
*   Collect the RESIDUAL
*
*   Collect the distributed array elements of the last
*   parallel level into a local array.
*
*   r    is the DISTRIBUTED array
*   rloc is the LOCAL array
*
**********************************************************
*F*/

      subroutine poisson_3d_collect(proc,mg, 
     &                         bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         bx1, ex1, bxgp1, exgp1,
     &                         by1, ey1, bygp1, eygp1,
     &                         bz1, ez1, bzgp1, ezgp1,
     &                         s, sloc, 
     &                         r, rloc, buf, gbuf)

      include 'meshf.h'

      integer mg(mg_0:mg_1,3), proc(p_0:p_1)

      integer bx1,ex1,bxgp1,exgp1,by1,ey1,bygp1,eygp1
      integer bz1,ez1,bzgp1,ezgp1

      dimension r(bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)
      dimension 
     & rloc(bx1-bxgp1:ex1+exgp1,by1-bygp1:ey1+eygp1,bz1-bzgp1:ez1+ezgp1)
      dimension s(bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)
      dimension 
     & sloc(bx1-bxgp1:ex1+exgp1,by1-bygp1:ey1+eygp1,bz1-bzgp1:ez1+ezgp1)

      dimension buf(1), gbuf(1)

      parameter (MSG_DBL = 4)
      integer bsize, glocsize, gread
      integer bx2,ex2,by2,ey2,bz2,ez2

*.... Size of the local buffer    
      bsize = 2*(ex-bx+1)*(ey-by+1)*(ez-bz+1) + 6

      glocsize = 2*mg(mg_gbuf,1)

c     bsize = bsize*word
      bsize = bsize*1

c      print*,proc(p_id),'  COLL ',bsize,glocsize,ex,ey,ez

      glocsize = bsize*proc(p_np)+1

*.... Begin loading the buffer ....*

      m = 1
      buf(m) = bx
      m = m+1
      buf(m) = ex
      m = m+1
      buf(m) = by
      m = m+1
      buf(m) = ey
      m = m+1
      buf(m) = bz
      m = m+1
      buf(m) = ez

      Do k = bz, ez
         Do j = by, ey
            Do i = bx, ex
               m = m+1
               buf(m) = r(i,j,k)              
               m = m+1
               buf(m) = s(i,j,k)              
            End Do
         End Do
      End Do 

*... NOTE: word is defined in meshf.h ......*

c     call PIgcol(buf,bsize,gbuf,glocsize,gread,PSAllProcs,MSG_DBL)
      call MPI_ALLGATHER(buf,bsize,MESH_PRECISION,
     &                  gbuf,bsize,MESH_PRECISION,
     &                  MPI_COMM_WORLD,ierr)

      nproc = proc(p_np)
      if( gread .ne. bsize*nproc ) then
           print*,'Warning: GCOL may have failed'
      endif

*...  Now load the local array .....*

      m = 1

      Do n = 0, nproc-1

      bx2 = gbuf(m)
      m = m+1
      ex2 = gbuf(m)
      m = m+1
      by2 = gbuf(m)
      m = m+1
      ey2 = gbuf(m)
      m = m+1
      bz2 = gbuf(m)
      m = m+1
      ez2 = gbuf(m)

      Do k = bz2, ez2
         Do j = by2, ey2
            Do i = bx2, ex2
               m = m+1
               rloc(i,j,k) = gbuf(m)              
               m = m+1
               sloc(i,j,k) = gbuf(m)              
            End Do
         End Do
      End Do

      m = m+1

      End Do 

      return
      end

*/*F
*********************************************************
*
*   SUBROUTINE POISSON_3D_REDISTRIBUTE
*
*   Redistribute from the coarse (local) to 
*   the refined (distributed) mesh
*
**********************************************************
*F*/
      subroutine poisson_3d_redistribute( bx, ex, bxgp, exgp,
     &                         by, ey, bygp, eygp,
     &                         bz, ez, bzgp, ezgp,
     &                         bx1, ex1, bxgp1, exgp1,
     &                         by1, ey1, bygp1, eygp1,
     &                         bz1, ez1, bzgp1, ezgp1,
     &                         su, sd, ru, rd)

      include 'meshf.h'
      integer bx1,ex1,bxgp1,exgp1,by1,ey1,bygp1,eygp1
      integer bz1,ez1,bzgp1,ezgp1

      dimension ru(bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)
      dimension su(bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)
      dimension 
     & rd(bx1-bxgp1:ex1+exgp1,by1-bygp1:ey1+eygp1,bz1-bzgp1:ez1+ezgp1)
      dimension 
     & sd(bx1-bxgp1:ex1+exgp1,by1-bygp1:ey1+eygp1,bz1-bzgp1:ez1+ezgp1)

      Do k = bz-bzgp, ez+exgp
         Do j = by-bygp, ey+eygp
            Do i = bx-bxgp, ex+exgp
               ru(i,j,k) = rd(i,j,k)
               su(i,j,k) = sd(i,j,k)
            End Do
         End Do
      End Do

      return
      end

