
* INSERT HEADER HERE
*/*F
**************************************************
*                                                *
*  subroutine FLUID_3D                           *
*                                                *
*  The adiabatic compressible hydro code.        *
*                                                *
*  This interface is suitable for directional    *
*  splitting methods such as PPM.                *
*                                                *
*  At the moment, a polytropic equation of       *
*  state (EOS) is assumed. More general EOS's    *
*  can be considered.                            *
*                                                *
*    Author: Andrea Malagoli                     *
*    Date: 16 June 1995                          *
*                                                *
**************************************************
*F*/
      subroutine fluid_3d(sz,proc,pgm,
     &              ux, uy, uz, rho, press, temp, scalar, x, y, z,
     &              gravx, gravy, gravz, phi, src, wrk, 
     &              step_par, grid_par, fluid_par, do_par )

      include 'meshf.h'
      include 'param.h'

      parameter(ndim = 3)
      integer sz(sz_0:sz_1,ndim,1)
      integer proc(p_0:p_1), pgm(*)

      dimension ux(1), uy(1), uz(1), rho(1), press(1)
      dimension temp(1), scalar(1), wrk(1), src(1)
      dimension x(1), y(1), z(1)
      dimension gravx(1), gravy(1), gravz(1), phi(1)

*.................... Local variables ..........................*

      character*1 sweep_dir(3,0:5), swpd 
      integer     sweep_combinations
      data sweep_combinations  /6/
      data sweep_dir /"x","y","z",  "x","z","y", "z","x","y",
     &                "z","y","x",  "y","z","x", "y","x","z" / 

      character*7 tmpfiles(2)

*.... In f77 we need to copy an array into another because
*.... we cannot assign pointers. This will change in f90.
*.... For the moment, this is no big problem, since the 
*.... arrays x, y, z  are small.
      parameter (n1d = 512, ngp = 4, nmax = n1d+2*ngp)
      parameter (ngrid = (n1d)*(gr_1-gr_0+1))
      dimension  csi(ngrid), g(nmax*2)      

      integer b, bgp, e, egp
      integer b1, e1, b2, e2

      character*8 gridinfo(3)
      dimension grid1_par(3)

*...............................................................*
      myid = proc(p_id)
      
      nstep = step_par(1)

      call AL_get_array_bounds(bx,ex,bxgp,exgp,"x",sz)
      call AL_get_array_bounds(by,ey,bygp,eygp,"y",sz)
      call AL_get_array_bounds(bz,ez,bzgp,ezgp,"z",sz)

*.... Since we are considering operator splitting here we
*.... must alternate the order of the directional sweeps
*.... in order to avoid biases due to the order of directional
*.... sweeps.

*.... Here we do XYZ XZY ZXY ZYX YZX YXZ .....
*...... This is the sweep combination at this time step ..........*
      isweep_comb = mod(nstep,sweep_combinations)

*>>>>>>>>> START THE 3D SWEEP >>>>>>>>>>>>>>>>>>>>>>>>>*
      Do nswp = 1, 3

*...... This is the direction along which to sweep ..........*
      swpd = sweep_dir(nswp, isweep_comb)
  
*.... Set up the indexes for the 1D sweep and copy the .
*.... appropriate coordinates ..........................

*>>>>>>>>>>>>>> X >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
      if( swpd .eq. "x" ) then         

*...... Collect some grid info: geom, bc_left, bc_rigt

	 call GR_get_geom_name(gridinfo(1),grid_par(4))
	 call GR_get_bc_name  (gridinfo(2) ,grid_par(5))
	 call GR_get_bc_name  (gridinfo(3),grid_par(6))
 
         grid1_par(1) = grid_par(4)
         grid1_par(2) = grid_par(5)
         grid1_par(3) = grid_par(6)

         b = bx
         e = ex
         bgp = bxgp
         egp = exgp
         b1  = bz
         e1  = ez
         b2  = by
         e2  = ey         
         nd  = (e+egp-b+bgp+1) 

********** A Security check
            if( nd .gt. nmax .and. myid .eq. 0) then
               print*,'Error in fluid3d: nd > nmax ',nd,nmax
               print*,'Edit fluid3d.f and change nmax'
               call exit(1)
            endif

         Do i = 1, nd*(gr_1-gr_0+1)
            csi(i) = x(i) 
         End Do
         Do i = 1, nd*2
            g(i) = gravx(i)
         End Do
      endif

*>>>>>>>>>>>>>> Y >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
      if( swpd .eq. "y" ) then         

	 call GR_get_geom_name(gridinfo(1),grid_par(10))
	 call GR_get_bc_name  (gridinfo(2) ,grid_par(11))
	 call GR_get_bc_name  (gridinfo(3),grid_par(12))

         grid1_par(1) = grid_par(10)
         grid1_par(2) = grid_par(11)
         grid1_par(3) = grid_par(12)

         b = by
         e = ey
         bgp = bygp
         egp = eygp
         b1  = bz
         e1  = ez
         b2  = bx
         e2  = ex          
         nd  = (e+egp-b+bgp+1)

********** A Security check
            if( nd .gt. nmax .and. myid .eq. 0) then
               print*,'Error in fluid3d: nd > nmax ',nd,nmax
               print*,'Edit fluid3d.f and change nmax'
               call exit(1)
            endif

         Do i = 1, nd*(gr_1-gr_0+1)
            csi(i) = y(i) 
         End Do
         Do i = 1, nd*2
            g(i) = gravy(i)
         End Do
      endif

*>>>>>>>>>>>>>> Z >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
      if( swpd .eq. "z" ) then

	 call GR_get_geom_name(gridinfo(1),grid_par(16))
	 call GR_get_bc_name  (gridinfo(2) ,grid_par(17))
	 call GR_get_bc_name  (gridinfo(3),grid_par(18))

         grid1_par(1) = grid_par(16)
         grid1_par(2) = grid_par(17)
         grid1_par(3) = grid_par(18)

         b = bz
         e = ez
         bgp = bzgp
         egp = ezgp
         b1  = by
         e1  = ey
         b2  = bx
         e2  = ex          
         nd  = (e+egp-b+bgp+1)

********** A Security check
            if( nd .gt. nmax .and. myid .eq. 0) then
               print*,'Error in fluid3d: nd > nmax ',nd,nmax
               print*,'Edit fluid3d.f and change nmax'
               call exit(1)
            endif

         Do i = 1, nd*(gr_1-gr_0+1)
            csi(i) = z(i) 
         End Do
         Do i = 1, nd*2
            g(i) = gravz(i)
         End Do
      endif
*.................................................*
*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
*.... Compute the offsets for the work array ....*
      nd  = (e+egp-b+bgp+1)
      nd5 = nd*5
      nd8 = nd*8
      nwold   = 1
      nwnew   = nwold  + nd8
      nuold   = nwnew  + nd8
      nunew   = nuold  + nd8
      nusrc   = nunew  + nd8
      nwsrc   = nusrc  + nd5

c      nuttop  = nutbot + nd
c      ncourno = nuttop + nd

*.................................................*

*>>>>>>>>>>>>>>>>>>>APPLY BOUNDARY CONDITIONS>>>>>>>>>>>>>>>>>>>*

*.... This is where all the parallelism is taken into account .....

        Call BC_Fluid(sz,proc,pgm,
     &                bx,ex,bxgp,exgp,by,ey,bygp,eygp,bz,ez,bzgp,ezgp,
     &                ux, uy, uz, rho, press, temp, scalar, x, y, z,
     &                gravx, gravy, gravz, phi, src,
     &                step_par, grid_par, fluid_par , do_par, swpd)

*>>>>>>>>>>>>>>>>>>>BEGIN THE TIME STEP ADVANCE>>>>>>>>>>>>>>>>>>*
      Do k = b1, e1
      Do j = b2, e2

*******DEBUG***********************
ccc	 print*,'Sweep ',swpd,j,k
***********************************

*.... Copy old 1d variables from 3d arrays .......*
         call Threed_to_Oned(bx,ex,bxgp,exgp,by,ey,bygp,eygp,
     &              bz,ez,bzgp,ezgp,  b,e,bgp,egp,
     &              ux, uy, uz, rho, press, temp, scalar, phi, src,
     &              g,x,y,z,wrk(nwold),wrk(nuold),wrk(nwsrc),wrk(nusrc), 
     &              step_par, grid_par, fluid_par, do_par, 
     &              j, k, swpd)

*.... Update the 1d equations ...........*
         call Fluid_Solver(b, e, bgp, egp, g, csi,
     &              wrk(nwold),wrk(nuold),wrk(nwsrc),wrk(nusrc),
     &              wrk(nwnew), wrk(nunew),             
     &              step_par, grid1_par, fluid_par, do_par, 
     &              j, k, swpd)

*.... Replace new 1d variables into 3d arrays .......*
         call Oned_to_Threed(bx,ex,bxgp,exgp,by,ey,bygp,eygp,
     &              bz,ez,bzgp,ezgp,  b,e,bgp,egp,
     &              ux, uy, uz, rho, press, temp, scalar,
     &              g, csi, wrk(nwnew), wrk(nunew),
     &              step_par, grid_par, fluid_par, do_par, 
     &              j, k, swpd)

*>>>>>>>>>>>>>>>>>>>END THE TIME STEP ADVANCE>>>>>>>>>>>>>>>>>>*
      End Do
      End Do 

*>>>>>>>>> END OF  THE 3D SWEEP >>>>>>>>>>>>>>>>>>>>>>>>>*
      End Do

*>>>>>>>>> APPLY ARTIFICIAL VISCOSITY HERE >>>>>>>>>>>>>>>*
c      art_visc = step_par(11)
c      if( art_visc .gt. 0.0 ) then
c          Call Art_Visc( ............ )
c      endif

      return
      end

*---------------------------------------------------------------*
*/*F
*    SUBROUTINE  Threed_to_Oned
*
*    Get the 1D arrays required for PPM (or similar codes)
*    from the 3D arrays.
*
*    Author: Andrea Malagoli
*    Date: 16 June 1995
*
*F*/
      subroutine Threed_to_Oned(bx,ex,bxgp,exgp,by,ey,bygp,eygp,
     &              bz,ez,bzgp,ezgp,  b,e,bgp,egp,
     &              ux, uy, uz, rho, press, temp, scalar, phi, src,
     &              g, x,y,z , wold, uold, usrc, wsrc,             
     &              step_par, grid_par, fluid_par, do_par, 
     &              js, ks, sweepdir)
      include 'meshf.h'
      include '3d.h'
      include 'param.h'
      include 'grid.h'
      integer b,e,bgp,egp
      dimension phi(bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)
      dimension src(bx-bxgp:ex+exgp,by-bygp:ey+eygp,bz-bzgp:ez+ezgp)
      dimension g(b-bgp:e+egp,2)

      parameter ( nvar = 8 , nsrc = 5 )
      dimension wold(nvar,b-bgp:e+egp),uold(nvar,b-bgp:e+egp)
      dimension wsrc(nsrc,b-bgp:e+egp), usrc(nsrc,b-bgp:e+egp)

      character*1 sweepdir

*.......... Local Variables ...................*
      integer beg, end
      logical do_grav, do_src, do_scalar
      data zero, half, one /0.d0, 0.5d0, 1.d0/

*.... Tolerance for the artificial viscosity ...*
      parameter (DTOL = 1.d-5)
*..............................................*

      gamma      = fluid_par(4)
      one_gamma1 = 1.d0/(gamma-1.d0)

      beg = b-bgp
      end = e+egp

      do_grav = do_par(2)
      do_src  = do_par(5)
      do_scalar = do_par(7)
      cvisc = fluid_par(20)

*...... Initialize the source terms to zero ..........*
      Do i = beg, end
      Do j = 1, nsrc
            usrc(j,i) = zero
            wsrc(j,i) = zero
      End Do
      End Do
*.....................................................*

*........ Fill the 1D variables ............*

*>>>>>>>>>>>>>>>>>> BEGIN X >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
      if( sweepdir .eq. 'x' ) then

         iy = js
         iz = ks

         Do  ix = beg, end
            wold(1,ix) = rho  (ix,iy,iz)
            wold(2,ix) = ux   (ix,iy,iz)
            wold(3,ix) = uy   (ix,iy,iz)
            wold(4,ix) = uz   (ix,iy,iz)
            wold(5,ix) = press(ix,iy,iz)
	    wold(6,ix) = temp (ix,iy,iz)
         End do

         if( do_scalar ) then
            Do  ix = beg, end
               wold(8,ix) = scalar (ix,iy,iz)
            End do
         endif 

*>>>>>>>>>>>> ARTIFICIAL VISCOSITY X >>>>>>>>>>>>>>>>>>>>>>*
*.... This is new. It is used for artificial viscosity ....*
*.... We construct a finite difference approximation to ...*
*.... Div(U) at i+1/2 j k .................................*
         if( cvisc .ge. 1.d-5 ) then
            Do  ix = beg+1, end-1
                ixm = ix-1

                uztp = (uz(ix,iy,iz+1)+uz(ixm,iy,iz+1))*0.5d0
                uzbt = (uz(ix,iy,iz-1)+uz(ixm,iy,iz-1))*0.5d0
                uytp = (uy(ix,iy+1,iz)+uy(ixm,iy+1,iz))*0.5d0
                uybt = (uy(ix,iy-1,iz)+uy(ixm,iy-1,iz))*0.5d0
                uxtp  = ux(ix ,iy,iz)
                uxbt  = ux(ixm,iy,iz)

                dz = z(iz+1,gr_x)-z(iz-1,gr_x)
                dy = y(iy+1,gr_x)-y(iy-1,gr_x)
                dx = x(ix,gr_dx)
                dxdy = dx/dy
                dxdz = dx/dz

            divu = (uxtp-uxbt)+dxdy*(uytp-uybt)+dxdz*(uztp-uzbt)+DTOL

	        wold(7,ix) = cvisc*max(-divu,zero)

            End do
         endif
*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*

*....... Fill in the gravitational potential, if we are ..*
*....... doing self gravity ..............................*

         if( do_grav ) then
            Do  ix = beg, end
               g(ix,2) = phi(ix,iy,iz)
            End Do
            Do  ix = beg+1, end-1
               dx2_i = half/(x(ix,gr_xr)-x(ix,gr_xl))
               g(ix,1) = (g(ix-1,2)-g(ix+1,2))*dx2_i
            End Do
         endif

*....... Include the external non-geometrical sources ......*
CCC         if( do_src ) then
CCCCC     call External Sources( _
CCC         endif

*>>>>>>>>>>>>>>>>>> END X >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
      endif

*>>>>>>>>>>>>>>>>>> BEGIN Y >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
      if( sweepdir .eq. 'y' ) then

         ix = js
         iz = ks

         Do  iy = beg, end
            wold(1,iy) = rho  (ix,iy,iz)
            wold(2,iy) = uy   (ix,iy,iz)
            wold(3,iy) = uz   (ix,iy,iz)
            wold(4,iy) = ux   (ix,iy,iz)
            wold(5,iy) = press(ix,iy,iz)
	    wold(6,iy) = temp (ix,iy,iz)
         End do

         if( do_scalar ) then
            Do  iy = beg, end
               wold(8,iy) = scalar (ix,iy,iz)
            End do
         endif 

*>>>>>>>>>>>> ARTIFICIAL VISCOSITY  Y >>>>>>>>>>>>>>>>>>>>>*
*.... This is new. It is used for artificial viscosity ....*
*.... We construct a finite difference approximation to ...*
*.... Div(U) at i j+1/2 k .................................*
         if( cvisc .ge. 1.d-5 ) then
            Do  iy = beg+1, end-1
                iym = iy-1

                uztp = (uz(ix,iy,iz+1)+uz(ix,iym,iz+1))*0.5d0
                uzbt = (uz(ix,iy,iz-1)+uz(ix,iym,iz-1))*0.5d0
                uxtp = (ux(ix+1,iy,iz)+ux(ix+1,iym,iz))*0.5d0
                uxbt = (ux(ix-1,iy,iz)+ux(ix-1,iym,iz))*0.5d0
                uytp  = uy(ix,iy,iz)
                uybt  = uy(ix,iym,iz)

                dz = z(iz+1,gr_x)-z(iz-1,gr_x)
                dx = x(ix+1,gr_x)-x(ix-1,gr_x)
                dy = y(iy,gr_dx)
                dydx = dy/dx
                dydz = dy/dz

           divu = (uytp-uybt)+dydx*(uxtp-uxbt)+dydz*(uztp-uzbt)+DTOL

	        wold(7,iy) = cvisc*max(-divu,zero)

            End do
         endif
*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*

*....... Fill in the gravitational potential, if we are ..*
*....... doing self gravity ..............................*

         if( do_grav ) then
            Do  iy = beg, end
               g(iy,2) = phi(ix,iy,iz)
            End Do
            Do  iy = beg+1, end-1
               dy2_i = half/(y(iy,gr_xr)-y(iy,gr_xl))
               g(iy,1) = (g(iy-1,2)-g(iy+1,2))*dy2_i
            End Do
         endif

*....... Include the external non-geometrical sources ......*
CCC         if( do_src ) then
CCCCC     call External Sources( _
CCC         endif

*>>>>>>>>>>>>>>>>>> END Y >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
      endif

*>>>>>>>>>>>>>>>>>> BEGIN Z >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
      if( sweepdir .eq. 'z' ) then

         ix = js
         iy = ks

         Do  iz = beg, end
            wold(1,iz) = rho  (ix,iy,iz)
            wold(2,iz) = uz   (ix,iy,iz)
            wold(3,iz) = ux   (ix,iy,iz)
            wold(4,iz) = uy   (ix,iy,iz)
            wold(5,iz) = press(ix,iy,iz)
	    wold(6,iz) = temp (ix,iy,iz)
         End do

         if( do_scalar ) then
            Do  iz = beg, end
               wold(8,iz) = scalar (ix,iy,iz)
            End do
         endif 

*>>>>>>>>>>>> ARTIFICIAL VISCOSITY  Z >>>>>>>>>>>>>>>>>>>>>*
*.... This is new. It is used for artificial viscosity ....*
*.... We construct a finite difference approximation to ...*
*.... Div(U) at i j k+1/2 .................................*
         if( cvisc .ge. 1.d-5 ) then
            Do  iz = beg+1, end-1
                izm = iz-1

                uytp = (uy(ix,iy+1,iz)+uy(ix,iy+1,izm))*0.5d0
                uybt = (uy(ix,iy-1,iz)+uy(ix,iy-1,izm))*0.5d0
                uxtp = (ux(ix+1,iy,iz)+ux(ix+1,iy,izm))*0.5d0
                uxbt = (ux(ix-1,iy,iz)+ux(ix-1,iy,izm))*0.5d0
                uztp  = uz(ix,iy,iz)
                uzbt  = uz(ix,iy,izm)

                dy = y(iy+1,gr_x)-y(iy-1,gr_x)
                dx = x(ix+1,gr_x)-x(ix-1,gr_x)
                dz = z(iz,gr_dx)
                dzdy = dz/dy
                dzdx = dz/dx

           divu = (uztp-uzbt)+dzdx*(uxtp-uxbt)+dzdy*(uytp-uybt)+DTOL

	        wold(7,ix) = cvisc*max(-divu,zero)

            End do
         endif
*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*

*....... Fill in the gravitational potential, if we are ..*
*....... doing self gravity ..............................*

         if( do_grav ) then
            Do  iz = beg, end
               g(iz,2) = phi(ix,iy,iz)
            End Do
            Do  iz = beg+1, end-1
               dz2_i = half/(z(iz,gr_xr)-z(iz,gr_xl))
               g(iz,1) = (g(iz-1,2)-g(iz+1,2))*dz2_i
            End Do
         endif

      endif

*....... Include the external non-geometrical sources ......*
CCC         if( do_src ) then
CCCCC     call External Sources( _
CCC         endif
*>>>>>>>>>>>>>>>>>> END Z >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
      return
      end
*---------------------------------------------------------------*
*/*F
*    SUBROUTINE Threed_to_Oned 
*
*    Replace the 1D arrays from PPM (or similar codes)
*    into the 3D arrays.
*
*    Author: Andrea Malagoli
*    Date: 16 June 1995
*
*F*/
 
       subroutine Oned_to_Threed(bx,ex,bxgp,exgp,by,ey,bygp,eygp,
     &              bz,ez,bzgp,ezgp,  b,e,bgp,egp,
     &              ux, uy, uz, rho, press, temp,scalar,
     &              g, csi, wnew, unew,
     &              step_par, grid_par, fluid_par, do_par, 
     &              js, ks, sweepdir)
      include 'meshf.h'
      include '3d.h'
      include 'param.h'
      integer b,e,bgp,egp
      dimension g(b-bgp:e+egp,2), csi(b-bgp:e+egp,gr_0:gr_1)

      parameter ( nvar = 8 , nsrc = 5 )
      dimension wnew(nvar,b-bgp:e+egp), unew(nvar,b-bgp:e+egp)

      character*1 sweepdir
      
*.......... Local Variables ...................*
      integer beg, end
      logical do_scalar
      data zero, half, one /0.d0, 0.5d0, 1.d0/
*..............................................*

      beg = b
      end = e

      do_scalar = do_par(7)

*>>>>>>>>>>> X >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
      if( sweepdir .eq. 'x' ) then

         iy = js
         iz = ks

         Do  ix = beg, end
	    dens            = wnew(1,ix)
            rho  (ix,iy,iz) = dens
            ux   (ix,iy,iz) = wnew(2,ix)
            uy   (ix,iy,iz) = wnew(3,ix)
            uz   (ix,iy,iz) = wnew(4,ix)
            pre             = wnew(5,ix) 
            press(ix,iy,iz) = pre 
            temp (ix,iy,iz) = pre/dens
         End Do

        if( do_scalar ) then
          Do  ix = beg, end
              scalar(ix,iy,iz) = wnew(8,ix)
          End Do 
        endif

      endif

*>>>>>>>>>>> Y >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
      if( sweepdir .eq. 'y' ) then

         ix = js
         iz = ks

         Do  iy = beg, end
	    dens            = wnew(1,iy)
            rho  (ix,iy,iz) = dens
            uy   (ix,iy,iz) = wnew(2,iy)
            uz   (ix,iy,iz) = wnew(3,iy)
            ux   (ix,iy,iz) = wnew(4,iy)
            pre             = wnew(5,iy) 
            press(ix,iy,iz) = pre 
            temp (ix,iy,iz) = pre/dens
         End do

         if( do_scalar ) then
           Do  iy = beg, end
              scalar(ix,iy,iz) = wnew(8,iy)
           End Do 
         endif

      endif

*>>>>>>>>>>> Z >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>*
      if( sweepdir .eq. 'z' ) then

         ix = js
         iy = ks

         Do  iz = beg, end
	    dens            = wnew(1,iz)
            rho  (ix,iy,iz) = dens
            uz   (ix,iy,iz) = wnew(2,iz)
            ux   (ix,iy,iz) = wnew(3,iz)
            uy   (ix,iy,iz) = wnew(4,iz)
            pre             = wnew(5,iz) 
            press(ix,iy,iz) = pre 
            temp (ix,iy,iz) = pre/dens
         End do

        if( do_scalar ) then
            Do  iz = beg, end
              scalar(ix,iy,iz) = wnew(8,iz)
            End Do 
        endif

      endif

      return
      end
