/****************************************************************
*								*
*	THE COMPUTATIONAL ASTROPHYSICS GROUP			*
*	   at UNIVERSITY OF CHICAGO				*
*		    and the 					*
*	MCS DIVISION AT ARGONNE NATIONAL LABORATORY		*
*								*
*	Three-dimensional Higher-Order Godunov method 		*
*       for Compressible Hydrodynamics with nonlinear           *
*       thermal conduction and self-gravity.                    *
*								*
*	Version written using the PICL portable library		*
*	for use on massively parallel MIMD machines.		*
*       							*
*	Date: 31 May 1995        			        *
*       							*
*	Authors:  Andrea Malagoli		                *
*		  Department of Astronomy			*
*		  University of Chicago				*
*	          933 E.56th St.				*
*		  Chicago, IL 60637				*
*								*
*		  malagoli@liturchi.uchicago.edu		*
*       							*
****************************************************************/
/* Most local definitions are now contained in meshc.h */

/* The macro "array" is defined as either "float" or "double"
 * depending on the options chosen.
 */ 
#include "meshc.h"
#include <stdio.h>

#ifdef FORTRANUNDERSCORE
#define c_driver c_driver_
#define f_driver f_driver_
#endif

#ifdef HAS_CAP 
#define c_driver C_DRIVER
#define f_driver F_DRIVER
#endif

array *vector();
float *fvector();
double *dvector();
int *ivector();

/*..............................................................*/
/*C void c_driver(sz,mg_li,mg_nli,ndim,proc,pgm)
 *    
 *  The C driver for the 3D code 
 *  Given the decomposition arrays sz, mg, proc
 *  allocate the necessary arrays dynamically
 *
 C*/ 
void c_driver(sz,mg_li,mg_nli,ndim,proc,pgm)
ALArrayPart *sz; 
MGdesc *mg_li, *mg_nli;
ALPdesc *proc; 
int *pgm, *ndim;
{

  array *x, *y, *z;  /* The grid geometry arrays */
  array *gravx;       /* The gravity vector: it has the form */
                      /* grav(1:nx, 2) where the second index */
                      /* denotes either the gravitational force */
                      /* or the potential*/
  array *gravy, *gravz;

  array *ux,*uy,*uz,*rho,*press,*temp, *scalar;
  array *r,*wrk,*d,*phi;
  array *wppm;

  int size;
  int nl, nh, nb, nt;
  int nx, ny, nz, nx2, ny2, nz2, nxyz;
  int s, e, sg, eg;
  int igbuf, ilbuf;
  float tot_mem;
  
  char *x_dir="x", *z_dir="z", *y_dir="y";
  /*.....................................*/
  /* arrays in the X direction */
  AL_get_array_bounds(&s,&e,&sg,&eg,x_dir,sz);

  nx     = e-s+sg+eg+1;
  nx2    = nx*2;

  x = vector(1,nx*GR_ELEMENTS);
  gravx  = vector(1,nx2);

  tot_mem = (nx*GR_ELEMENTS + nx2)*sizeof(array);
/*.................................*/

/*.................................*/
/* arrays in the Y direction */
  AL_get_array_bounds(&s,&e,&sg,&eg,y_dir,sz);

  ny     = e-s+sg+eg+1;
  ny2    = ny*2;

  y = vector(1,ny*GR_ELEMENTS);
  gravy  = vector(1,ny2);

  tot_mem = tot_mem + (ny*GR_ELEMENTS + ny2)*sizeof(array);
/*.................................*/

/*.................................*/
/* arrays in the Z direction */
  AL_get_array_bounds(&s,&e,&sg,&eg,z_dir,sz);

  nz     = e-s+sg+eg+1;
  nz2    = nz*2;

  z = vector(1,nz*GR_ELEMENTS);
  gravz  = vector(1,nz2);

  tot_mem = tot_mem + (nz*GR_ELEMENTS + nz2)*sizeof(array);
/*.................................*/

/*.................................*/
/* Global 3D arrays */
/*  array *ux,*uy,*uz,*rho,*press,*temp; */
/*  array *r,*d,*phi      */
 
  nxyz = nx*ny*nz;
  
  ux    = vector(1,nxyz);
  uy    = vector(1,nxyz);
  uz    = vector(1,nxyz);
  rho   = vector(1,nxyz);
  press = vector(1,nxyz);
  temp  = vector(1,nxyz);
  r   = vector(1,nxyz);

  tot_mem = tot_mem + 7.*nxyz*sizeof(array);

#ifdef DO_PASSIVE_SCALAR
  scalar = vector(1,nxyz);
  tot_mem = tot_mem + nxyz*sizeof(array);
#endif

  
#ifdef DO_SELF_GRAVITY
  phi = vector(1,nxyz);
  tot_mem = tot_mem + nxyz*sizeof(array);
#endif


/*.................................*/

/*.................................*
 *  Work array for PPM  *
 *  Eventually, we can allocate this statically 
 */ 
  nh = nx > ny ? nx : ny;
  nh = nz > nh ? nz : nh;
 
  nt = 30*(nh);
  wppm  = vector(1,nt);

  tot_mem = tot_mem + 30.*nt*sizeof(array);
/*.................................*/

/*.................................*/
/* Work array for the multigrid */
  igbuf = mg_nli->mg_gbuf;
  ilbuf = mg_nli->mg_lbuf;


/*.... This adds extra safety factor .....*/
  nt = 4*( nxyz + igbuf );
 
  wrk = vector(1,nt);

  tot_mem = tot_mem + nt*sizeof(array);
 /*.................................*/

  tot_mem = tot_mem / 1.e6;
  /* printf("\n"); */
  printf("Total memory allocated on the node %6.2f MBytes\n",tot_mem);
  /* printf("\n"); */

 /*.................................*/
  f_driver(sz,mg_li,mg_nli,ndim,proc,pgm, 
		/**/ ux, uy, uz, rho, press, temp, scalar,
                     x, y, z,
                     gravx, gravy, gravz, phi,
                     r, wrk, wppm);

}

/*-------------------------------------------------------------*/
array *vector( nl,  nh)
int nl, nh;
{
        array *v;
        int n;
        n = nh-nl+1;
        v=(array *)malloc(n*sizeof(array));
        if( !v ) {
           printf("Allocation error in vector() %d\n",n*sizeof(array));
           exit(1); 
        }
        return v-nl;
}

float *fvector( nl,  nh)
int nl, nh;
{
        float *v;
        int n;
        n = nh-nl+1;
        v=(float *)malloc(n*sizeof(double));
        if( !v ) printf("Allocation error in fvector()\n");
        return v-nl;
}

double *dvector( nl,  nh)
int nl, nh;
{
        double *v;
        int n;
        n = nh-nl+1;
        v=(double *)malloc(n*sizeof(double));
        if( !v ) printf("Allocation error in dvector()\n");
        return v-nl;
}


int *ivector( nl,  nh)
int nl, nh;
{
        int *v,n;
        n = nh-nl+1;
        v=( int *)malloc(n*sizeof(int));
        if( !v ) printf("Allocation error in ivector()\n");
        return v-nl;
}

