
/* BNF glomming utility  by Guy L. Steele Jr.  March 16, 1993
   Updated for building cross-reference table---GLS 3/17/93
   Updated to ignore "..." tokens, new BNFcrossref* environment.--GLS 3/30/93
   Updated not to boldface "defined" entries in crossref.--GLS 5/5/93

   This hack is something like "cat" with a filter.  It takes any
   number of file names and copies selected portions to standard output.

   Usage: glombnf [ -noconstraints ] [ -crossref ] file1 file2 file3 ...

   It copies BNF information, delimited by lines containing \BNF abd \FNB.

   It copies constraints information, delimited by lines containing
   \begin{constraints} and \end{constraints}.  (However, the switch
   -noconstraints will suppress transcriptuion of constraints for all
   following files.)

   It copies lines containing LaTeX \chapter and \section commands,
   transforming them to \section and \subsection commands, respectively.
   However, it puts them in the output only when necessary to identify
   following BNF.  For example, if a chapter contains no BNF, then that
   \chapter command (transmuted to a \section command) is not sent to
   the output; instead, \stepcounter{section} is output to maintain
   the section numbering correctly.  Similarly for \section.

   It merges multiple blocks of BNF into a single BNF block when the
   multiple blocks are not interspersed with constraints or sectioning
   commands.

   It ignores commands and delimiters if "%" appears earlier in the line.
   Thus it "works" to comment out these commands in a LaTeX source file.

   The -crossref switch causes a second chapter to be generated containing
   cross-reference information.  The LaTeX command \Fortranrule may be
   used in a source file to flag nonterminals defined by Fortran 90.
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define BUFLEN 1000

char *current_file = "";

#define LOSE(why) { fprintf(stderr,"glombnf: %s: %s\n", current_file, why); exit(1); }

#define STARTBNF do { fprintf(stdout, "\t\t\t\t\t\t\t\t\t\\BNF\n");	\
		      bnf_output = 1; } while (0)
#define ENDBNF do { fprintf(stdout, "\t\t\t\t\t\t\t\t\t\\FNB\n");	\
		    bnf_output = 0; } while (0)

#define NUM_XREFS 6 /* number of cross-references per line */

char line_buffer[BUFLEN], chapter_buffer[BUFLEN], section_buffer[BUFLEN];

int chapter_pending, section_pending;
int within_bnf, within_constraints;
int bnf_output;
int noconstraints;
int crossref, crossref_bnf_rule_index;

/* "panic" means "present and not in comment".  I know it's lousy software
    engineering, but I give myself a quota of one bad joke per hack. */

char *panic(str, pat)
  char *str, *pat;
{
  char *comment = strchr(str, '%');
  char *found = strstr(str, pat);
  if (!comment || !found || comment > found) return found;
  return NULL;
}

#define MAXREFS 50
#define MAXITEMS 500
#define SEPARATORS " \t\n[]{}"
#define MAXSTRINGPOOL (50*MAXITEMS)

char *poolptr;

/* In each row, there is a name, a definition, and some number of refs. */

char *crossref_data[MAXITEMS][MAXREFS+2];
char stringpool[MAXITEMS*50];

void crossref_analyze(line)
  char *line;		/* The string pointed to is expendable. */
{
  int j;
  char *word = strtok(line, SEPARATORS);
  if (!word) crossref_bnf_rule_index = -1;
  else {
    for ( ; word; word = strtok(NULL, SEPARATORS)) {
      if (word[0] == '\\' && (crossref_bnf_rule_index < 0)) break;
      if (word[0] != '\\' && strcmp(word, "...")) {
	for (j = 0; ; j++) {
	  if (j == MAXITEMS-1) LOSE("too many BNF rules to crossref");
	  /* Strip the prefix "scalar-" and suffix "-list". */
	  if (!strncmp(word, "scalar-", 7)) word += 7;
	  if (!strcmp(word+strlen(word)-5, "-list")) word[strlen(word)-5] = 0;
	  /* If symbol not seen before, make an entry. */
	  if (!crossref_data[j][0]) {
	    if ((poolptr + strlen(word)) > (stringpool + MAXSTRINGPOOL))
	      LOSE("stringpool overflow");
	    strcpy(poolptr, word);
	    crossref_data[j][0] = poolptr;
	    poolptr += strlen(word)+1;
	  }
	  if (!strcmp(word, crossref_data[j][0])) {
	    /* Here's the entry; now process it. */
	    if (crossref_bnf_rule_index < 0) {
	      /* Start of a new BNF rule */
	      if (crossref_data[j][1])
		LOSE(crossref_data[j][1]);
	      crossref_data[j][1] = crossref_data[j][0];
	      crossref_bnf_rule_index = j;
	    }
	    else {
	      /* Reference within a BNF rule */
	      char **foo = &crossref_data[j][2];
	      char *ruleref = crossref_data[crossref_bnf_rule_index][0];
	      while (*foo && *foo != ruleref) ++foo;
	      if (foo == &crossref_data[j][1+MAXREFS])
		LOSE("too many crossrefs");
	      if (!*foo) *foo = ruleref;
	    }
	    break;
	  }
	}
      }
    }
  }
}

int crossref_compar(x, y)
  const void *x, *y;
{
  char **a = (char **) x;
  char **b = (char **) y;
  if (*a == *b) return 0;
  if (!*a) return 1;
  if (!*b) return -1;
  return (strcmp(*a, *b));
}

void print_crossref_portion(terminals, defined)
  int terminals, defined;
{
  int j;
  int desired = !!terminals + 2*!!defined;
  int found;
  char **foo, **bar;
  fprintf(stdout, "\\begin{BNFcrossref%s}\n", defined ? "" : "*");
  for (j = 0; crossref_data[j][0]; j++) {
    found = !islower(crossref_data[j][0][0]) + 2*!!crossref_data[j][1];
    if (desired == found) {
      fprintf(stdout, "{\\%s %s}\\>",
	      terminals ? "tt" : "it",
	      crossref_data[j][0]);
      if (crossref_data[j][1])
	fprintf(stdout, "\\ref{%s-rule}", crossref_data[j][1]);
      for (foo = bar = &crossref_data[j][2]; *foo; foo++) {
	if ((foo > bar) && ((foo - bar) % NUM_XREFS == 0))
	  fprintf(stdout, "\\\\\n   \\>\\>");
	else fprintf(stdout, "\n     \\>");
	fprintf(stdout, "\\ref{%s-rule}", *foo);
      }
      fprintf(stdout, "\\\\\n");
    }
  }
  fprintf(stdout, "\\end{BNFcrossref%s}\n", defined ? "" : "*");
}

void print_crossref() {
  qsort(crossref_data, MAXITEMS, (MAXREFS+2)*sizeof(char *), crossref_compar);
  fprintf(stdout, "\\chapter{Syntax Cross-reference}\n");
  fprintf(stdout, "\\crossrefpreface\n");
  fprintf(stdout, "\\section{Nonterminal Symbols That Are Defined}\n");
  print_crossref_portion(0, 1);
  fprintf(stdout, "\\section{Nonterminal Symbols That Are Not Defined}\n");
  print_crossref_portion(0, 0);
  fprintf(stdout, "\\section{Terminal Symbols}\n");
  print_crossref_portion(1, 0);
}

main(argc, argv)
  int argc; char *argv[];
{
  int j;
  char *line, *loc;
  chapter_pending = 0;
  section_pending = 0;
  within_bnf = 0;
  within_constraints = 0;
  bnf_output = 0;
  noconstraints = 0;
  crossref = 0;
  crossref_bnf_rule_index = -1;
  poolptr = stringpool;
  fprintf(stdout, "\\chapter{Syntax Rules}\n");
  fprintf(stdout, "\\BNFpreface\n");
  fprintf(stdout, "\\begin{BNFchapter}\n");
  for (j = 1; j < argc; j++) {
    if (argv[j][0] == '-') {
      if (!strcmp(&argv[j][1], "noconstraints")) noconstraints = 1;
      else if (!strcmp(&argv[j][1], "crossref")) crossref = 1;
      else LOSE("unknown switch");
    }
    else {
      FILE *f;
      current_file = argv[j];
      f = fopen(current_file, "r");
      if (!f) LOSE("No such file or directory");
      within_bnf = 0;
      within_constraints = 0;
      while (!feof(f)) {
	line = fgets(line_buffer, BUFLEN, f);
	if (!line) { line = line_buffer; line_buffer[0] = 0; }
	if (loc = panic(line, "\\chapter")) {
	  if (within_bnf) LOSE("\\chapter command appears within BNF");
	  ++chapter_pending;
	  section_pending = 0;
	  strcpy(chapter_buffer, loc+8);
	  /*        printf("FOO: %s", chapter_buffer); */
	}
	else if (loc = panic(line, "\\section")) {
	  if (within_bnf) LOSE("\\section command appears within BNF");
	  ++section_pending;
	  strcpy(section_buffer, loc+8);
	  /*        printf("BAR: %s", section_buffer); */
	}
	else if (panic(line, "\\BNF")) {
	  int need_bnf_open = chapter_pending || section_pending
	                      || !bnf_output;
	  if (within_bnf) LOSE("\\BNF command appears within BNF");
	  if (within_constraints)
	    LOSE("\\BNF command appears within constraints");
	  if (bnf_output && need_bnf_open) ENDBNF;
	  for ( ; chapter_pending; --chapter_pending) {
	    if (chapter_pending > 1)
	      fprintf(stdout, "\\stepcounter{section}\n");
	    else {
	      fprintf(stdout, "\\section");
	      fputs(chapter_buffer, stdout);
	    }
	  }
	  for ( ; section_pending; --section_pending) {
	    if (section_pending > 1)
	      fprintf(stdout, "\\stepcounter{subsection}\n");
	    else {
	      fprintf(stdout, "\\subsection");
	      fputs(section_buffer, stdout);
	    }
	  }
	  if (need_bnf_open) STARTBNF;
	  else fprintf(stdout, "\n");
	  within_bnf = 1;
	  crossref_bnf_rule_index = -1;
	}
	else if (panic(line, "\\FNB")) {
	  if (!within_bnf) LOSE("\\FNB command appears outside BNF");
	  within_bnf = 0;
	}
	else if (panic(line, "\\begin{constraints}") && !noconstraints) {
	  if (within_bnf) LOSE("\\begin{constraints} appears within BNF");
	  if (within_constraints)
	    LOSE("\\begin{constraints} appears within constraints");
	  if (chapter_pending) LOSE("constraints after \\chapter without BNF");
/*	  if (section_pending) LOSE("constraints after \\section without BNF"); */
	  if (bnf_output) ENDBNF;
	  fputs(line, stdout);
	  within_constraints = 1;
	}
	else if (panic(line, "\\end{constraints}") && !noconstraints) {
	  if (!within_constraints)
	    LOSE("\\end{constraints} appears outside constraints");
	  fputs(line, stdout);
	  within_constraints = 0;
	}
	else if (loc = panic(line, "\\Fortranrule")) {
	  loc = strchr(loc, '{');
	  if (loc) loc = strchr(loc+1, '{');
	  if (loc) {
	    crossref_bnf_rule_index = -1;
	    crossref_analyze(loc);
	    crossref_bnf_rule_index = -1;
	  }
	}
	else if (within_bnf) {
	  fputs(line, stdout);
	  crossref_analyze(line);
	}
	else if (within_constraints) {
	  fputs(line, stdout);
	}
      }
      if (within_bnf) LOSE("within BNF at end of file");
      if (within_constraints) LOSE("within constraints at end of file");
    }
  }
  if (bnf_output) ENDBNF;
  fprintf(stdout, "\\end{BNFchapter}\n");
  if (crossref) print_crossref();
}
