/* -*-mb0-c-*-

   SB-PRAM simulator

   (C) 1994 by Michael Bosch (hirbli@cs.uni-sb.de)
   and Stefan Franziskus (stefran@cs.uni-sb.de)

   Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee is hereby granted, provided
   that the above copyright notice appear in all copies.
*/

#include "config.h"

static const char sccsid[] = "@(#)sysc.c	1.16";

#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdarg.h>

#include "processor.h"
#include "simul.h"
#include "glovars.h"

static struct descriptor **ld;

static inline UWORD GlobalAddr(struct vpregs *vp, UWORD adr)
{
	 if(NEGATIV(adr)) adr += vp->BASE;
	 else if(vp->pp->MOD & PROTADD) adr += (vp->PROT << 16);
	 if(adr>GlobalMemLen) dostop_error(vp, "SIGSEGV :)");
	 /*### PROTLO & PROTHI */
	 return adr;
}

#if 0
static void sysc_debug(const char *format, ...)
{
	va_list arg;
	va_start(arg, format);
	fprintf(stderr, "SYSC: ");
	vfprintf(stderr, format, arg);
	fprintf(stderr, "\n");
	va_end(arg);
}
#else
#ifdef __GNUC__
#define sysc_debug(a...)
#else
#define sysc_debug
#endif
#endif

static WORD pram_close(struct descriptor **fdp)
{
	struct descriptor *fd = *fdp;
sysc_debug("close(%d)", fdp-ld);
	if(!fd) return -1;
	*fdp = NULL;
	if(--(fd->opencnt)) return 0;
	else return close(fd->fd);
}

WORD pram_std_open(WORD fd)
{
sysc_debug("std_open(%d)", fd);
	if((unsigned)fd >= 3) return -1;
	pram_close(&ld[fd]);
	ld[fd] = &desc[fd];
	desc[fd].opencnt++;
	return 0;
}

void get_string(WORD *s, char *d)
{
	while(*s) *d++ = (char)*s++;
	*d++ = '\0';
}

unsigned convert_open_mode(WORD mode)
{
	int i;
	unsigned newmode = 0;
	static unsigned modetable[] =
	{
		O_WRONLY, O_RDWR, O_NDELAY, O_APPEND,
		0, 0, 0, 0,
		0, O_CREAT, O_TRUNC, O_EXCL,
		O_NDELAY, 0, O_NONBLOCK, 0
	};

	for(i=0; i<16; i++)
	{
		if(mode & 1) newmode |= modetable[i];
		mode >>= 1;
	}
	return newmode;
}

int pram_global_open(struct descriptor *fd, const char *name, unsigned mode, unsigned flags)
{
	int i;
	i = open(name, mode, flags);
	if(i<0) return -1;
	fd->fd = i;
	fd->isatty = 1;                          /*  isatty(i); */
	fd->word_type = fd->isatty?0x20:0x40;
	
	strcpy(fd->name, name);
	fd->mode = mode;
	fd->opencnt = 0;
	return 0;
}

static WORD pram_open(WORD adr, WORD mode, WORD flags)
{
	char name[256];
	int i, fdn;
	struct descriptor **fdp, *fd;

	get_string(GlobalMem + adr, name);
sysc_debug("open(\"%s\", 0x%x, 0x%x)", name, mode, flags);
	for(i=0; i<LFILEANZ; i++)
		if(!ld[i]) break;
	if(i>=LFILEANZ) return -1;
	fdp = &ld[i];
	fdn = i;

	for(i=0; i<FILEANZ; i++)
		if(!desc[i].opencnt) break;
	if(i>=FILEANZ) return -1;
	fd = &desc[i];

	if(0>pram_global_open(fd, name, convert_open_mode(mode), flags))
		return -1;
	fd->opencnt++;
	*fdp = fd;
	return fdn;
}

static WORD pram_read(struct descriptor **fd, WORD adr, WORD len)
{
	WORD *buffer;
	char help[4096];
    int gok, i;

sysc_debug("read(%d, 0x%x, %d)", fd-ld, adr, len);
	buffer = GlobalMem + adr;

	if(!*fd) return -1;
	i = (*fd)->fd;
	if((*fd)->isatty)
	{
		for(gok=0; len>0;)
		{
			int r, l;
			unsigned char *h;
			l = len>4096? 4096: len;
			r = read(i, help, l);
			if(r>=0)
			{
				gok += r;
				for(h=help; r>0; r--) *buffer++ = *h++;
			}
			else gok = r;
			if(r!=l) break;
			len -= r;
		}
	}
	else
	{
		gok = read(i, buffer, len*4);
		if(gok>0) gok /= 4;
	}

	return gok;
}

static WORD pram_write(struct descriptor **fd, WORD adr, WORD len)
{
	WORD *buffer;
	char help[4096];
	int i, gok;

sysc_debug("write(%d, 0x%x, %d)", fd-ld, adr, len);
	buffer = GlobalMem + adr;

	if(!*fd) return -1;
	i = (*fd)->fd;
if((*fd)->isatty)
	{
		for(gok=0; len>0;)
		{
			int r, l;
			unsigned char *h;
			r = l = len>4096? 4096: len;
			if(l>=0)
				for(h=help; l>0; l--) *h++ = (char)*buffer++;
			
			if((l = write(i, help, r))>=0)
				gok += l;
			else
				gok = l;
			if(r!=l) break;
			len -= r;
		}
	}
	else
	{
		gok = write(i, buffer, len*4);
		if(gok>0) gok /= 4;
	}

	return gok;
}

static void pram_internal_dup2(struct descriptor **from,
							   struct descriptor **to)
{
	if(*to) pram_close(to);
	if(*from)
	{
		*to = *from;
		(*from)->opencnt++;
	}
}

static WORD pram_sys_fork(WORD quellproz)
{
	int i, q;
sysc_debug("sys_fork(0x%x)", quellproz);
	quellproz >>= 5;
	if(quellproz>=4096) return -1;
	q = pnummap[quellproz];
	if(q<0) return -1;
	for(i=0; i<LFILEANZ; i++)
		pram_internal_dup2(&ldesc[q][i], &ld[i]);
	return 0;
}

static WORD pram_lseek(struct descriptor **fd, WORD offset, WORD whence)
{
	int i;
	if(!*fd) return -1;
	i = (*fd)->fd;
sysc_debug("lseek(%d, %d, %d)", fd-ld, offset, whence);
    return lseek(i, 4*offset, whence);
}

void sysc_reset(void)
{
	int i, j;
	for(i=3; i<FILEANZ; i++)
		if(desc[i].opencnt)
		{
			close(desc[i].fd);
			desc[i].opencnt = 0;
		}
	for(i=0; i<3; i++)
	{
		desc[i].opencnt = 1;
		desc[i].fd = i;
		desc[i].isatty = isatty(i);
	}
	for(i=0; i<VPnum*PPnum; i++)
	{
		ld = &ldesc[i][0];
		for(j=0; j<LFILEANZ; j++) ld[j] = NULL;
		pram_std_open(0);
		pram_std_open(1);
		pram_std_open(2);
	}
}

int pram_ioctl(int fd, int request, int *arg)
{
	
}


#define PAR1 vp->R[7]
#define PAR2 vp->R[8]
#define PAR3 vp->R[9]
#define RET vp->R[5]

#define fd_conv(fd) (ld + (fd))

int SYSCSimul(struct vpregs *vp)
{
	ld = &ldesc[vp - VirtualProcessor][0];
	switch(vp->R[5])
	{
	case 0:
		RET = pram_open(GlobalAddr(vp, PAR1), PAR2, PAR3);
		break;
	case 1:
		RET = pram_read(fd_conv(PAR1), GlobalAddr(vp, PAR2), PAR3);
		break;
	case 2:
		RET = pram_write(fd_conv(PAR1), GlobalAddr(vp, PAR2), PAR3);
		break;
	case 3:
		RET = pram_close(fd_conv(PAR1));
		break;
	case 4:
		RET = pram_lseek(fd_conv(PAR1), PAR2, PAR3);
		break;
	case 5:
		RET = pram_std_open(PAR1);
		break;
	case 8:
		dostop_exit(vp);
		RET = 1;
		break;
	case 11:
		RET = pram_sys_fork(PAR1);
		break;
	case 26:
		RET = pram_ioctl(fd_conv(PAR1), PAR2, PAR3);
		break;
	default:
		return 0;
	}
	return 1;
}
