/* -*-mb0-c-*-

   SB-PRAM simulator

   (C) 1994 by Michael Bosch (hirbli@cs.uni-sb.de)
   and Stefan Franziskus (stefran@cs.uni-sb.de)

   Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee is hereby granted, provided
   that the above copyright notice appear in all copies.
*/

#include "config.h"

static const char sccsid[] = "@(#)stimu.c	1.60";

#include <stdio.h>

#include "processor.h"
#include "glovars.h"
#include "simul.h"
#include "net.h"

#define NZVCFlags (CFlag|ZFlag|NFlag|VFlag)
#define FZNIIFlags (FZFlag|FNFlag|FNANFlag|FINFFlag)

/**********************************
  Features
 **********************************/

#ifdef FAST
#undef IFOPT
#define IFOPT(a) if(options & ((a)&~OPT_SLOW))
#endif

/**********************************
  Exceptions
 **********************************/

#define error(S) \
do { \
	dostop_error(vp, S); \
	goto skip_execute; \
} while(0)

#define TestVpEx() vp->testex = 1
#define TestPpEx() vp->testex = 1, pp->testex = 1

#define ILLEGAL(TEXT) \
do { \
	IFOPT(OPT_SIMULATE_ILLEGAL) \
		dostop_illegal(vp, TEXT); \
	else \
	{ \
		SR |= IOPEXFlag; \
		TestVpEx(); \
	} \
	goto skip_execute; \
} while(0)

#define PROTCHECK if(!(vp->SR & SUPERFlag) && !(pp->MOD & GLSUPER) \
					 && (pp->MOD & PROTECINT)) \
	ILLEGAL("privileged instruction")

/**********************************
  Register ausmaskieren
 **********************************/

#define GetR1 (vp->R[GetR1Num])
static inline UWORD GetS2_func(struct vpregs *vp, UWORD Command)
{
	return (Command&(1<<14))? GetI13: GetR1;
}
#define GetS2 GetS2_func(vp, Command)
#define GetR2 (vp->R[GetR2Num])
#define GetRd (vp->R[GetRdNum])

static inline UWORD GetRdSR_func(struct vpregs *vp, UWORD Command, UWORD SR)
{
	int regno = (Command>>RdPos) & 31;
	if(regno) return vp->R[regno];
	else return SR;
}
#define GetRdSR GetRdSR_func(vp, Command, SR)
#define isI13 (Command & (1<<14))

/**********************************
  Load-Handling
 **********************************/

#define DLoad(VAL) \
do { \
	int regno = (Command>>RdPos)&31; \
	if(!regno) PROTCHECK; \
	vp->dl_regno = regno; \
	vp->dl_val = (VAL); \
} while(0)

#define Load(VAL) \
do { \
	int regno = (Command>>RdPos)&31; \
	if(!regno) PROTCHECK; \
	if(dl_regno>=0) ILLEGAL("local/global load collision"); \
	dl_regno = regno; \
	dl_val = (VAL); \
} while(0)

#define DO_OP_AND val & mem
#define DO_OP_OR val | mem
#define DO_OP_ADD val + mem
#define DO_OP_MAX (val<mem)? mem: val
#define DO_OP_NONE val
#define INVALWRITE 0
#define FORCEWRITE 1       /* fuer Shadow */

#define NetBlast(MODUS, OP, ADDR, DATA, FORCE) \
do { \
	UWORD vaddr = (ADDR); \
	UWORD paddr = vaddr; \
	WORD mem, val=(DATA); \
    \
	if(NEGATIV(paddr)) paddr += vp->BASE; \
	else if(pp->MOD & PROTADD) paddr += (vp->PROT << 16); \
	if(paddr>GlobalMemLen) error("global bus error"); \
	if(!(SR & SUPERFlag) && !(pp->MOD & GLSUPER) \
	   && (pp->MOD & PROTECGLOB) \
	   && (paddr < (vp->PROT << 16) || (paddr >> 16) > (vp->PROT >> 16))) \
		ILLEGAL("global segmentation violation"); \
	\
	if(MODUS!=MOD_ST) mem = GlobalMem[paddr]; \
	if(MODUS==MOD_MP || MODUS==MOD_LD) DLoad(mem); \
	if((MODUS!=MOD_LD) && \
     ((!(SR & SHADOWFlag)) || FORCE)) GlobalMem[paddr] = DO_##OP; \
	IFOPT(OPT_NET) \
		 EmitNetPack(MODUS, OP, vaddr, paddr, DATA, vp); \
} while(0)

/**********************************
  Compute Ergebniszuweisung
 **********************************/

#define Assign(ERG) \
do \
{ \
	UWORD ergebnis = (ERG); \
	int regno = (Command>>RdPos)&31; \
	if(regno>=2) vp->R[regno] = ergebnis; \
} \
while(0)

#define Assign_SetZN(ERG) \
do \
{ \
	UWORD ergebnis = (ERG); \
	int regno; \
	SR |= (ergebnis>>31) << NFlagPos; /*### wirklich gut? */ \
	if(!ergebnis) SR |= ZFlag; \
	regno = (Command>>RdPos)&31; \
	if(regno>=2) vp->R[regno] = ergebnis; \
} \
while(0)

/**********************************
  Float Macros
 **********************************/

#define FAssign_Set(ERG) \
do \
{ \
	union int_float q; \
	int regno; \
	q.f = (ERG); \
	Clear(SR, FZNIIFlags); \
	if(q.f < 0.0) Set(SR, FNFlag); \
	if(q.f == 0.0) Set(SR, FZFlag); /*### ISNAN ISINF */ \
	regno = (Command>>RdPos)&31; \
	if(regno>=2) vp->R[regno] = q.i; \
} \
while(0)

/**********************************
  Conditions fuer Branches
 **********************************/

static inline int cond(struct vpregs *vp, UWORD cmd, UWORD SR)
{
	int ret, cond;
	cond = (cmd >> RdPos) & 31;
	CondCounter[cond]++;
	switch(cond & 0x17)
	{
	case 0: ret = 1; break;
	case 1: ret = SR & NFlag; break;
	case 2: ret = SR & ZFlag; break;
	case 3: ret = SR & VFlag; break;
	case 4: ret = SR & CFlag; break;
	case 5: ret = SR & (CFlag|ZFlag); break;
	case 6: ret = (SR & ZFlag) || (((SR >> NFlagPos)^(SR >> VFlagPos)) & 1);
		break;
	case 7: ret = (((SR >> NFlagPos)^(SR >> VFlagPos)) & 1); break;
	case 16: ret = ModuloBit; break;
	case 19: ret = SR & SHADOWFlag; break;
	case 20: ret = SR & LdZFlag; break;
	case 21: ret = SR & LdNFlag; break;
	case 22: ret = SR & DATAEXFlag; break;
	case 23: ret = SR & KONTEXFlag; break;
	default: ret = 0; error("unimplemented condition"); break;
	}
 skip_execute:
	return (cond&8)? !ret: ret;
}

static inline int fcond(UWORD cmd, UWORD SR)
{
	int ret, cond;
	cond = (cmd >> RdPos) & 31;
	FCondCounter[cond]++;
    switch(cond)
    {
	case 0: ret = SR & FNFlag; break;
	case 1: ret = SR & FZFlag; break;
	case 2: ret = SR & FINFFlag; break;
	case 3: ret = SR & FNANFlag; break;
	case 4: ret = SR & FIEXFlag; break;
	case 5: ret = SR & FIOPFlag; break;
	case 6: ret = SR & FUFFlag; break;
	case 7: ret = SR & FOFFlag; break;
	case 8: ret = ~SR & FNFlag; break;
	case 9: ret = ~SR & FZFlag; break;
	case 10: ret = ~SR & FINFFlag; break;
	case 11: ret = ~SR & FNANFlag; break;
	case 12: ret = ~SR & FIEXFlag; break;
	case 13: ret = ~SR & FIOPFlag; break;
	case 14: ret = ~SR & FUFFlag; break;
	case 15: ret = ~SR & FOFFlag; break;
	case 16: ret = !(SR & (FNANFlag|FZFlag|FNFlag)); break;
	case 17: ret = (SR & FZFlag) || !(SR & (FNANFlag|FNFlag)); break;
	case 18: ret = (SR & FNFlag) && !(SR & (FNANFlag|FZFlag)); break;
	case 19: ret = (SR & FZFlag) || ((SR & FNFlag) && !(SR & FNANFlag)); break;
	case 20: ret = !(SR & (FNANFlag|FZFlag)); break;
	case 21: ret = SR & FPEXFlag; break;
	case 22: ret = SR & FASFlag; break;
	case 23: ret = SR & FAIFlag; break;
	case 24: ret = SR & (FNANFlag|FZFlag|FNFlag); break;
	case 25: ret = (SR & FNANFlag) || ((SR & FNFlag) && !(SR & FZFlag)); break;
	case 26: ret = (SR & (FNANFlag|FZFlag)) || !(SR & FNFlag); break;
	case 27: ret = (SR & FNANFlag) || !(SR & (FNFlag|FZFlag)); break;
	case 28: ret = SR & (FNANFlag|FZFlag); break;
	case 29: ret = ~SR & FPEXFlag; break;
	case 30: ret = ~SR & FASFlag; break;
	case 31: ret = ~SR & FAIFlag; break;
	default: ret = 0; break;
	}
	return ret;
}

#ifdef FAST
void SimulateNStepsFast(struct vpregs *vp, int len, unsigned options)
#else
void SimulateNStepsSlow(struct vpregs *vp, int len, unsigned options)
#endif
{
	UWORD PC, SR, Command, p1, p2, ue, e, e_hi, e_lo;
	ULWORD e_mul;
	int dl_regno;
	WORD dl_val, se;
	struct ppregs *pp;

	for(; len--; vp++)
	{
		dl_regno = vp->dl_regno;
		dl_val = vp->dl_val;
		vp->dl_regno = -1;

		PC = vp->R[1];
		SR = vp->SR;
		pp = vp->pp;

		if(vp->testex || pp->ExtEx)
		{
			int ExNR;

			vp->testex = 0;
			ExNR = pp->ExtEx;
			if(!ExNR && !(SR & MASKFlag) && !(pp->MOD & GLMASK))
			{
				int e;
				for(e=11; e<16; e++)
					if((SR >> (e-9)) & (~(pp->MOD) >> (20-e)) & 1)
					{
						SR |= MASKFlag|SUPERFlag;
						ExNR = e;
						break;
					}
			}
			if(!ExNR);
			else if(ExNR==15 && (options&OPT_SIMULATE_SYSC) && SYSCSimul(vp))
			{
				SR &= ~(SYSCFlag|MASKFlag|SUPERFlag);
				vp->testex = 1;
				goto skip_execute;
			}
/*			else if(ExNR==11)
			{
				SR &= ~(IOPEXFlag|MASKFlag|SUPERFlag);
				vp->testex = 1;
				goto skip_execute;
			} */
			else
			{
				IFOPT(OPT_TRACE) trace_log(ExNR);
				Command = ProgramMem[ExNR];
				goto skip_fetch;
			}
		}

		IFOPT(OPT_TRACE) trace_log(PC);
		if(PC>=ProgramMemLen) error("program bus error");
		Command = ProgramMem[PC++]; /* PC may be changed in switch */

	skip_fetch:

		OpcodeCounter[Command>>OpcodePos]++;
		switch(Command>>OpcodePos)
		{
			/***************************************************
			  Load/Store, Push/Pop, MP/SYNC
			  ***************************************************/
			
		case CMD_LDG:
			NetBlast(MOD_LD, OP_NONE, GetR2 + GetS2, 0, INVALWRITE);
			break;
			
		case CMD_LDL:
			p1 = GetR2 + GetS2;
			Load(LocMemLoad(pp, vp, SR, p1));
			break;
			
		case CMD_POPNG:
			p1 = GetR2 + GetI13;
			NetBlast(MOD_LD, OP_NONE, p1, 0, INVALWRITE);
			GetR2 = p1;
			break;
			
		case CMD_POPIG:
			p1 = GetR2 + GetI13;
			NetBlast(MOD_LD, OP_NONE, p1, 0, INVALWRITE);
			GetR2 = p1 + 1;
			break;
			
		case CMD_POPNL:
			p1 = GetR2 + GetI13;
			Load(LocMemLoad(pp, vp, SR, p1));
			GetR2 = p1;
			break;
			
		case CMD_POPIL:
			p1 = GetR2 + GetI13;
			Load(LocMemLoad(pp, vp, SR, p1));
			GetR2 = p1 + 1;
			break;

		case CMD_PSHNG:   /*### Attn: GetR2 == R0 ! */
			p1 = GetR2 + GetI13;
			NetBlast(MOD_ST, OP_NONE, p1, GetRdSR, FORCEWRITE);
			GetR2 = p1;
			break;
			
		case CMD_PSHIG:
			if(isI13) 
			{
				p1 = GetR2 + GetI13;
				NetBlast(MOD_ST, OP_NONE, p1, GetRdSR, FORCEWRITE);
			}
			else 
			{
				p1 = GetR2;
				NetBlast(MOD_ST, OP_NONE, p1, 0, FORCEWRITE);
			}
			GetR2 = p1 + 1;
			break;
			
		case CMD_PSHNL:
			p1 = GetR2 + GetI13;
			LocMemStore(pp, vp, SR, p1, GetRdSR);
			GetR2 = p1;
			break;
			
		case CMD_PSHIL:
			if(isI13) 
			{
				p1 = GetR2 + GetI13;
				LocMemStore(pp, vp, SR, p1, GetRdSR);
			}
			else 
			{
				p1 = GetR2;
				LocMemStore(pp, vp, SR, p1, 0);
			}
			GetR2 = p1 + 1;
			break;
			
#define MP(NAME) \
		case CMD_MP##NAME: \
			p2 = GetRd; \
			NetBlast(MOD_MP, OP_##NAME, GetR2 + GetI13, p2, INVALWRITE); \
			break;
		MP(AND)
		MP(OR)
		MP(ADD)
		MP(MAX)
#undef MP
								
#define MP(NAME) \
		case CMD_SYNC##NAME: \
			p2 = GetRd; \
			NetBlast(MOD_SY, OP_##NAME, GetR2 + GetI13, p2, INVALWRITE); \
			break;
		MP(AND)
		MP(OR)
		MP(ADD)
		MP(MAX)
#undef MP

		case CMD_STG:
			if(isI13) NetBlast(MOD_ST, OP_NONE, \
									 GetR2 + GetI13, GetRdSR, INVALWRITE);
			else NetBlast(MOD_ST, OP_NONE, GetR2, 0, INVALWRITE);
			break;
			
		case CMD_STL:

			if(isI13) LocMemStore(pp, vp, SR, GetR2 + GetI13, GetRdSR);
			else LocMemStore(pp, vp, SR, GetR2, 0);
			break;
			
			
			/***************************************************
			  Intern
			  ***************************************************/
			
		case CMD_GETSR:
			PROTCHECK;
			Assign(SR);
			break;
			
		case CMD_GETMOD:
			PROTCHECK;
			Assign(pp->MOD);
			break;
			
		case CMD_GETLOG:
			PROTCHECK;
			Assign(vp->LOG);
			break;
			
		case CMD_GETLD:
			PROTCHECK;
			Assign(vp->LD);
			break;
			
		case CMD_PUTSR:
			PROTCHECK;
			SR = ((SR & GetS2) | GetR2);
			TestVpEx();
			break;
			
		case CMD_PUTMOD:
			PROTCHECK;
			pp->MOD = ((pp->MOD & GetS2) | GetR2);
			TestPpEx();
			break;
			
		case CMD_PUTLOG:
			PROTCHECK;
			vp->LOG = ((vp->LOG & GetS2) | GetR2);
			break;
			
		case CMD_PUTLD:
			PROTCHECK;
			vp->LD = ((vp->LD & GetS2) | GetR2);
			break;
			
		case CMD_GETNR:
			PROTCHECK;
			Assign(pp->NR | (vp->nr<<5));
			break;
			
		case CMD_GETCT:
			PROTCHECK;
			Assign(pp->CT);
			break;
			
		case CMD_GETA:
			PROTCHECK;
			Assign(pp->A);
			break;
			
		case CMD_GETPR:
			PROTCHECK;
			Assign(vp->PROT);
			break;
			
		case CMD_GETBAS:
			PROTCHECK;
			Assign(vp->BASE);
			break;
			
		case CMD_PUTNR:
			PROTCHECK;
			pp->NR = GetS2 << 10;
			break;
			
		case CMD_PUTCT:
			PROTCHECK;
			pp->CT = GetS2;
			break;
			
		case CMD_PUTA:
			PROTCHECK;
			pp->A = GetS2;
			break;
			
		case CMD_PUTPR:
			PROTCHECK;
			ue = GetS2;
			vp->PROT = ue;
			break;
			
		case CMD_PUTBAS:
			PROTCHECK;
			vp->BASE = GetS2;
			break;
			
		case CMD_RES:
			PROTCHECK;
			PC = GetI19;
			SR = 0;
			vp->BASE = 0;
			vp->PROT = 0;
			vp->LOG = 0;
			vp->LD = 0;
			vp->dl_regno = -1;
			dl_regno = -1;
			break;
			
			/***************************************************
			  Compute
			  ***************************************************/
		case CMD_ADD:
			Clear(SR, NZVCFlags);
			p1 = GetS2;
			p2 = GetR2;
			e = p1 + p2;
			if(NEGATIV((p1 & p2 & ~e) | (~p1 & ~p2 & e)))
				Set(SR, VFlag);
			if(e<p1) Set(SR, CFlag);
			Assign_SetZN(e);
			break;
			
		case CMD_ADC:
			p1 = GetS2;
			p2 = GetR2;
			e = p1 + p2 + ((SR >> CFlagPos) & 1);
			Clear(SR, NZVCFlags);
			if(NEGATIV((p1 & p2 & ~e) | (~p1 & ~p2 & e)))
				Set(SR, VFlag);
			if(NEGATIV((p1 & p2) | ((p1 | p2) & ~e)))
				Set(SR, CFlag);
			Assign_SetZN(e);
			break;
			
		case CMD_SUB:
			p1 = GetS2;
			p2 = GetR2;
			e = p1-p2;
			Clear(SR, NZVCFlags);
			if(NEGATIV((~p1 & p2 & e) | (p1 & ~p2 & ~e)))
				Set(SR, VFlag);
			if(NEGATIV((~p1 & (p2 | e)) | (p2 & e)))
				Set(SR, CFlag);
			Assign_SetZN(e);
			break;
			
		case CMD_SBC:
			p1 = GetS2;
			p2 = GetR2;
			e = p1-p2-((SR >> CFlagPos) & 1);
			Clear(SR, NZVCFlags);
			if(NEGATIV((~p1 & p2 & e) | (p1 & ~p2 & ~e)))
				Set(SR, VFlag);
			if(NEGATIV((~p1 & (p2 | e)) | (p2 & e)))
				Set(SR, CFlag);
			Assign_SetZN(e);
			break;
			
		case CMD_AND:
			Clear(SR, NZVCFlags);
			Assign_SetZN(GetS2 & GetR2);
			break;
			
		case CMD_OR:
			Clear(SR, NZVCFlags);
			Assign_SetZN(GetS2 | GetR2);
			break;
			
		case CMD_NAND:
			Clear(SR, NZVCFlags);
			Assign_SetZN(~(GetS2 & GetR2));
			break;
			
		case CMD_XOR:
			Clear(SR, NZVCFlags);
			Assign_SetZN(GetS2 ^ GetR2);
			break;
			
		case CMD_MUL:
			p1 = GetS2;
			p2 = GetR2;
			e_mul = (LWORD)(WORD)p1 * (LWORD)(WORD)p2;
			e_lo = (WORD)e_mul;
			e_hi = (WORD)(e_mul >> 32);
			Clear(SR, NZVCFlags);
			if(NEGATIV(e_lo)? ~e_hi: e_hi) Set(SR, VFlag);
			if(NEGATIV(p1)) e_hi+=p2;
			if(NEGATIV(p2)) e_hi+=p1;
			if(e_hi!=0) Set(SR, CFlag);
			Assign_SetZN(e_lo);
			break;
			
		case CMD_MLHS:
			p1 = GetS2;
			p2 = GetR2;
			e_mul = (LWORD)(WORD)p1 * (LWORD)(WORD)p2;
			e_hi = (WORD)(e_mul >> 32);
			Clear(SR, NZVCFlags);
			Assign_SetZN(e_hi);
			break;
			
		case CMD_MLHU:
			p1 = GetS2;
			p2 = GetR2;
			e_mul = (ULWORD)(UWORD)p1 * (ULWORD)(UWORD)p2;
			e_hi = (UWORD)(e_mul >> 32);
			Clear(SR, NZVCFlags);
			Assign_SetZN(e_hi);
			break;
			
		case CMD_RM:
			p1 = GetS2;
			for(e=-1; p1; e++) p1>>=1;
			Clear(SR, NZVCFlags);
			Assign_SetZN(e);
			break;
			
		case CMD_GETHI:
			Assign(GetI19 << 13);
			break;
			
		case CMD_GETLO:
			Assign(GetI19);
			break;
			
		case CMD_LSL:
			Clear(SR, NZVCFlags);
			p1 = GetS2 & 31;
			e = GetR2;
			if(p1)
			{
				e <<= (p1-1);
				if(NEGATIV(e)) SR |= CFlag;
				e <<= 1;
			}
			Assign_SetZN(e);
			break;
			
		case CMD_LSR:
			Clear(SR, NZVCFlags);
			p1 = GetS2 & 31;
			ue = GetR2;
			if(p1)
			{
				ue >>= (p1-1);
				if(ue&1) SR |= CFlag;
				ue >>= 1;
			}
			Assign_SetZN(ue);
			break;
			
		case CMD_ASL:
			Clear(SR, NZVCFlags);
			p1 = GetS2 & 31;
			se = p2 = GetR2;
			if(p1)
			{
				se <<= (p1-1);
				if(NEGATIV(se)) SR |= CFlag;
				se <<= 1;
				if((se>>p1) != p2) SR |= VFlag;
			}
			Assign_SetZN(se);
			break;
			
		case CMD_ASR:
			Clear(SR, NZVCFlags);
			p1 = GetS2 & 31;
			se = GetR2;
			if(p1)
			{
				se >>= (p1-1);
				if(se&1) SR |= CFlag;
				se >>= 1;
			}
			Assign_SetZN(se);
			break;
			
		case CMD_ROL:
			Clear(SR, NZVCFlags);
			p1 = GetS2 & 31;
			ue = GetR2;
			if(p1)
			{
				ue = (ue << p1) | (ue >> (32-p1));
				if(ue&1) SR |= CFlag;
			}
			Assign_SetZN(ue);
			break;
			
		case CMD_ROR:
			Clear(SR, NZVCFlags);
			p1 = GetS2 & 31;
			ue = GetR2;
			if(p1)
			{
				ue = (ue >> p1) | (ue << (32-p1));
				if(NEGATIV(ue)) SR |= CFlag;
			}
			Assign_SetZN(ue);
			break;
			
		case CMD_ROCL:
			e = (SR >> CFlagPos) & 1;
			p1 = GetS2 & 31;
			ue = GetR2;
			if(p1)
			{
				UWORD rh = ue >> (32-p1);
				Clear(SR, NZVCFlags);
				ue = (ue << p1) | (e << (p1-1)) | (rh >> 1);
				if(rh&1) SR |= CFlag;
			} else Clear(SR, NFlag|ZFlag|VFlag);
			Assign_SetZN(ue);
			break;
			
		case CMD_ROCR:
			e = (SR >> CFlagPos) & 1;
			p1 = GetS2 & 31;
			ue = GetR2;
			if(p1)
			{
				UWORD lh = (ue << (32-p1));
				Clear(SR, NZVCFlags);
				ue = (ue >> p1) | (e << (32-p1)) | (lh << 1);
				if(NEGATIV(lh)) SR |= CFlag;
			} else Clear(SR, NFlag|ZFlag|VFlag);
			Assign_SetZN(ue);
			break;

#define FLOATOP(NAM, OP, ABS) \
		case CMD_##NAM: \
		{ \
			union int_float a, b; \
			a.i = GetR2; \
			b.i = GetR1; \
			a.f = a.f OP b.f; \
			if(ABS && a.f < 0.0) a.f = -a.f; \
			FAssign_Set(a.f); \
			break; \
		}
			
FLOATOP(FADD, +, 0)
FLOATOP(FADA, +, 1)
FLOATOP(FSUB, -, 0)
FLOATOP(FSBA, -, 1)
FLOATOP(FMUL, *, 0)
FLOATOP(FMLA, *, 1)

		case CMD_FTOI:
		{
			union int_float a;
			a.i = GetR1;
			a.i = (WORD)a.f;
			FAssign_Set(a.f);
			break;
		}
	
		case CMD_ITOF:
			FAssign_Set((float)(WORD)GetS2);
			break;

			/***************************************************
			  Jump
			  ***************************************************/
			
		case CMD_B:
			if(cond(vp, Command, SR)) PC += GetI19 - 1;
			break;
			
		case CMD_FB:
			if(fcond(Command, SR)) PC += GetI19 - 1;
			break;
			
		case CMD_J:
			if(cond(vp, Command, SR)) PC = GetR2 + GetS2;
			break;
			
		case CMD_FJ:
			if(fcond(Command, SR)) PC = GetR2 + GetS2;
			break;
			
		case CMD_J2:
			if(cond(vp, Command, SR)) PC = GetI19;
			break;
			
		case CMD_FJ2:
			if(fcond(Command, SR)) PC = GetI19;
			break;
			
		case CMD_SYSC:
			Set(SR, SYSCFlag);
			TestVpEx();
			break;
			
		case CMD_BSRG: /*### GetRd: Assign... */
			NetBlast(MOD_ST, OP_NONE, GetRd, PC, FORCEWRITE);
			PC += GetI19 - 1;
			GetRd++;
			break;
			
		case CMD_BSRL:
			LocMemStore(pp, vp, SR, GetRd, PC);
			PC += GetI19 - 1;
			GetRd++;
			break;
			
		case CMD_JSRG:
			NetBlast(MOD_ST, OP_NONE, GetRd, PC, FORCEWRITE);
			PC = GetR2 + GetI13;
			GetRd++;
			break;
			
		case CMD_JSRL:
			LocMemStore(pp, vp, SR, GetRd, PC);
			PC = GetR2 + GetI13;
			GetRd++;
			break;
			
		case CMD_JEXG:
			NetBlast(MOD_ST, OP_NONE, GetRd, PC, FORCEWRITE);
			PC = GetI19;
			GetRd++;
			break;
			
		case CMD_JEXL:
			LocMemStore(pp, vp, SR, GetRd, PC);
			PC = GetI19;
			GetRd++;
			break;

		case CMD_BREAK:
			switch(BreakPoint[GetBreak].type)
			{
			case BRK_ALL:
				dostop_break(vp);
				break;
			case BRK_VP:
				if((vp-VirtualProcessor) == BreakPoint[GetBreak].proznum)
					dostop_break(vp);
				break;
			case BRK_PP:
				dostop_break(vp);           /* ### */
				break;
			case BRK_VROW:
				dostop_break(vp);           /* ### */
				break;
			default:
				ILLEGAL("illegal breakpoint--type, please send bug report");
				break;
			}
			Command = BreakPoint[GetBreak].opcode;
			goto skip_fetch;
			break;
			
		case 0: /* switch Sprungtabellenstart bei 0: Spart ein add */
		default:
			ILLEGAL("illegal opcode");
			break;
		}
		
	skip_execute:
		
		vp->R[1] = PC; /* PC may be changed by load-packet */

		if(dl_regno>=0)
		{
			if(!dl_regno)
			{
				/* Usermode wird schon bei Load/DLoad abgefangen */
				SR = dl_val;
				TestVpEx();
			}
			else
			{
				vp->R[dl_regno] = dl_val;
				Clear(SR, LdNFlag|LdZFlag);
				SR |= ((UWORD)dl_val>>31) << LdNFlagPos; /*### wirklich gut? */
				if(!dl_val) SR |= LdZFlag;
			}
		}
		vp->SR = SR;
		IFOPT(OPT_BREAKPOINT)
		{
			switch(bkpt.test)
			{
			case BP_RANGE:
				if((vp->R[bkpt.regno] - bkpt.val1) < bkpt.val2)
					dostop_break(vp);
				break;
			case BP_MASK:
				if((vp->R[bkpt.regno] & bkpt.val1) == bkpt.val2)
					dostop_break(vp);
				break;
			}
		}
	}

	IFOPT(OPT_NET) SimulateNetRound(ModuloBit);
}
