/* -*-mb0-c-*-

   SB-PRAM simulator

   (C) 1994 by Michael Bosch (hirbli@cs.uni-sb.de)
   and Stefan Franziskus (stefran@cs.uni-sb.de)

   Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee is hereby granted, provided
   that the above copyright notice appear in all copies.
*/

#include "config.h"

static const char sccsid[] = "@(#)simulate.c	1.38";

#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <fcntl.h>

#include "processor.h"
#include "glovars.h"
#include "simul.h"
#include "interface.h"

static RETSIGTYPE CoreHandler()
{
	break_flags |= BRK_DUMP;
	signal(SIGHUP, CoreHandler);
}

static RETSIGTYPE KillHandler()
{
	break_flags |= BRK_DUMP|BRK_STOP|BRK_EXIT;
}

static RETSIGTYPE BreakHandler()
{
	break_flags |= BRK_STOP;
	signal(SIGINT, BreakHandler);
}

void dostop_error(struct vpregs *vp, const char *s)
{
	break_flags |= BRK_STOP;
	sim_print(IO_STDERR, "error: vp=#%d, pc=$%08x, reason=%s\n",
			vp-VirtualProcessor, vp->R[1], s);
}

void dostop_illegal(struct vpregs *vp, const char *s)
{
	break_flags |= BRK_STOP;
	sim_print(IO_STDERR, "ILLEGAL: vp=#%d, pc=$%08x, reason=%s\n",
			vp-VirtualProcessor, vp->R[1], s);
}

void dostop_break(struct vpregs *vp)
{
	break_flags |= BRK_STOP;
	sim_print(IO_STDERR, "BREAKPOINT: vp=#%d, pc=$%08x\n",
			  vp-VirtualProcessor, vp->R[1]);
}

void dostop_exit(struct vpregs *vp)
{
	break_flags |= BRK_STOP;
	sim_print(IO_STDERR, "EXIT: vp=#%d, pc=$%08x\n",
			  vp-VirtualProcessor, vp->R[1]);
}

#ifdef MOD_AUTONICE
static RETSIGTYPE NiceHandler()
{
#ifdef SVR4
	nice(nice(0)<15 ? 1:0);        /* I'm a genius :) */
#else
	if(getpriority(PRIO_PROCESS, 0) < 15)
		nice(1);
#endif
}
#endif

void trace_log(UWORD pc)
{
	fwrite(&pc, sizeof(UWORD), 1, TraceFile);
}

#define HOST_IRQ 0x40
#define PP_IRQ   0x20

#define interrupt(start, end, VAL) \
do { \
int _val=(VAL) & 0xff; \
if(_val&0x80) \
{ \
	if(_val==0x80) \
	{ \
		start = 0; \
		end = PPnum; \
	} \
	else \
	{ \
		if(_val!=0x81) sim_print(IO_STDERR, "illegal IRQ $%02x\n", _val); \
		start = end = 0; \
	} \
} \
else \
{ \
	if(_val>=PPnum) \
	{ \
		sim_print(IO_STDERR, "illegal IRQ $%02x\n", _val); \
		start = end = 0; \
	} \
	else \
	{ \
		start = _val; \
		end = start + 1; \
	} \
} \
} while(0)

void Simulate(unsigned single, unsigned options, int out_steps)
{
	long long steps = 0;
#ifdef MOD_AUTONICE
	struct itimerval value = {{3600, 0}, {3600, 0}};
#endif

	fflush(stdout);
	fflush(stderr);
	
	signal(SIGHUP, CoreHandler);
	signal(SIGTERM, KillHandler);
	signal(SIGINT, BreakHandler);

#ifdef MOD_AUTONICE
#ifdef SVR4
	nice(nice(0)<5 ? 5-nice(0) : 0);
/*	priocntl(P_PID, P_MYID, PC_SETPARMS, ...); */
#else
	if(getpriority(PRIO_PROCESS, 0) < 5)
		nice(5 - getpriority(PRIO_PROCESS, 0));
#endif /* SVR4 */
	signal(SIGALRM, NiceHandler);
	setitimer(ITIMER_REAL, &value, (struct itimerval *)NULL);
#endif /* MOD_AUTONICE */
	
	(void)get_time();
	break_flags = single;
	do
	{
		do
		{
			int ppn, ppe;
			struct ppregs *pp;
			UWORD host_irq;
			host_irq = *ExternalInterrupts & 0x1ff;
			if(!(host_irq & 0x100))
			{
				interrupt(ppn, ppe, host_irq);
				for(ppn=0; ppn<ppe; ppn++)
				{
					PhysicalProcessor[ppn].FpgaRegs.IR |= HOST_IRQ;
					PhysicalProcessor[ppn].testex = 1;
				}
				*ExternalInterrupts = 0x100;
			}
			for(pp=PhysicalProcessor, ppn=0; ppn<PPnum; ppn++, pp++)
			{
				int e, ExNR = 0;
				UWORD MOD = pp->MOD2;
				UWORD ct;

				ct = ++(pp->CT);
				if(!ct)
				{
					pp->ctex = 1;
					pp->testex = 1;
				}

				if(pp->testex && !(MOD & GLMASK))
				{
					unsigned ex_msk;
					pp->testex = 0;

					ex_msk = pp->FpgaRegs.IR & pp->FpgaRegs.MR;
					if(ex_msk)
					{
						for(e=7; ex_msk>>=1; e--);
						if(!((MOD>>(20-e)) & 1))
						{
							if(!ModuloBit) ExNR = e;
							else pp->testex = 1;
						}
					}
					if(ExNR);
					else if(pp->ctex && !(MOD & COUNTERmsk))
					{
						pp->ctex = 0;
						ExNR = 8;
					}
					else if((MOD & VPEX1) && !(MOD & VPEX1msk)) ExNR=9;
					else if((MOD & VPEX2) && !(MOD & VPEX2msk)) ExNR=10;
					if(ExNR)
					{
						sim_print(IO_STDERR, "Exception #%d\n", ExNR);
						pp->MOD = pp->MOD | GLMASK | GLSUPER;
					}
				}
				if(MOD != pp->MOD) pp->testex = 1;
				pp->MOD2 = pp->MOD;
				pp->ExtEx = ExNR;
			}
			if(options & OPT_SLOW)
				SimulateNStepsSlow(VirtualProcessor, VPnum*PPnum, options);
			else
				SimulateNStepsFast(VirtualProcessor, VPnum*PPnum, options);
			ModuloBit ^= 1;
			steps++;
		}
		while(!break_flags);
		
		if(break_flags & BRK_DUMP)
		{
			pram_dump_core("pcore");
			break_flags &= ~BRK_DUMP;
		}
	}
	while(!(break_flags & BRK_STOP));
	if(out_steps)
	{
		if(steps>=(1LL<<32))
			sim_print(IO_NOTICE, "Stop nach $%x%08x Runden, ",
					  (unsigned)(steps>>32), (unsigned)steps);
		else		
			sim_print(IO_NOTICE, "Stop nach %u Runden, ", (unsigned)steps);

		sim_print(IO_NOTICE, "%.3f kIps\n", VPnum*PPnum*steps/get_time()/1000);
	}
	if(break_flags & BRK_EXIT) exit(1);
}

#define error(S) \
do { \
	dostop_error(vp, S); \
	goto skip_execute; \
} while(0)

#define TestVpEx() vp->testex = 1
#define TestPpEx() vp->testex = 1
#define ILLEGAL(TEXT) \
do { \
	IFOPT(OPT_SIMULATE_ILLEGAL) \
		dostop_illegal(vp, TEXT); \
	else \
	{ \
		SR |= IOPEXFlag; \
		TestVpEx(); \
	} \
	goto skip_execute; \
} while(0)

#define LocalMemBase 0x400000        /* 0x400000  set zero only for testing */
#define LocalMemLen   0x10000
#define FpgaBase     0x000000
#define FpgaLen           0xe
#define UartBase     0x100000
#define UartLen           0x8

#define RANGE(X) ((adr2=addr-X##Base)<X##Len)

#define FPGA_IR 0
#define FPGA_MR 1
#define FPGA_PR 3

unsigned ppirq_cnt;

int LocMemLoad(struct ppregs *pp, struct vpregs *vp, UWORD SR, 
			   UWORD addr)
{
	UWORD adr2;
	if(!(SR & SUPERFlag) && !(pp->MOD & GLSUPER)
	   && (pp->MOD & PROTECLOC))
		ILLEGAL("local segmentation violation");
	if(RANGE(LocalMem))
	{
		return(pp->LocalMem[adr2]);
	}
	else if(RANGE(Fpga))
	{
		switch(adr2)
		{
		case FPGA_IR:
			return pp->FpgaRegs.IR;
		case FPGA_MR:
			return pp->FpgaRegs.MR & 0x7f;
		}
	}
	else if(RANGE(Uart))
	{
		if(!(pp->UartRegs.LSR & UART_DR))
		{
			if(read(pp->UartRegs.fd, &pp->UartRegs.RBR, 1)==1)
				pp->UartRegs.LSR |= UART_DR;
		}
		
		switch(adr2)
		{
		case UART_RBR:
			if(!(pp->UartRegs.LCR & UART_DLAB))
			{
				if(pp->UartRegs.LSR & UART_DR)
					pp->UartRegs.LSR &= ~UART_DR;
				return(pp->UartRegs.RBR);
			}
			break;
		case UART_LCR:
			return pp->UartRegs.LCR;
			break;
		case UART_LSR:
			return pp->UartRegs.LSR;          /* UART-Docu: 0110 0000 */
			break;
		default:
			error("Error in Uart-Access");
			break;
		}
	}
	else error("local bus error");
 skip_execute:
	return(0);
}

void LocMemStore(struct ppregs *pp, struct vpregs *vp, UWORD SR, 
				 UWORD addr, WORD data)
{
	UWORD adr2, old;
	UCHAR outchar;
	if(!(SR & SUPERFlag) && !(pp->MOD & GLSUPER)
	   && (pp->MOD & PROTECLOC))
		ILLEGAL("local segmentation violation");
	if(RANGE(LocalMem))
	{
		pp->LocalMem[adr2] = data;
	}
	else if(RANGE(Fpga))
	{
		switch(adr2)
		{
		case FPGA_IR:
			pp->FpgaRegs.IR &= data;
			pp->testex = 1;
			break;
		case FPGA_MR:
			pp->FpgaRegs.MR = data;
			pp->testex = 1;
			break;
		case FPGA_PR:
			old = pp->FpgaRegs.PR;
			if(~old & 0x100)
			{
				int ppn, ppe;
				ppirq_cnt--;
				interrupt(ppn, ppe, old);
				for(ppn=0; ppn<ppe; ppn++)
				{
					PhysicalProcessor[ppn].FpgaRegs.IR &= ~PP_IRQ;
					PhysicalProcessor[ppn].testex = 1;
				}
			}
			if(~data & 0x100)
			{
				int ppn, ppe;
				if(ppirq_cnt)
				{
					sim_print(IO_STDERR, "too many PPIRQs\n");
					pp->FpgaRegs.PR = data|0x100;
				}
				else
				{
					pp->FpgaRegs.PR = data;
					ppirq_cnt++;
					interrupt(ppn, ppe, data);
					for(ppn=0; ppn<ppe; ppn++)
					{
						PhysicalProcessor[ppn].FpgaRegs.IR |= PP_IRQ;
						PhysicalProcessor[ppn].testex = 1;
					}
				}
			}
		}
	}
	else if(RANGE(Uart))
	{
		switch(adr2)
		{
		case UART_THR:
			outchar = data;
			if(!(pp->UartRegs.LCR & UART_DLAB))
				write(pp->UartRegs.fd, &outchar, 1); 
			break;
		case UART_LCR:
			pp->UartRegs.LCR = data;
			break;
		case UART_LSR:
			pp->UartRegs.LSR = data;       /* UART-Docu: 0110 0000 */
			break;
		default:
			break;
		}
	}
	else error("local bus error");
 skip_execute:
}
