/* -*-mb0-c-*-

   SB-PRAM simulator

   (C) 1994 by Michael Bosch (hirbli@cs.uni-sb.de)
   and Stefan Franziskus (stefran@cs.uni-sb.de)

   Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee is hereby granted, provided
   that the above copyright notice appear in all copies.
*/

#include "config.h"
#include "interface.h"

static const char copyright[] =
	"@(#)PRAM simulator (C) 1994 by hirbli & stefran & cls";
static const char sccsid[] = "@(#)init.c	1.63";

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/filio.h>
#include <sys/termios.h>
#include <sys/stropts.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>

#include "init.h"
#include "processor.h"
#include "simul.h"
#include "glovars.h"
#include "net.h"

#define NEW_NULL(COUNT, TYPE) (TYPE *) malloc((COUNT) * sizeof(TYPE))

#define NEW(COUNT, TYPE, RET) \
do { \
    TYPE *ptr; \
	int cnt; \
    ptr = malloc((COUNT) * sizeof(TYPE)); \
    IFOPT(OPT_RANDOMIZE) \
	    for(cnt=0; cnt<((COUNT)*sizeof(TYPE)); cnt++) \
		    *((char *)ptr + cnt) =  (char ) rand();  \
    (RET) = (TYPE *)ptr; \
} while(0)


#ifdef MOD_DEV_MAPPING
#include "/home/dst/pram/pram/src/pram.h"
#endif
extern void *pram_malloc(unsigned size);
extern void *pramprg_malloc(unsigned size);
extern void pram_free(void *ptr);
extern void pramprg_free(void *ptr);
#define NEW_DEVPRAM(COUNT, TYPE) (TYPE *)pram_malloc((COUNT) * sizeof(TYPE))
#define NEW_DEVPROG(COUNT, TYPE) (TYPE *)pramprg_malloc((COUNT) * sizeof(TYPE))

#ifdef MOD_FILE_MAPPING
/* nochnix */
#endif
extern void *pram_fmap(unsigned size, const char *name);
extern void pram_funmap(void *ptr, const char *name);
#define NEW_SMFILE(COUNT, TYPE, NAME) \
        (TYPE *)pram_fmap((COUNT) * sizeof(TYPE), NAME)

UWORD *pram_realloc(UWORD *mem, int size, int new_size)
{
  UWORD *tmp;

  sim_print(IO_NOTICE, "Doing realloc\n");
  
  IFOPT(OPT_FILE_MAPPING)
  {
	  sim_print(IO_STDERR, "pram_realloc not supported with "
				"OPT_FILE_MAPPING\n");
	  exit(1);
  }
  else IFOPT(OPT_DEV_MAPPING)
  {
	  sim_print(IO_STDERR, "pram_realloc not supported with "
				"OPT_DEV_MAPPING\n");
	  exit(1);
 }
  else IFOPT(OPT_LAZY_MALLOC)
  {
	  if(mem == GlobalMem)
	  {
		  tmp = mem;
		  mem = NEW_SMFILE(new_size, UWORD, "/dev/zero");
		  memcpy(mem, tmp, size);
		  pram_funmap(tmp, "/dev/zero");
		  return mem;
	  }
	  else
		  return realloc(mem, new_size * sizeof(UWORD));
  }
  else
  {
	  return realloc(mem, new_size * sizeof(UWORD));
  }
  
  return (UWORD *)NULL;
}


char *TraceFileName = NULL;
char *dflt_cfg_file = "/home/pram/etc/pramsimrc";


struct option_desc command_opt[] =
{
	{'h', "help",        O_HELP, {0}, ""},
	{'M', "globmem",     I_INT(&GlobalMemLen),
		 "length of global memory [words]"},
	{'P', "progmem",     I_INT(&ProgramMemLen),
		 "length of program memory [words]"},
	{'L', "locmem",      I_INT(&LocalMemLen),
		 "length of local memory [words/PP]"},
	{'v', "virtProz",    I_INT(&VPnum),
		 "# of virtual processors (VPs) per PP"},
	{'p', "physProz",    I_INT(&PPnum),
		 "# of physical processors (PPs)"},
	{'l', "littleRound", I_INT(&KRnum),
		 "# of VP per little round (for GETNR only)"},
    {' ', "textstart",   I_STRING(textstart),
    	 "textstart for user--prog"},
	{' ', "datastart",   I_STRING(datastart),
	     "datastart for user--prog"},
	{'s', "syscsimul",   I_FLAG(OPT_SIMULATE_SYSC),
		 "simulate some syscalls (open(), write()...)"},
	{'i', "illsimul",    I_FLAG(OPT_SIMULATE_ILLEGAL),
		 "stop at illegal and print message"},
	{'t', "trace",       I_STRING(TraceFileName), ""},
	{'S', "shared",      I_FLAG(OPT_DEV_MAPPING),
		 "use shared mem interface to /dev/pram"},
    {'F', "fmapped",     I_FLAG(OPT_FILE_MAPPING),
		 "use shared mem interface to a file (/tmp/dev_pram)"},
    {'U', "uart",        I_FLAG(OPT_UART),
          "simulate uart for each pp (only with X-Win)"},
    {'R', "randomize",   I_FLAG(OPT_RANDOMIZE),
          "fill allocated memory & registers with random values"},
    {' ', "lazymalloc", I_FLAG(OPT_LAZY_MALLOC),
		  "do a lazy malloc for the global memory"},
	{' ', "extern-net",  I_FLAG(OPT_NET|OPT_EXTERN_NET),
		 "enable external network test-software"},
	{' ', "net-init",    I_FLAG(OPT_NET|OPT_NET_INIT),
		 "enable stg-packet generation on loading files"},
	{' ', "net-debug",   I_FLAG(OPT_NET|OPT_NET_DEBUG),
		 "dump all network packets"},
	{' ', "net-op-test", I_FLAG(OPT_NET|OPT_OP_TEST),
		 "test for network errors (different ops on same cell)"},
	{' ', "net-er",      I_FLAG(OPT_NET|OPT_OP_TEST|OPT_ER_TEST),
    	 "warning if concurrent read"},
	{' ', "net-ew",      I_FLAG(OPT_NET|OPT_OP_TEST|OPT_EW_TEST),
	      "warning if concurrent write"},
	{' ', "net-stat",    I_FLAG(OPT_NET|OPT_NET_STAT),
		 "network statistics"},
	{' ', "test-slow",   I_FLAG(OPT_TEST_SLOW),
		 "just to benchmark the slower simulation routine"},
};

#define command_opt_len (sizeof(command_opt)/sizeof(*command_opt))

void usage(const char *format, ...)
{
	int i;
	va_list arg;
	struct option_desc *co;

	if(format)
	{
		va_start(arg, format);
		vfprintf(stderr, format, arg);
		va_end(arg);
	}
	else
	{
		for(co=command_opt, i=0; i<command_opt_len; i++, co++)
		{
			fprintf(stderr, "%c  %c%c --%-15s  %s\n",
					(co->type==O_FLAG && !co->val.flags)? 'd': ' ',
					co->short_opt==' '? ' ': '-',
					co->short_opt, co->long_opt, co->comment);
		}
	}
	exit(format? 1: 0);
}

int eval_params(int argc, char *argv[])
{ 
	int par, i;
	
	for(par=1; par<argc; par++)
	{
		if(argv[par][0] != '-') break;

		if(argv[par][1] == '-')
		{
			for(i=0; i<command_opt_len; i++)
				if(!strcmp(argv[par]+2, command_opt[i].long_opt))
					break;
		}
		else
		{
			for(i=0; i<command_opt_len; i++)
				if(argv[par][1]==command_opt[i].short_opt)
					break;
		}
		if(i>=command_opt_len)
			usage("unknown option '%s'\n", argv[par]);
		else
		{
			switch(command_opt[i].type)
			{
			case O_INT:
				if(1!=sscanf(argv[++par], "%i",
							 command_opt[i].val.int_val))
					usage("Option --%s expects a number\n",
						  command_opt[i].long_opt);
				break;
			case O_STRING:
				*command_opt[i].val.string_val = argv[++par];
				break;
			case O_FLAG:
				if(!command_opt[i].val.flags)
					usage("Option --%s disabled\n",
						  command_opt[i].long_opt);
				else if(command_opt[i].val.flags > 0)
					options |= command_opt[i].val.flags;
				else
					options &= command_opt[i].val.flags;
				break;
			case O_FUNCTION:
				(command_opt[i].val.function)(argv[++par]);
				break;
			case O_HELP:
				usage(NULL);
				break;
			}
		}
	}
	return par;
}

int read_defaults(int mode)
{
  char config_string[80];
  int success;
  int argc=0;
  static char *argv[32];
  FILE *configfile;
  static int par;
  
  if((configfile=fopen(".pramsimrc", "r"))==NULL)
	  if((configfile=fopen(dflt_cfg_file, "r"))==NULL)
	  {
		  sim_print(IO_NOTICE, "Notice: running without config--file.\n");
		  return -1;
	  }
  
  success=fscanf(configfile,"%s", config_string);
  while(success!=-1)
	{
	  argv[++argc] = strdup(strtok(config_string, " \n\t"));
	  success=fscanf(configfile,"%s", config_string);
	}
  argc++;

  if(!mode)
	par = eval_params(argc, argv);
 
  if((par<argc) && mode)
	if(load_file(argc-par, &argv[par])) exit(1);

  fclose(configfile);

  return 0;
}

void open_uart_term(int ppnum, struct ppregs *pp)
{
#ifdef MOD_UART
	int fds, fdm;
	char *pty;
	char arg[20], arg2[10];
	char null[10];
	pid_t pid;

	fdm = open("/dev/ptmx", O_RDWR);
	grantpt(fdm);
	unlockpt(fdm); 
	pty = ptsname(fdm);
	if((fds = open(pty, O_RDWR | O_NDELAY | O_NONBLOCK))>=0)
	{
		ioctl(fds, I_PUSH, "ptem");        /* push ptem */
#if 0
    	ioctl(fds, I_PUSH, "ldterm");      /* push ldterm */
#endif
		ioctl(fds, I_PUSH, "charproc");
		
		pp->UartRegs.fd = fds;
		if(strlen(strrchr(pty, '/')+1)==1)
			sprintf(arg, "-S0%s%d", (strrchr(pty, '/')+1), fdm);
		else
			sprintf(arg, "-S%s%d", (strrchr(pty, '/')+1), fdm);
		sprintf(arg2, "%d", ppnum);
		sim_print(IO_NOTICE, "%s %s %d %d\n", arg, pty, fdm, fds);
		pid = fork();
		if(pid<0)
		{
			perror("fork");
			exit(1);
		}
		if(pid==0)
		{
			close(0);
			close(1);
			close(2);
/*			ioctl(fds, I_FLUSH, FLUSHRW); */
			close(fds);
			execlp("pram-term", "pram-term", arg, arg2, (char *)0);
/*			execlp("xterm", "xterm", arg, "-rv", 
				   "-fg", "white", "-bg", "black", 
				   "-l", (char *)0); */
			sim_print(IO_STDERR, "Error while execlp\n");
			_exit(1);
		}
		pp->UartRegs.pid = pid;
		
		close(fdm);

		while(read(fds, null, 8)!=8)
			;
	}
	else
		sim_print(IO_STDERR, "No more pts\n");
	return;
#else
	sim_print(IO_STDERR, "Uart not supported\n");
    exit(5);
#endif
}

int init(void)
{
	int ppn, vpn, i;
	struct ppregs *pp;
	struct vpregs *vp;
	UWORD *LMptr;

	signal(SIGFPE, SIG_IGN);

	sim_print(IO_NOTICE, "       ***** PRAM-SIMULATOR (v2.0) *****\n");
	sim_print(IO_NOTICE, "          >>>>>>>> DAMN FAST <<<<<<<<\n");
	sim_print(IO_NOTICE, "       (c) by hirbli & stefran  07.10.94\n");
	sim_print(IO_NOTICE, "You have %d physical processors with %d "
			  "vP's each \n\n", PPnum, VPnum);

	IFOPT(OPT_RANDOMIZE) srand(time(NULL));

    NEW(PPnum, struct ppregs, PhysicalProcessor);
	NEW(PPnum * VPnum, struct vpregs, VirtualProcessor);
	NEW(LocalMemLen * PPnum, UWORD, LocalMem);
	ldesc = (struct descriptor *(*)[LFILEANZ])
		NEW_NULL(PPnum * VPnum * LFILEANZ, void *);
	IFOPT(OPT_DEV_MAPPING)
	{
		ProgramMem = NEW_DEVPROG(ProgramMemLen, UWORD);
		GlobalMem = NEW_DEVPRAM(GlobalMemLen + 1, UWORD);
		ExternalInterrupts = GlobalMem + GlobalMemLen;
	}
	else IFOPT(OPT_FILE_MAPPING)
	{
		ProgramMem = NEW_SMFILE(ProgramMemLen, UWORD, "/tmp/dev_pramprg");
		GlobalMem = NEW_SMFILE(GlobalMemLen, UWORD, "/tmp/dev_pram");
		ExternalInterrupts = NEW_SMFILE(1, UWORD, "/tmp/dev_extint");
	}
	else IFOPT(OPT_LAZY_MALLOC)
	{
		NEW(ProgramMemLen, UWORD, ProgramMem);
		GlobalMem = NEW_SMFILE(GlobalMemLen, UWORD, "/dev/zero");
		NEW(1, UWORD, ExternalInterrupts);
	}
	else
	{
		NEW(ProgramMemLen, UWORD, ProgramMem);
		NEW(GlobalMemLen, UWORD, GlobalMem);
		NEW(1, UWORD, ExternalInterrupts);
	}
	if(!PhysicalProcessor || !VirtualProcessor || !LocalMem || !ldesc
	   || !ProgramMem || !GlobalMem)
	{
		sim_print(IO_STDERR, "Not enough memory\n");
		return 1;
	}

	*ExternalInterrupts = 0x100;
	LMptr = LocalMem;
	for(ppn=0, pp=PhysicalProcessor; ppn<PPnum; ppn++, pp++)
	{
		pp->LocalMem = LMptr;
		LMptr += LocalMemLen;
	}
	for(i=0; i<4096; i++) pnummap[i] = -1;
	for(vpn=0, vp=VirtualProcessor; vpn<VPnum*PPnum; vpn++, vp++)
	{
		ppn = (vpn / KRnum) % PPnum;
		vp->pp = PhysicalProcessor + ppn;
		vp->nr = (vpn % KRnum) + KRnum * (vpn / (PPnum * KRnum));
		pnummap[(ppn<<5) + vp->nr] = vpn;
	}
	for(i=0; i<FILEANZ; i++)
		desc[i].opencnt = 0;
	IFOPT(OPT_UART)
		for(ppn=0, pp=PhysicalProcessor; ppn<PPnum; ppn++, pp++)
			open_uart_term(ppn, pp);
	
	NumBreaks = 0;
	
	return 0;

}

void reset(void)
{
	int ppn, vpn;
	struct ppregs *pp;
	struct vpregs *vp;
	
	ModuloBit = 1;
	for(ppn=0, pp=PhysicalProcessor; ppn<PPnum; ppn++, pp++)
	{
		IFOPT(OPT_RANDOMIZE)
			;
		else
			pp->NR = ppn << 10;
		pp->CT = 0;
		pp->A = 1;
		pp->MOD = GLSUPER|GLMASK;
		pp->MOD2 = pp->MOD;
		pp->ExtEx = 0x100;
		pp->UartRegs.LSR = 0x60;
		pp->FpgaRegs.MR = 0;        /* ###       */
		pp->ctex = 0;
		pp->testex = 0;
		
	}
	for(vpn=0, vp=VirtualProcessor; vpn<VPnum*PPnum; vpn++, vp++)
	{
		vp->R[0] = 0;
		vp->R[1] = 0; /* PC=0 */
		vp->SR=0;
		vp->testex = 0;
		vp->dl_regno = -1;
	}
	sysc_reset();
}

/* extern l_mem_type *lok_mem; */

int load(FILE *fp)
{
	char buf[100];
	int len=0, ppn;
	UWORD offset=0, val;
	enum section {S_none, S_Program, S_Global, S_Local} section=S_none;

	while(1)
	{
		switch(section)
		{
		case S_Program:
			len=ProgramMemLen-offset;
			while(--len>=0 && 1==fscanf(fp, " 0x%x", &val))
				ProgramMem[offset++] = val;
			break;
		case S_Global:
			len=GlobalMemLen-offset;
			IFOPT(OPT_NET_INIT)
			{
				int vpn=0, mb=ModuloBit;
				while(--len>=0 && 1==fscanf(fp, " 0x%x", &val))
				{
					GlobalMem[offset] = val;
					EmitNetPack(MOD_ST, OP_NONE, offset, offset, val,
								VirtualProcessor+vpn);
					if(++vpn>=VPnum*PPnum)
					{
						vpn=0;
						SimulateNetRound(ModuloBit);
						ModuloBit ^= 1;
					}
					offset++;
				}
				do
				{
					SimulateNetRound(ModuloBit);
					ModuloBit ^= 1;
				}
				while(mb!=ModuloBit);
			}
			else
			{
				while(--len>=0 && 1==fscanf(fp, " 0x%x", &val))
					GlobalMem[offset++] = val;
			}
			break;
		case S_Local:
			len=LocalMemLen-offset;
			while(--len>=0 && 1==fscanf(fp, " 0x%x", &val))
			{
				for(ppn = 0; ppn < PPnum; ppn++)
					PhysicalProcessor[ppn].LocalMem[offset] = val;
				offset++;
			}
			break;
		default:
			len = 0;
			break;
		}
		if(len<0)
		{
			sim_print(IO_STDERR, "cod-file too long\n");
			return 0;
		}
	    if(1==fscanf(fp, " %s", buf))
		{
			if(!strcmp(buf, ".CODE")) section = S_Program;
			else if(!strcmp(buf, ".GDATA")) section = S_Global;
			else if(!strcmp(buf, ".LDATA")) section = S_Local;
			else
			{
				sim_print(IO_STDERR, "unknown directive '%s'\n", buf);
				return 0;
			}

			fscanf(fp, " %s", buf);
			if(!strcmp(buf, ".ORIGIN"))
			{
				if(1!=fscanf(fp, " 0x%x", &val))
				{
					sim_print(IO_STDERR, ".ORIGIN: missing parameter\n");
					return 0;
				}
				offset = val;

				switch(section)
				{
				case S_Program:
				  if(offset>ProgramMemLen)
					{
					  ProgramMem = pram_realloc(ProgramMem, 
												ProgramMemLen, offset+100);
					  ProgramMemLen = offset+100;
					}
				  break;
				case S_Global:
				  if(offset>GlobalMemLen)
					{
					  GlobalMem = pram_realloc(GlobalMem, 
											   GlobalMemLen, offset+100);
					  GlobalMemLen = offset+100;
					}
				  break;
				case S_Local:
/*				  if(offset>LocalMemLen)
					LocalMem = pram_realloc(LocalMem, 
										  LocalMemLen, offset); */
				  break;
				default:

				  break;
				}
			}
			else
			{
				sim_print(IO_STDERR, "unknown directive '%s'\n", buf);
				return 0;
			}
		}
		else return 1;
	}
}

int de_init(void)
{
    int ppn;
	struct ppregs *pp;

	IFOPT(OPT_TRACE)
		pclose(TraceFile);
	IFOPT(OPT_DEV_MAPPING)
	{
		pram_free(GlobalMem);
		pramprg_free(ProgramMem);
	}
	else IFOPT(OPT_FILE_MAPPING)
	{
		pram_funmap(GlobalMem, "/tmp/dev_pram");
		pram_funmap(ProgramMem, "/tmp/dev_pramprg");
		pram_funmap(ExternalInterrupts, "/tmp/dev_extint");
	}
	else IFOPT(OPT_LAZY_MALLOC)
    {
	    pram_funmap(GlobalMem, "/dev/zero");
		free(ProgramMem);
    }
    if(LocalMem)          
		free(LocalMem);
	if(PhysicalProcessor) 
		free(PhysicalProcessor);
	if(VirtualProcessor)  
		free(VirtualProcessor);
	if(ldesc) 
		free(ldesc);
	if(ExternalInterrupts)
		free(ExternalInterrupts);
	
	IFOPT(OPT_UART)
		for(ppn=0, pp=PhysicalProcessor; ppn<PPnum; ppn++, pp++)
		{
			close(pp->UartRegs.fd);
			kill(pp->UartRegs.pid, SIGKILL);
		}
	LocalMemLen   = 1024;
	GlobalMemLen  = 135168;
	ProgramMemLen = 2048;

	return 0;
}

int coff_load(int argc, char *argv[])
{
	int status, i=0, largc=0;
	char vps[20];
	char *largv[200];
	char *file[1];
	
	sim_print(IO_NOTICE, "Relocating file %s...", argv[0]);
	fflush(stdout);
	fflush(stderr);

	sprintf(vps, "%d\0", VPnum * PPnum);
	
	largv[largc++]=strdup("loader");
	largv[largc++]=strdup("-o");
	largv[largc++]=strdup(tmpnam((char *)NULL));
	file[0]=malloc(strlen(largv[largc-1])+5);
	strcpy(file[0], largv[largc-1]);
	strcat(file[0], ".cod");

	if(textstart)
	{
		largv[largc++]=strdup("--textstart");
		largv[largc++]=strdup(textstart);
	}
	if(datastart)
	{
		largv[largc++]=strdup("--datastart");
		largv[largc++]=strdup(datastart);
	}
	
	while(i<last_p_env)
	{
		if(p_env[i].name!=NULL)
		{
			largv[largc++]=strdup("-L");
			largv[largc++]=strdup(p_env[i].name);
			largv[largc++]=strdup(p_env[i].str_val);
		}
		i++;
	}
	
	largv[largc++]=strdup("-L");
	largv[largc++]=strdup("procs");
	largv[largc++]=strdup(vps);
  
	i = 0;
	while(argc--)
		largv[largc++] = argv[i++];
  
	largv[largc] = (char *)NULL;
    
	if(fork()==0)
		execvp("loader", largv);
	else
		wait(&status);

	sim_print(IO_NOTICE, " done\n");       /* Done Relocating */
	
	load_file(1, file);
	
	unlink(file[0]);

	return 0;
}


int load_file(int argc, char *argv[])
{
	FILE *fp;
	char magic;

	if(argc==0)
	{
		sim_print(IO_STDERR, "load_file: too few arguments\n");
		return 1;
	}
	
	fp=fopen(argv[0], "r");
	if(!fp)
	{
		if(!strcmp(argv[0], ""))
		{
			sim_print(IO_STDERR, "Warning: no cod-file loaded\n");
			return 0;
		}
		else
		{
			sim_print(IO_STDERR, "Couldn't open input-file %s\n", argv[0]);
			return 1;
		}
	}
	else
	{
		fread(&magic, sizeof(char), 1, fp);
		rewind(fp);
		
		if(magic=='.')
		{
			sim_print(IO_NOTICE, "Loading file %s...", argv[0]);
			fflush(stdout);
			fflush(stderr);
			load(fp);
			sim_print(IO_NOTICE, " done\n");
		}
		else
			coff_load(argc, argv);
		
	}
	fclose(fp);	
	return 0;
}

set_file_and_args(int argc, char *argv[])
{
	
	coff_load(argc, argv);
	
	

}

int main(int argc, char **argv)
{
	int par, ppn;
	struct ppregs *pp;

	VPnum = 2;
	PPnum = 2;
	KRnum = 1;
	LocalMemLen   = 1024;
	GlobalMemLen  = 135168;
	ProgramMemLen = 2048;

	options = 0;
			
	read_defaults(0);
	
	par = eval_params(argc, argv);
	
	last_p_env = 0;
	
	if(init())
		exit(1);

	read_defaults(1);

	load_argc = argc-par;
	load_argv = &argv[par];
	load_file(load_argc, load_argv);

	IFOPT(OPT_NET)
		InitNet(PPnum, VPnum);
	IFOPT(OPT_NET_STAT)
		memset((char *)&NetStat[0][0].last_ct, 0, 
			   8*15*sizeof(struct NetStatistics));

	if(TraceFileName)
	{
		char *tfn = alloca(strlen(TraceFileName)+10);
		sprintf(tfn, "gzip >%s", TraceFileName);
		TraceFile = popen(tfn, "w");
		if(TraceFile)
		{
			UWORD bla = PPnum * VPnum;
			fwrite(&bla, sizeof(UWORD), 1, TraceFile);
			options |= OPT_TRACE;
		}
		else
			sim_print(IO_STDERR, "Couldn't open tracepipe %s.\n", tfn);
	}

	reset();
	do_cmd();

	de_init();
	
	return 0;
}
