/* -*-mb0-c-*-

   SB-PRAM simulator

   (C) 1994 by Michael Bosch (hirbli@cs.uni-sb.de)
   and Stefan Franziskus (stefran@cs.uni-sb.de)

   Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee is hereby granted, provided
   that the above copyright notice appear in all copies.
*/

#include "config.h"

static const char sccsid[] = "@(#)command.c	1.34";

#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <setjmp.h>
#include <ctype.h>
#include <stdlib.h>

#include "processor.h"
#include "glovars.h"
#include "simul.h"
#include "interface.h"

static void disass(UWORD *mem, UWORD *adr, UWORD *end)
{
	UWORD Command;
	UWORD breakptr;
	char out_str[80];

	while (adr<end)
	{  
		Command=*adr;
		if((Command>>OpcodePos) !=CMD_BREAK)
		{
			disassemble(Command, adr-mem, out_str);
			sim_print(IO_DISASS, "%04x %08X  %s\n", adr-mem, Command, out_str);
		}
		else
		{
			breakptr = GetBreak;
			disassemble(BreakPoint[breakptr].opcode, adr-mem, out_str);
			sim_print(IO_DISASS, "%04x %08X  %s\t\tBREAK %04x\n", 
				   adr-mem, Command, out_str, breakptr);
		}
		
  		adr++;
	}
}

static void unassemble(UWORD adr)
{
	disass(ProgramMem, ProgramMem+adr, ProgramMem+adr+1);
}

#define TRENNER " \n\t,"
#define nolines 22

enum arg {ARG_OPT, ARG_PROZ, ARG_RANGE, ARG_VAL, ARG_FILE, ARG_REG, ARG_END};
static const char *arg_name[] =
{
	"&", "PROZ", "MEM", "VAL", "FILE", "REG"
};
static int arg_len[] =
{
	1, 1, 3, 1, 1, 1, 0
};

enum speicher {MEM_GLOBAL, MEM_LOCAL, MEM_PROGRAM};

static struct icmd
{
	const char *cmd;
	int args[10];
	const char *help;
}
icmds[] =
{
#define CMD_DESCR 1
#include "sim_cmds.h"
};

enum sim_cmd 
{
#define CMD_NAME(x) CMD_LABEL_##x,
#include "sim_cmds.h"
};

#define COUNT(x) (sizeof(x)/sizeof(*x))

union parameter
{
	int val;
	struct reg
	{
		short reg;
		short is_pp;
	} reg;
	char *file;
	UWORD *adr;
};

static jmp_buf error_jmp_buf;
static int AktVP;

static void error(const char *s)
{
	sim_print(IO_NOTICE, "Error: %s\n", s);
	longjmp(error_jmp_buf, 1);
}


static unsigned rdnum(char *s)
{
	unsigned val;
	char *end;
	switch(*s)
	{
	case '%':
		val = (unsigned long) strtol(s+1, &end, 2);
		break;
	case '$':
		val = (unsigned long) strtol(s+1, &end, 16);
		break;
	case '#':
		val = (unsigned long) strtol(s+1, &end, 10);
		break;
	case '0':
		if(s[1]=='x') val = (unsigned long) strtol(s+2, &end, 16);
		else val = strtol(s, &end, 8);
		break;
	default:
		val = (unsigned long) strtol(s, &end, 10);
		break;
	}
	return val;
}

static char conv(UWORD c)
{
	if(c>256) return ' ';
	if(isprint(c)) return c;
	else return '.';
}

static int command(char *cmd)
{
	char *s;
	int arg_nr, rarg_nr, *argp, val, reg, command_nr, optional;
	union parameter par[10];
	static int default_val[3], old_command;

	if(setjmp(error_jmp_buf))
	{
		return 1;
	}

	s = strtok(cmd, TRENNER);
	if(s)
	{
		for(command_nr=0; command_nr<COUNT(icmds); command_nr++)
			if(!strcmp(s, icmds[command_nr].cmd)) break;
	}
	else command_nr = old_command;
	if(command_nr==COUNT(icmds))
	{
		sim_print(IO_NOTICE, "no such command '%s'\n", s);
		return 1;
	}
	else old_command = command_nr;
	optional = 0;
	rarg_nr = -1;
	for(arg_nr=0, argp = icmds[command_nr].args; *argp!=ARG_END;)
	{
		if(*argp==ARG_OPT)
		{
			optional = 1;
			argp++;
		}
		if(rarg_nr == -1)
		{
			s = strtok(NULL, TRENNER);
			if(!s)
			{
				if(!optional) error("Nicht genug Argumente");
				rarg_nr = arg_nr;
			}
		}
		switch(*argp++)
		{
		case ARG_PROZ:
			if(s)
			{
				val = rdnum(s);
				if((unsigned)val >= PPnum*VPnum)
					error("Prozessornummer zu gro");
			}
			else val = AktVP;
			par[arg_nr++].val = val;
			break;
		case ARG_RANGE: {
			char *s2;
			int pnr, beg, end, dmem, mem, epl, memlen;
			UWORD *membeg;
			dmem = mem = *argp++;
			epl = *argp++;
			if(s)
			{
				s2 = strchr(s, ':');
				if(s2)
				{
					*s2 = '\0';
					if(!strcasecmp("G", s)) mem = MEM_GLOBAL;
					else if(!strcasecmp("P", s)) mem = MEM_PROGRAM;
					else if(*s=='L'||*s=='l') pnr=rdnum(s+1), mem = MEM_LOCAL;
					else error("fehlerhafte Bereichsangabe");
					s = s2 + 1;
				}
				beg = rdnum(s);
				s2 = strchr(s, '-');
				if(s2) end = rdnum(s2+1);
				else if(epl>=0) end = beg+epl*nolines;
				else error("Diese Bereichsangabe muss vollstndig sein");
			}
			else
			{
				beg = default_val[dmem];
				if(epl>=0) end = beg+epl*nolines;
				else error("Diese Bereichsangabe muss vollstndig sein");
			}
			switch(mem)
			{
			case MEM_GLOBAL:
				membeg = GlobalMem;
				memlen = GlobalMemLen;
				break;
			case MEM_PROGRAM:
				membeg = ProgramMem;
				memlen = ProgramMemLen;
				break;
			case MEM_LOCAL:
				membeg = VirtualProcessor[pnr].pp->LocalMem;
				memlen = LocalMemLen;
				break;
			}
			if((unsigned)beg>(unsigned)memlen)
				error("Begin des Bereichs oberhalb der Speichergrenze");
			if((unsigned)end>(unsigned)memlen)
				end = memlen;
			if(dmem == mem) default_val[dmem] = end;
			par[arg_nr++].adr = membeg;
			par[arg_nr++].adr = membeg + beg;
			par[arg_nr++].adr = membeg + end;
			break;
		}
		case ARG_VAL:
			par[arg_nr++].val = s? rdnum(s): 0;
			break;
		case ARG_FILE:
			par[arg_nr++].file = s;
			break;
		case ARG_REG:
			if((reg=VPReg_Search(s))!=-1)
			{
				par[arg_nr].reg.reg = reg;
				par[arg_nr++].reg.is_pp = 0;
			}
			else if((reg=PPReg_Search(s))!=-1)
			{
				par[arg_nr].reg.reg=reg;
				par[arg_nr++].reg.is_pp = 1;
			}
			else error("Kein gltiger Registername");
			break;
		}
	}
	if(rarg_nr == -1) rarg_nr = arg_nr;

#define GET_PAR_OR_DEFAULT(field, nr, deflt) \
	(((nr)<rarg_nr)? par[nr].field: (deflt))

	switch(command_nr)
	{
	default:
#define CMD_CODE 1
#define CMD_NAME(x) break; case CMD_LABEL_##x:
#include "sim_cmds.h"
		break;
	}
	return 0;
}

#ifndef USE_READLINE

void do_cmd(void)
{
	char buf[100];
	while(1)
	{
		sim_print(IO_STDERR, "\033[1mPRAM P%d = (p%d, v%d)>\033[0m ", 
				  AktVP,
				  VirtualProcessor[AktVP].pp-PhysicalProcessor,
				  VirtualProcessor[AktVP].nr);
		if(!fgets(buf, sizeof(buf), stdin)) break;
		if(2==command(buf)) break;
	}
}

#else /* !USE_READLINE */

#include <readline/readline.h>
#include <readline/history.h>

char *completion()
{
	return NULL;
}

void do_cmd(void)
{
	while(1)
	{
		char prompt[20], *line;
		rl_completion_entry_function = (Function *)completion;
		sprintf(prompt, "\033[1mPRAM P%d = (p%d, v%d)>\033[0m ",
				AktVP,
				VirtualProcessor[AktVP].pp-PhysicalProcessor,
				VirtualProcessor[AktVP].nr);
		line = readline(prompt);
		if (line)
		{
			add_history(line);
			if(2==command(line)) break;
			free(line);
		}
		else
		{
			printf("^D\n");
			break;
		}
	}
}

#endif /* !USE_READLINE */
