/* -*-mb0-c-*-

   SB-PRAM simulator

   (C) 1994 by Michael Bosch (hirbli@cs.uni-sb.de)
   and Stefan Franziskus (stefran@cs.uni-sb.de)

   Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee is hereby granted, provided
   that the above copyright notice appear in all copies.
*/

static const char copyright[] =
	"@(#)PRAM simulator (C) 1994 by hirbli & stefran";
static const char sccsid[] = "@(#)init.c	1.41";

#include <stdarg.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <signal.h>

#include "init.h"
#include "processor.h"
#include "simul.h"
#include "glovars.h"
#include "net.h"

#define NEW(COUNT, TYPE) (TYPE *)malloc((COUNT) * sizeof(TYPE))

#ifdef MOD_DEV_MAPPING
#include "/home/dst/pram/pram/src/pram.h"
#endif
extern void *pram_malloc(unsigned size);
extern void *pramprg_malloc(unsigned size);
extern void pram_free(void *ptr);
extern void pramprg_free(void *ptr);
#define NEW_DEVPRAM(COUNT, TYPE) (TYPE *)pram_malloc((COUNT) * sizeof(TYPE))
#define NEW_DEVPROG(COUNT, TYPE) (TYPE *)pramprg_malloc((COUNT) * sizeof(TYPE))

char *TraceFileName = NULL;

struct option_desc command_opt[] =
{
	{'h', "help",        O_HELP, {}, ""},
	{'M', "globmem",     I_INT(&GlobalMemLen),
		 "length of global memory [words]"},
	{'P', "progmem",     I_INT(&ProgramMemLen),
		 "length of program memory [words]"},
	{'L', "locmem",      I_INT(&LocalMemLen),
		 "length of local memory [words/PP]"},
	{'v', "virtProz",    I_INT(&VPnum),
		 "# of virtual processors (VPs) per PP"},
	{'p', "physProz",    I_INT(&PPnum),
		 "# of physical processors (PPs)"},
	{'l', "littleRound", I_INT(&KRnum),
		 "# of VP per little round (for GETNR only)"},
	{'s', "syscsimul",   I_FLAG(OPT_SIMULATE_SYSC),
		 "simulate some syscalls (open(), write()...)"},
	{'i', "illsimul",    I_FLAG(OPT_SIMULATE_ILLEGAL),
		 "stop at illegal and print message"},
	{'t', "trace",       I_STRING(TraceFileName), ""},
	{'S', "shared",      I_FLAG(OPT_DEV_MAPPING),
		 "use shared mem interface to /dev/pram"},
	{' ', "extern-net",  I_FLAG(OPT_NET|OPT_EXTERN_NET),
		 "enable external network test-software"},
	{' ', "net-init",    I_FLAG(OPT_NET|OPT_NET_INIT),
		 "enable stg-packet generation on loading files"},
	{' ', "net-debug",   I_FLAG(OPT_NET|OPT_NET_DEBUG),
		 "be verbose"},
	{' ', "net-op-test", I_FLAG(OPT_NET|OPT_OP_TEST),
		 "test for possible network errors and uninitialized variables"},
	{' ', "net-stat",    I_FLAG(OPT_NET|OPT_NET_STAT),
		 "calculate statistics over network-traffic"},
	{' ', "test-slow",   I_FLAG(OPT_TEST_SLOW),
		 "just to benchmark the slower simulation routine"},
};

#define command_opt_len (sizeof(command_opt)/sizeof(*command_opt))

void usage(const char *format, ...)
{
	int i;
	va_list arg;
	struct option_desc *co;

	if(format)
	{
		va_start(arg, format);
		vfprintf(stderr, format, arg);
		va_end(arg);
	}
	else
	{
		for(co=command_opt, i=0; i<command_opt_len; i++, co++)
		{
			fprintf(stderr, "%c  %c%c --%-15s  %s\n",
					(co->type==O_FLAG && !co->val.flags)? 'd': ' ',
					co->short_opt==' '? ' ': '-',
					co->short_opt, co->long_opt, co->comment);
		}
	}
	exit(format? 1: 0);
}

int eval_params(int argc, char *argv[])
{ 
	int par, i;
	
	options = 0;

	for(par=1; par<argc; par++)
	{
		if(argv[par][0] != '-') break;

		if(argv[par][1] == '-')
		{
			for(i=0; i<command_opt_len; i++)
				if(!strcmp(argv[par]+2, command_opt[i].long_opt))
					break;
		}
		else
		{
			for(i=0; i<command_opt_len; i++)
				if(argv[par][1]==command_opt[i].short_opt)
					break;
		}
		if(i>=command_opt_len)
			usage("unknown option '%s'\n", argv[par]);
		else
		{
			switch(command_opt[i].type)
			{
			case O_INT:
				if(1!=sscanf(argv[++par], "%i",
							 command_opt[i].val.int_val))
					usage("Option --%s expects a number\n",
						  command_opt[i].long_opt);
				break;
			case O_STRING:
				*command_opt[i].val.string_val = argv[++par];
				break;
			case O_FLAG:
				if(!command_opt[i].val.flags)
					usage("Option --%s disabled\n",
						  command_opt[i].long_opt);
				else if(command_opt[i].val.flags > 0)
					options |= command_opt[i].val.flags;
				else
					options &= command_opt[i].val.flags;
				break;
			case O_FUNCTION:
				(command_opt[i].val.function)(argv[++par]);
				break;
			case O_HELP:
				usage(NULL);
				break;
			}
		}
	}
	return par;
}


int init(void)
{
	int ppn, vpn, i;
	struct ppregs *pp;
	struct vpregs *vp;
	UWORD *LMptr;

	signal(SIGFPE, SIG_IGN);

	fprintf(stderr, "       ***** PRAM-SIMULATOR (v2.0) *****\n");
	fprintf(stderr, "          >>>>>>>> DAMN FAST <<<<<<<<\n");
	fprintf(stderr, "       (c) by hirbli & stefran  07.10.94\n");
	fprintf(stderr, "You have %d physical processors with %d vP's each \n\n",
		   PPnum, VPnum);
	
	PhysicalProcessor = NEW(PPnum, struct ppregs);
	VirtualProcessor = NEW(PPnum * VPnum, struct vpregs);
	LocalMem = NEW(LocalMemLen * PPnum, UWORD);
	ldesc = NEW(PPnum * VPnum * LFILEANZ, void *);
	IFOPT(OPT_DEV_MAPPING)
	{
		ProgramMem = NEW_DEVPROG(ProgramMemLen, UWORD);
		GlobalMem = NEW_DEVPRAM(GlobalMemLen + PPnum + 1, UWORD);
	}
	else
	{
		ProgramMem = NEW(ProgramMemLen, UWORD);
		GlobalMem = NEW(GlobalMemLen + PPnum + 1, UWORD);
	}
	if(!PhysicalProcessor || !VirtualProcessor || !LocalMem || !ldesc
	   || !ProgramMem || !GlobalMem)
	{
		fprintf(stderr, "Not enough memory\n");
		return 1;
	}
	ExternalInterrupts = GlobalMem + GlobalMemLen + 1;
	for(ppn=-1; ppn<PPnum; ppn++) ExternalInterrupts[ppn] = -1;
	LMptr = LocalMem;
	for(ppn=0, pp=PhysicalProcessor; ppn<PPnum; ppn++, pp++)
	{
		pp->LocalMem = LMptr;
		LMptr += LocalMemLen;
	}
	for(i=0; i<4096; i++) pnummap[i] = -1;
	for(vpn=0, vp=VirtualProcessor; vpn<VPnum*PPnum; vpn++, vp++)
	{
		ppn = (vpn / KRnum) % PPnum;
		vp->pp = PhysicalProcessor + ppn;
		vp->nr = (vpn % KRnum) + KRnum * (vpn / (PPnum * KRnum));
		pnummap[(ppn<<5) + vp->nr] = vpn;
	}
	for(i=0; i<FILEANZ; i++)
		desc[i].opencnt = 0;
	return 0;
}

void reset(void)
{
	int ppn, vpn;
	struct ppregs *pp;
	struct vpregs *vp;
	
	ModuloBit = 1;
	for(ppn=0, pp=PhysicalProcessor; ppn<PPnum; ppn++, pp++)
	{
		pp->NR = ppn << 10;
		pp->CT = 0;
		pp->A = 1;
		pp->MOD = GLSUPER|GLMASK;
		pp->MOD2 = pp->MOD;
		pp->ExtEx = -1;
	}
	for(vpn=0, vp=VirtualProcessor; vpn<VPnum*PPnum; vpn++, vp++)
	{
		vp->R[0] = 0;
		vp->R[1] = 0; /* PC=0 */
		vp->testex = 0;
	}
	sysc_reset();
}

/* extern l_mem_type *lok_mem; */

int load(FILE *fp)
{
	char buf[100];
	int len;
	UWORD offset=0, val;
	enum section {S_none, S_Program, S_Global} section=S_none;

	while(1)
	{
		switch(section)
		{
		case S_Program:
			len=ProgramMemLen-offset;
			while(--len>=0 && 1==fscanf(fp, " 0x%x", &val))
				ProgramMem[offset++] = val;
			break;
		case S_Global:
			len=GlobalMemLen-offset;
			IFOPT(OPT_NET_INIT)
			{
				int vpn=0, mb=ModuloBit;
				while(--len>=0 && 1==fscanf(fp, " 0x%x", &val))
				{
					GlobalMem[offset] = val;
					EmitNetPack(MOD_ST, OP_NONE, offset, offset, val,
								VirtualProcessor+vpn);
					if(++vpn>=VPnum*PPnum)
					{
						vpn=0;
						SimulateNetRound(ModuloBit);
						ModuloBit ^= 1;
					}
					offset++;
				}
				do
				{
					SimulateNetRound(ModuloBit);
					ModuloBit ^= 1;
				}
				while(mb!=ModuloBit);
			}
			else
			{
				while(--len>=0 && 1==fscanf(fp, " 0x%x", &val))
					GlobalMem[offset++] = val;
			}
			break;
		default:
			len = 0;
			break;
		}
		if(len<0)
		{
			fprintf(stderr, "cod-file too long\n");
			return 0;
		}
		if(1==fscanf(fp, " %s", buf))
		{
			if(!strcmp(buf, ".CODE")) section=S_Program;
			else if(!strcmp(buf, ".GDATA")) section=S_Global;
			else if(!strcmp(buf, ".ORIGIN"))
			{
				if(1!=fscanf(fp, " 0x%x", &val))
				{
					fprintf(stderr, ".ORIGIN: missing parameter\n");
					return 0;
				}
				offset = val;
			}
			else
			{
				fprintf(stderr, "unknown directive '%s'\n", buf);
				return 0;
			}
		}
		else return 1;
	}
}

int load_file(const char *filename)
{
	FILE *fp;
	fp=fopen(filename, "r");
	if(!fp)
	{
		if(!strcmp(filename, ""))
		{
			fprintf(stderr, "Warning: no cod-file loaded\n");
			return 0;
		}
		else
		{
			fprintf(stderr, "Couldn't open cod-file\n");
			return 1;
		}
	}
	else
	{
		fprintf(stderr, "Loading file %s...", filename);
		fflush(stdout);
		load(fp);
		fprintf(stderr, " done\n");
	}
	fclose(fp);	
	return 0;
}

int main(int argc, char **argv)
{
	int par;
	VPnum = 2;
	PPnum = 2;
	KRnum = 1;
	LocalMemLen = 1000;
	GlobalMemLen = 1000000;
	ProgramMemLen = 50000;
	par = eval_params(argc, argv);
	if(init()) exit(1);
	IFOPT(OPT_NET)
		InitNet(PPnum, VPnum);
	IFOPT(OPT_NET_STAT)
		memset((char *)&NetStat[0][0].last_ct, 0, 8*15*sizeof(struct NetStatistics));
	while(par<argc)
		if(load_file(argv[par++])) exit(1);
	if(TraceFileName)
	{
		char *tfn = alloca(strlen(TraceFileName)+10);
		sprintf(tfn, "gzip >%s", TraceFileName);
		TraceFile = popen(tfn, "w");
		if(TraceFile)
		{
			UWORD bla = PPnum * VPnum;
			fwrite(&bla, sizeof(UWORD), 1, TraceFile);
			options |= OPT_TRACE;
		}
		else
			fprintf(stderr, "Couldn't open tracepipe %s.\n", tfn);
	}
	reset();
	do_cmd();
	IFOPT(OPT_TRACE)
		pclose(TraceFile);
	IFOPT(OPT_DEV_MAPPING)
	{
		pram_free(GlobalMem);
		pramprg_free(ProgramMem);
	}
	return 0;
}
