/* ----------------------------------------------------------------------
 *  file name:  timeit.c
 *
 *  Programmer:  chsmith@agate.cs.unh.edu (Craig H Smith)
 *  Description: Defines timing routines and helper functions.
 *   Most important are the functions punch_in, punch_out and
 *   init_timer.
 *
 *  Date: Sun Sep 17 14:49:13 1995
 *
 *  Modification History
 * 
 * ----------------------------------------------------------------------- */

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <errno.h>
#include <string.h>

#include "timeit.h"

/* -------------------- Tue Sep 12 13:40:34 1995 --------------------
 * function:	init_timer
 * arguments:	
 * returns:	
 * description:	Set the count down timer to aproximately 166 hours.
 *	
 */

void init_timer()
{
  itimerval_t val, oval;

  val.it_value.tv_sec = 10000;
  val.it_value.tv_usec = 0;
  val.it_interval.tv_sec = 0;
  val.it_interval.tv_usec = 0;
  if (setitimer(ITIMER_REAL, &val, &oval) != 0)
  {
    perror("setitimer failed");
    exit(EXIT_FAILURE);
  }
}

/* -------------------- Fri Sep 15 22:15:10 1995 --------------------
 * function:	clear_timer
 * arguments:	itimerval_t * (IN)
 * returns:	
 * description:	clear the memory for the structure passed in.
 *	
 */

void clear_timer(itimerval_t *p)
{
  memset((void *)p, 0, sizeof(*p));
}

/* -------------------- Sun Sep 17 13:56:44 1995 --------------------
 * function:	punch_in
 * arguments:	itimerval_t * (OUT) - receives the current value of the timer.
 * returns:	
 * description:	Call this function to start recording a region of code.
 *   This function merely takes a snap shot of the current timer.
 *	
 */
void punch_in(itimerval_t *ptimer)
{
  /*
	 Grab the current time and store it in *ptimer.
  */

  if ( getitimer(ITIMER_REAL, ptimer) != 0 )
  {
	error("getitimer");
  }
}

/* -------------------- Sun Sep 17 13:58:27 1995 --------------------
 * function:	 punch_out
 * arguments:	itimerval_t (IN)  - the variable punched in on.
 *              itimerval_t (OUT) - keeps the elapsed time for matching
 *                punch_in function.
 * returns:	
 * description:	
 *   calcuate the difference between checkin and current time,
 *   then add that time to ptimer.
 *	
 */

void punch_out(itimerval_t checkin, itimerval_t *ptimer)
{
  itimerval_t now_time;
  if ( getitimer(ITIMER_REAL, &now_time) != 0 )
  {
	error("getitimer");
  }
  now_time = subtract_time(checkin, now_time);
  *ptimer = add_time(*ptimer, now_time);
}

/* -------------------- Sun Sep 17 14:00:48 1995 --------------------
 * function:	subtract_time
 * arguments:	itimerval_t (IN) - timing value 1.
 *              itimerval_t (IN) - timing value 2.
 * returns:	difference betwee timing value 1 and 2.
 * description:	even does borrows!
 *	
 */


itimerval_t subtract_time(itimerval_t then, itimerval_t now)
{
  /*
	 then > now
	 remember that the interval counter counts DOWN time.
   */
  long secs, usecs;
  usecs = then.it_value.tv_usec - now.it_value.tv_usec;
  if ( usecs < 0 )
  {
	/* borrow a second from the subtractend */
	usecs += SEC_TO_USECS;
	then.it_value.tv_sec--;
  }
  secs = then.it_value.tv_sec - now.it_value.tv_sec ;

  /* now return the proper structure. */
  now.it_value.tv_sec = secs;
  now.it_value.tv_usec = usecs;

  return now;
}

/* -------------------- Sun Sep 17 14:01:57 1995 --------------------
 * function:	add_time
 * arguments:	itimerval_t (IN) - value 1
 *              itimerval_t (IN) - value 2
 * returns:	the sum of the two times.
 * description:	Adds the two times together.  Correctly deals
 *  with carries.
 *	
 */

itimerval_t add_time(itimerval_t t1, itimerval_t t2)
{
  /*
	 1. add the usecs together
	 2. add the seconds together, and if appropriate add the carry
	 3. set the usecs to the remainder of dividing by a second,
	    just in case step 2 did actually change the sec count.
  */
  t1.it_value.tv_usec += t2.it_value.tv_usec;
  t1.it_value.tv_sec += ((long)t1.it_value.tv_usec / (long)SEC_TO_USECS) +
	t2.it_value.tv_sec;
  t1.it_value.tv_usec %= SEC_TO_USECS;
  return t1;
}


/* -------------------- Sun Sep 17 14:02:54 1995 --------------------
 * function:	print_time
 * arguments:	char * (IN)      - qualification string
 *              itimerval_t (IN) - a duration of time
 * returns:	
 * description:	print the time preceeded by string.
 *	
 */

void print_time(char * string, itimerval_t total)
{
  printf("%s: %3ld secs %6ld usecs\n", string, total.it_value.tv_sec,
		 total.it_value.tv_usec);
}

void error(char *str)
{
    perror(str);
    exit(-1);
}
