/*************************************************/
/*      hash.c  - simple hashtable variant       */
/*      Hashtable implementation in fork95 v0.72 */
/*      15.3.1999 Oliver Fritzen                 */
/*      18.3.1999 Christoph W. Kessler           */
/*************************************************/

#include <fork.h>
#include <io.h>
#include "hashtable.h"


hashtable *new_hashtable (int (*hashFn)(void *), int Size) 
{
  /************************************/
  /*          Constructor             */
  /************************************/
  int i;
  hashtable *H;
  H = (hashtable *) shmalloc(sizeof(hashtable));
  H->Head = (HashTableItem **) shmalloc(Size*sizeof(HashTableItem));
  H->Lock = (fair_lock *)      shmalloc(Size*sizeof(fair_lock));
  for(i=0;i<Size;i++) {
    fair_lock_init(&(H->Lock[i]));
    H->Head[i] = NULL;
  }
  H->Size = Size;
  H->hashFn = hashFn;
  H->NumElts = 0;
  return H;
}


void hashtableKill (hashtable *H) 
{
  /************************************/
  /*           Destructor             */
  /************************************/
  HashTableItem *item,*item_old;
  int i;

  for(i=0; i<(H->Size);i++) { /* deleting Head Array */
    
    fair_lockup(&(H->Lock[i]));  /* ----- enter c.s. ----- */
    item = H->Head[i];
    while(item != NULL) {
      item_old = item;
      item = item->next;
      free(item_old);
    }
    fair_unlock(&(H->Lock[i]));  /* ----- leave c.s. ----- */
  }
  free(H->Head);
  
  free(H->Lock);                     /* deleting Lock Array */
  
  free(H);
}


void hashtableEnter (hashtable *H, void *entry) 
{
  /************************************/
  /*          New Entry               */
  /************************************/
  HashTableItem *item;
  int hashval;
  hashval = (*(H->hashFn))(entry) % (H->Size);
  item = (HashTableItem *) shmalloc(sizeof(HashTableItem));
  item->data = entry;
  fair_lockup(&(H->Lock[hashval]));     /* enter critical section */
  item->next = H->Head[hashval];
  H->Head[hashval]=item;
  fair_unlock(&(H->Lock[hashval]));     /* leave critical section */
  H->NumElts++;
}

void hashtableSingleEnter (hashtable *H, void *entry, 
			   int (*equals)(void *, void *)) 
{
  /************************************/
  /*  New Entry, allows entry only if */
  /*  it not already exists in hash   */
  /************************************/
  HashTableItem *item,*newItem;
  int hashval = (*(H->hashFn))(entry) % (H->Size);
  fair_lockup(&(H->Lock[hashval]));           /* enter critical section */
  item = H->Head[hashval];
  while((item!=NULL)&&(!(*equals)(item->data,entry)))
     item=item->next;
  if(!(*equals)(item->data,entry)) 
  {
    newItem = (HashTableItem *) shmalloc(sizeof(HashTableItem));
    newItem->next = H->Head[hashval];
    newItem->data = entry;
    H->Head[hashval] = newItem;
    (H->NumElts)++;
  }
  fair_unlock(&(H->Lock[hashval]));           /* leave critical section */
}


void *hashtableLookup (hashtable *H, void *entry, 
		      int (*equals)(void *, void *)) 
{
  /************************************/
  /*          Search Entry            */
  /************************************/
  HashTableItem *item;
  void *ret;
  int hashval = (*(H->hashFn))(entry) % (H->Size);
  fair_lockup(&(H->Lock[hashval]));           /* enter critical section */
  item = H->Head[hashval];
  while(item!=NULL) {
    if((*equals)(item->data,entry)) break; 
    item=item->next;
  }
  if(item == NULL) ret = NULL;
  else ret = item->data;
  fair_unlock(&(H->Lock[hashval]));           /* leave critical section */
  return ret;
}


void *hashtableExtract (hashtable *H, void *entry,
			int (*equals)(void *, void *) ) 
{ /************************************/
  /*          Search & Remove Entry   */
  /************************************/
  HashTableItem *item,*item_old;
  void *ret;
  int found=0,
      hashval;

  hashval = (*(H->hashFn))(entry) % (H->Size);
  fair_lockup(&(H->Lock[hashval]));     /* enter critical section */
  item = H->Head[hashval];
  
  if(item==NULL) {                   /* if list is empty */
    
    fair_unlock(&(H->Lock[hashval]));   /* leave c.s. (1)         */
    return NULL;
  }            
  else if ((*equals)(item->data,entry)) { /* if first-in-list matches */
    H->Head[hashval] = item->next;
    fair_unlock(&(H->Lock[hashval]));   /* leave c.s. (2)         */
    ret = item->data;
    free(item);
    H->NumElts--;
    return ret;
  }
  else {
    item_old = item;
    item = item->next;
    while((!found)&&(item!=NULL)) {
      if((*equals)(item->data,entry)) found=1;
      else {
	item_old = item;
	item = item->next;
      }
    }
    if(found) {
      item_old->next = item->next;
      fair_unlock(&(H->Lock[hashval])); /* leave c.s. (3)         */
      ret = item->data;
      free(item);
      H->NumElts--;
      return ret;
    }
    else {
      fair_unlock(&(H->Lock[hashval])); /* leave c.s. (4)         */
      return NULL;
    }
  }
}


int hashtableSize(hashtable *H) {
  /************************************/
  /*      return actual size of ht    */
  /************************************/
  return H->Size;
}


int hashtableNumElts(hashtable *H) {
  /************************************/
  /*    return actual # of elements   */
  /************************************/
  return H->NumElts;
}


