/*************************************************/
/*                                               */
/*      RTree.c                                  */
/*      random search tree implementation        */
/*	in fork95                                */ 
/*      v0.22                                    */
/*      29.06.1999 Christoph W. Kessler &        */
/*	           Oliver Fritzen                */
/*                                               */
/*      preliminary final(?) version             */
/*                                               */
/*                                               */
/*      for later debugging I just commented out */
/*      the output routines                      */
/*                                               */
/*************************************************/

#include <io.h> 
#include <fork.h>
#include "RTree.h"

 async void outputRec(node n,int d) { 
   int i; 
   if(n == NULL) return; 
   outputRec(n->left,d+1); 
   if(n->isAlive) { 
     printf("%20s %3d/%3d ",n->key,n->alive,n->size); 
   } 
   else { 
     printf("           *deleted* %3d/%3d ",n->alive,n->size);   
   } 
   for(i=0; i<d; i++) prS("-+"); 
   prS("\n"); 
   outputRec(n->right,d+1); 
 } 

 async void output(RTree T) { 
   char *s1,*s2; 
   pprintf("size = %3d | alive = %3d | depth = %2d\n", 
 	  T->size,      T->alive, T->depth); 
   outputRec(T->root,0); 
   prS("\n"); 
 } 

async RTree new_RTree(int (*cmp_keys)(void *,void *)) {
  RTree T = (RTree) shmalloc(sizeof(struct tree));
  T->size = 0;
  T->depth = 0;
  T->alive = 0;
  T->root = NULL;
  T->rootlock = new_RWLock();
  T->compare = cmp_keys;
  return T;
}

async node new_node(void *key, void *inf) {
  node n = (node) shmalloc(sizeof(struct treenode));
  n->size = 1;
  n->key = key;
  n->inf = inf;
  n->left = NULL;
  n->right = NULL;
  n->isAlive = 1;
  n->alive = 1;
  n->leftlock = new_RWLock();
  n->rightlock = new_RWLock();
  return n;
}

async void KillRec (node n) {
  if(n->left) KillRec(n->left);
  if(n->right) KillRec(n->right);
  shfree(n);
}

async void RTreeKill (RTree T) {
  KillRec(T->root);
  shfree(T);
}

async int RTreeAlives(RTree T) {
  if(T->root) return T->root->alive;
  return T->alive;
}

async int RTreeGetSize(RTree T) {
  return T->size;
}

async void *RTreeLookup(RTree T, void *key) {
  int ev;
  void *ret;
  rw_lock *rwl,*rwl_old;
  
  node n = T->root;
  rwl = T->rootlock;
  ret = NULL;
  
  rw_lockup(rwl,RW_READ);
  while(n) {
    rwl_old = rwl;
    ev = T->compare(key,n->key);
    if((ev == 0)&&(n->isAlive)) {
      ret = n->inf;
      break;
    }
    if (ev > 0) {
      rw_lockup(n->rightlock,RW_READ);
      rwl = n->rightlock;
      n = n->right;
      rw_unlock(rwl_old,RW_READ,0);
    }
    else {
      rw_lockup(n->leftlock,RW_READ);
      rwl = n->leftlock;
      n = n->left;
      rw_unlock(rwl_old,RW_READ,0);
    } 
  }
  rw_unlock(rwl,RW_READ,0);
  return ret;
}  


async int InsertRec(RTree T, node n, void *key, void *inf, 
		    rw_lock *rwl,int depth) {
  /* return values:                                         */
  /*                                                        */
  /* ret == 0 ---> inserted into new node, no problem       */
  /* ret == 1 ---> problem appeared switching from readlock */
  /*               to writelock (non-atomary!),repeat op    */

  int ret = 1;
  node temp;
  rw_lock *templock;
  int ev = (T->compare)(key,n->key);
  if(ev > 0) {           
    /* for the right branch ... */     
    rw_lockup(n->rightlock,RW_READ);
    if(n->right) {      /* still haven't found what I'm looking for...*/
      temp = n->right;
      templock = n->rightlock;
      rw_unlock(rwl,RW_READ,0);
      ret = InsertRec(T,temp,key,inf,templock,depth+1);
    }
    else {              /* found leaf ! */
      node n_new;
      rw_unlock(n->rightlock,RW_READ,0); /* switch from reader lock...      */
      rw_lockup(n->rightlock,RW_WRITE);  /* ...to writer lock (not atomic!) */
      if(n->right) {    /* problem: somebody rushed into the tiny gap! */
	rw_unlock(n->rightlock,RW_WRITE,0);
	rw_unlock(rwl,RW_READ,0);
	ret = 1;       /* return error code */
      }
      else {
	n_new = new_node(key,inf);
	n->right = n_new;
	rw_unlock(n->rightlock,RW_WRITE,0);
	rw_unlock(rwl,RW_READ,0);
	ret = 0;
      }      
    }
  }
  else {
    /* and for the left branch */
    rw_lockup(n->leftlock,RW_READ);
    if(n->left) {      /* still haven't found what I'm looking for...*/
      temp = n->left;
      templock = n->leftlock;
      rw_unlock(rwl,RW_READ,0);
      ret = InsertRec(T,temp,key,inf,templock,depth+1);
    }
    else {              /* found leaf ! */
      node n_new;
      rw_unlock(n->leftlock,RW_READ,0); /* switch from reader lock...      */
      rw_lockup(n->leftlock,RW_WRITE);  /* ...to writer lock (not atomic!) */ 
      
      if(n->left) {    /* problem: somebody rushed into the tiny gap! */
	rw_unlock(n->leftlock,RW_WRITE,0);
	rw_unlock(rwl,RW_READ,0);
	ret = 1;       /* return error code */
      }
      else {
	n_new = new_node(key,inf);
	n->left = n_new;
	 
	rw_unlock(n->leftlock,RW_WRITE,0);
	rw_unlock(rwl,RW_READ,0);
	ret = 0;
      }      
    }
  }
  if(ret == 0) {
    syncadd(&(n->size),1);
    syncadd(&(n->alive),1);
    syncmax(&(T->depth),depth);
  }
  return ret;
}

async void RTreeInsert(RTree T, void *key, void *inf) {
  int ret;
  syncadd(&(T->size),1);
  syncadd(&(T->alive),1);
  rw_lockup(T->rootlock,RW_READ);
  if(!T->root) {
    rw_unlock(T->rootlock,RW_READ,0);
    rw_lockup(T->rootlock,RW_WRITE);
    if(T->root) {
      rw_unlock(T->rootlock,RW_WRITE,0);
      rw_lockup(T->rootlock,RW_READ);
      ret = InsertRec(T,T->root,key,inf,T->rootlock,1);
    }
    else {
      T->root = new_node(key,inf);
      rw_unlock(T->rootlock,RW_WRITE,0);
    }
  }
  else {
    ret = InsertRec(T,T->root,key,inf,T->rootlock,1);
    while (ret == 1) {
      rw_lockup(T->rootlock,RW_READ);
      ret = InsertRec(T,T->root,key,inf,T->rootlock,1);
    }
  }
}



async int SingleInsertRec(RTree T, node n, rw_lock *rwl, 
                          void *key,void *inf, int depth) {
  
  /**********************************************************/
  /*                         WARNING!                       */
  /*                                                        */
  /* THIS OPERATION IS NOT YET IMPLEMENTED. THE CODE BELOW  */
  /* IS EXTREMELY BUGGY AND WON'T WORK!!!                   */
  /* (If you wan't to implement a SingleInsert function,    */
  /* I recommend you to rewrite it completely. The given    */
  /* fragments below won't be very useful...)               */
  /*                                                        */
  /**********************************************************/
    




  /* return values:                                         */
  /*                                                        */
  /* ret == 0 ---> inserted into new node                   */
  /* ret == 1 ---> problem appeared switching from readlock */
  /*               to writelock (non-atomary!),repeat op    */
  /* ret == 2 ---> overwrote alive node with same key value */
  /* ret == 3 ---> overwrote dead node with same key value  */
     
  int ret = 1;
  void *temp;
  rw_lock *templock;
  int ev = (T->compare)(key,n->key);
  if(ev > 0) {
    /* right branch */
    pprintf("right\n");
    rw_lockup((templock = n->rightlock),RW_READ);
    if(temp = n->right) {
      rw_unlock(rwl,RW_READ,0);
      SingleInsertRec(T, temp, templock, key, inf, depth + 1);
    }
    else {
      rw_unlock(templock,RW_READ,0);
      rw_lockup(templock,RW_WRITE);
      if(temp == n->right) {
        n->right = new_node(key,inf);
        rw_unlock(rwl,RW_READ,0);
      }
      else {
        rw_unlock(templock,RW_WRITE,0);
        rw_lockup(templock,RW_READ);
        SingleInsertRec(T, temp, templock, key, inf, depth + 1);
      }
    }
  }
  else if(ev < 0) {
    /* left branch */
    pprintf("left\n");
    rw_lockup((templock = n->leftlock),RW_READ);
    if(temp = n->left) {
      rw_unlock(rwl,RW_READ,0);
      SingleInsertRec(T, temp, templock, key, inf, depth + 1);
    }
    else {
      rw_unlock(templock,RW_READ,0);
      rw_lockup(templock,RW_WRITE);
      if(temp == n->left) {
        n->left = new_node(key,inf);
        rw_unlock(rwl,RW_READ,0);
      }
      else {
        rw_unlock(templock,RW_WRITE,0);
        rw_lockup(templock,RW_READ);
        rw_unlock(rwl,RW_READ,0);
        SingleInsertRec(T, temp, templock, key, inf, depth + 1);
      }
    }
  }
  else {
    /* node found, replacing... */
    pprintf("replacing\n");
    rw_unlock(rwl,RW_READ,0);
    rw_lockup(rwl,RW_WRITE);
    if((ev =(T->compare)(key,n->key))==0) {
      n->key = key;
      n->inf = inf;
      rw_unlock(rwl,RW_WRITE,0);
    }
    else {
      rw_unlock(rwl,RW_WRITE,0);
      rw_lockup(rwl,RW_READ);
      SingleInsertRec(T, n, rwl, key, inf, depth);
    }
  }
  return ret;
}
    
async void RTreeSingleInsert(RTree T, void *key, void *inf) {

  /**********************************************************/
  /*                         WARNING!                       */
  /*                                                        */
  /* THIS OPERATION IS NOT YET IMPLEMENTED. THE CODE BELOW  */
  /* IS EXTREMELY BUGGY AND WON'T WORK!!!                   */
  /* (If you wan't to implement a SingleInsert function,    */
  /* I recommend you to rewrite it completely. The given    */
  /* fragments below won't be very useful...)               */
  /*                                                        */
  /**********************************************************/
    
  int ret;
  rw_lockup(T->rootlock,RW_READ);
  if(T->root) {
    ret = SingleInsertRec(T,T->root,T->rootlock,key,inf,0);
  }
  else {
    rw_unlock(T->rootlock,RW_READ,0);
    rw_lockup(T->rootlock,RW_WRITE);
    if(T->root) {
      rw_unlock(T->rootlock,RW_WRITE,0);
      rw_lockup(T->rootlock,RW_READ);
      ret = SingleInsertRec(T,T->root,T->rootlock,key,inf,0);
    }
    else {
      T->root = new_node(key,inf);
      rw_unlock(T->rootlock,RW_WRITE,0);
    }
  }
}
  
async int DeleteRec(RTree T, node n, rw_lock *rwl, void *key) {
  int ret = 1;
  node temp;
  rw_lock *templock;
  int ev = (T->compare)(key,n->key);
  if(ev > 0){           
    /* for the right branch ... */
    rw_lockup(n->rightlock,RW_READ);
    if(temp = n->right) {
      templock = n->rightlock;
      rw_unlock(rwl, RW_READ, 0);
      ret = DeleteRec(T, temp, templock, key);
    }
    else {
      rw_unlock(rwl,RW_READ,0);
      rw_unlock(templock,RW_READ,0);
      return 2;
    }
  }
  else if(ev < 0) {
    /* for the left branch ... */
    rw_lockup(n->leftlock,RW_READ);
    if(temp = n->left) {
      templock = n->leftlock;
      rw_unlock(rwl, RW_READ, 0);
      ret = DeleteRec(T, temp, templock, key);
    }
    else {
      rw_unlock(rwl,RW_READ,0);
      rw_unlock(templock,RW_READ,0);
      return 2;
    }
  }
  else if(n->isAlive) {
    /* node found, deleting... */
    rw_unlock(rwl,RW_READ,0);
    rw_lockup(rwl,RW_WRITE);
    if((T->compare)(key,n->key)) {
      rw_unlock(rwl,RW_WRITE,0);
      return 1;
    }
    else {
      n->isAlive = 0;
      ret = 0;
      rw_unlock(rwl,RW_WRITE,0);   
    }
  }
  else {
    rw_unlock(rwl,RW_READ,0);
    return 2;
  }
  if(ret == 0) syncadd(&(n->alive),-1);
  return ret;
}


async void RTreeDelete(RTree T, void *key) {
  int ret = 1;
  {
    rw_lockup(T->rootlock,RW_READ);
    if(! T->root) {
      ret = 2;
      rw_unlock(T->rootlock,RW_READ,0);
    }
    else ret = DeleteRec(T,T->root,T->rootlock,key);
  } while(ret == 1);
  if(ret == 0) {
    syncadd(&(T->alive),-1);
  }
}



async void seqBuildArrayRec(node n, node *array, int offset) {
  /* Builds array which contains all nodes still alive */
  if(!n) return;
  if(n->left) {
    seqBuildArrayRec(n->left, array, offset);
    if(n->isAlive) {
      array[offset + n->left->alive] = n;
      if(n->right) { 
	seqBuildArrayRec(n->right, array, offset + n->left->alive + 1);
      }
    }
    else {
      seqBuildArrayRec(n->right, array, offset + n->left->alive);
      shfree(n);
    }
  }
  else {
    if(n->isAlive) {
      array[offset] = n;
      if(n->right) seqBuildArrayRec(n->right,array, offset + 1);
    }
    else {
      if(n->right) seqBuildArrayRec(n->right,array, offset);
      shfree(n);
    }
  }   
}

sync void BuildArrayRec(sh node n, node *array, int offset) {
  /* Like SeqBuildArrayRec, just with multiple processors */
  sh int p = groupsize();
  if(p == 1) farm seqBuildArrayRec(n,array,offset);
  else {
    if((n->left) && (n->right)) {    /* node is father of two */
      if(n->isAlive) {
	array[offset + n->left->alive] = n;
	fork(2; @ = $%2; $ = $/2) {
	  if(@) BuildArrayRec(n->left, array, offset);
	  else BuildArrayRec(n->right, array, offset + n->left->alive + 1);
	}
      }
      else {
	fork(2; @ = $%2; $ = $/2) {
	  if(@) BuildArrayRec(n->left, array, offset);
	  else BuildArrayRec(n->right, array, offset + n->left->alive);
	}
      }
    }
    else if(n->left) {              /* node has left son only */
      BuildArrayRec(n->left, array, offset);
      if(n->isAlive) {
	array[offset + n->left->alive] = n;
      }
    }
    else if(n->right) {             /* node has right son only */
      if(n->isAlive) {
	array[offset] = n;
	BuildArrayRec(n->right, array, offset+1);
      }
      else {
	BuildArrayRec(n->right, array, offset);
      }
    }
    else {                          /* node is leaf */
      if(n->isAlive) array[offset] = n;
    }
  }
}


async node seqRebalRec(RTree T, node *array, int from, int to, int depth) {
  /* uses nodes from array to rebuild balanced tree */
  node ret;  
  syncmax(&(T->depth),depth);  /* recalculate depth of tree */
  
  if(from == to) {    /* recursion end */
    ret = array[from];
    ret->left = NULL;
    ret->right = NULL;
    ret->alive = ret->size = 1;
    return ret;
  }
  else {
    int size = 1;
    int med = (from + to)/2;
    ret = array[med];
    if(med > from) {
      ret->left = (node) seqRebalRec(T, array, from, med-1,depth + 1);
      size += ret->left->size;
    }
    else ret->left = NULL;
    if(med < to) {
      ret->right = (node) seqRebalRec(T, array, med+1, to, depth + 1);
      size += ret->right->size;
    }
    else ret->right = NULL;
    ret->size = ret->alive = size;
    return ret;
  }
}

sync node RebalRec(RTree T, sh node *array, sh int from, sh int to, sh int depth) {
  /* Just like SeqRebalRec, only with multiple processors */
  node ret;
  sh int p = groupsize();
  if(from > to) return NULL;
  seq syncmax(&(T->depth),depth); /* recalculate depth of tree */
  if(p == 1) {
    seq ret = seqRebalRec(T, array, from, to, depth);
  }
  else {
    if(from == to) {
      ret = array[from];
      ret->left = NULL;
      ret->right = NULL;
      ret->alive = ret->size = 1;
      return ret;
    }
    else {
      sh int med;
      med = (to + from)/2;      
      ret = array[med];
      ret->size = 1;
      fork(2; @=$%2; $=$/2) {   /* split up into left and right branch */
	if(@) {
	  ret->left = RebalRec(T, array, from, med - 1, depth + 1);	  
	  if(ret->left) seq syncadd(&(ret->size),ret->left->size);
	}
	else {
	  ret->right = RebalRec(T, array, med + 1, to, depth + 1);
	  if(ret->right) seq syncadd(&(ret->size),ret->right->size);
	}
      }
    }
  }
  seq ret->alive = ret->size;
  return ret; 
}


sync void RTreeRebal(sh RTree T) {
   /* writes nodes that are alive into array,   */
  /*  and builds new balanced RTree from array */
  sh node *array;
  sh int last;
  if(T->root) seq last = T->root->alive - 1;
  else return;
  T->depth = 0;
  seq array = (node *) shmalloc(T->root->alive * sizeof(struct treenode));
  BuildArrayRec(T->root, array, 0);
  T->root = (node) RebalRec(T, array, 0, last, 0);
  T->size = T->alive;
  seq shfree(array);
}























