// trv: TRACEFILE VISUALIZER FOR FORK95 TRACEFILES
// (c) 1999   C. Kessler

#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#define min(a,b) ((a)<=(b)?(a):(b))
#define max(a,b) ((a)>=(b)?(a):(b))

// input: a Fork95 tracefile generated by writeTraceFile("name","title")
// output: an xfig source file for a graphical visualization of the input

// These defines must be the same as in /include/fork.h:
#define TRACE_END 0
#define TRACE_SPLIT 1
#define TRACE_GROUP 2
#define TRACE_BARRIER_ENTRY 3
#define TRACE_BARRIER_EXIT 4
#define TRACE_LOCKUP_WAIT 5
#define TRACE_LOCKUP_DONE 6
 
int shms;
#define MHz 28.0        /* SBPRAM prototype */
#define vPs 32.0
#define NCOLORS 192   // user-defined colors in xfig color table

writePrologue( FILE *fp )
{
 fprintf(fp,"#FIG 3.2\nLandscape\nCenter\nInches\nLetter\n100.00\nSingle\n-2\n1200 2\n");
}

writeColorTable( FILE *fp )
{
 int i;
 for (i=0; i<NCOLORS; i++)
   fprintf(fp,"0 %d #%2.2x%2.2x%2.2x\n", i+32, 
           (i<NCOLORS/2)? 32+NCOLORS/2-i/2: 32+i/3, // red
           5*i/6+32,                                // green
           255-3*i/4 );                             // blue
}

inline int color( g ) {
  int c;
  if (g >= shms) //  heap location in joins:
     return 32+NCOLORS-1;  // g = 2*shms - g;
  c = 32 + (int)((float)NCOLORS*((float)(g)/(float)shms));
  if (c<32 || c>=32+NCOLORS)
     printf("Error: unknown color %d\n", c );
  return c;
}

int main ( int argc, char **argv )
{
 FILE *fp, *out;
 char *title, *filename, *outfilename;
 int nprocs, t0, tf, firstgps;
 int shloads, shstores, shmpadds, shmpmaxs, shmpands, shmpors;
 int *shload, *shstore, *mpadd, *mpmax, *mpand, *mpor;  // local counters
 int *lastevent_offs, *lastevent_color;
 int *nbarriers, // number of barriers executed in trace period
     *tbarriers; // accumulated time each proc. spends on barrier wait
 int *lastbarriertime; // start of last barrier event of each processor
 int *nlockups, // number of lockups executed in trace period
     *tlockups; // accumulated time each proc. spends on lockup wait
 int *lastlockuptime; // start of last lockup event of each processor
 int xmax = 13000, xoff = 600,
     ymax,         yoff = 450;
 int rborder = 50, bborder = 50;
 int dropP = 40; // Pi moved downwards from graphics line y coordinate
 int ycorr = 30; // move upwards the inbetween-text by that many points
 int tdist = xmax - xoff - rborder;   // horizontal space for bars
 int inbetw = 180;     // vertical space between processor bars
 int width = 240;      // width of a processor bar 
 int evpid, ev, pid, t, gps, i;
 int xli, xre, yli, yre;
 float ftdist;
#define Xoffset( tt ) ((int)(tdist*(float)((tt)-t0) / ftdist) + xoff )
#define Yoffset( pp ) ((pp)*(width+inbetw) + yoff)
 
 if (argc<2) filename = "trace";
 else        filename = argv[1];
 if (!(fp = fopen( filename, "r" ))) {
   printf("Error: cannot open file %s\n", filename);
   exit(1);
 }
 outfilename = strcat( filename, ".fig" );
 if (!(out = fopen( outfilename, "w" ))) {
   printf("Error: cannot open file %s\n", outfilename);
   exit(1);
 }
 fscanf( fp, "%d %x %x %x %x %d %d %d %d %d %d \n",
        &nprocs, &t0, &tf, &shms, &firstgps, 
        &shloads, &shstores, &shmpadds, &shmpmaxs, &shmpands, &shmpors );
 title = (char *)malloc(100);
 if (!fgets( title, 100, fp ))
    printf("\nWarning: error in reading title\n");
 printf("\nCreating xfig source %s from tracefile %s\n", outfilename, filename );
 printf("nprocs=%d starttime=%d stoptime=%d shmemsize=%d firstgps=%d title=%s\n",
         nprocs, t0, tf, shms, firstgps, title );
 if (nprocs <= 0) { printf("Error: wrong number of processors\n"); exit(1); }
 if (shms<=1) { printf("Error: wrong shared memory size\n"); exit(1); }
 if (t0>=tf) { printf("Error: starttime must precede stoptime\n"); exit(1); }
 ftdist = (float)(tf - t0);

 writePrologue( out );
 writeColorTable( out );

 // draw black(7) rectangle:
 ymax = Yoffset(nprocs) + bborder;
 fprintf(out,
     "2 2 0 1 0 7 0 0 -1 0.000 0 0 -1 0 0 5\n\
         0 0 %d 0 %d %d 0 %d 0 0\n", xmax, xmax, ymax, ymax );

 // read statistics:
 shstore = (int *)malloc( nprocs * sizeof(int));
 shload  = (int *)malloc( nprocs * sizeof(int));
 mpadd = (int *)malloc( nprocs * sizeof(int));
 mpmax = (int *)malloc( nprocs * sizeof(int));
 mpand = (int *)malloc( nprocs * sizeof(int));
 mpor = (int *)malloc( nprocs * sizeof(int));
 for (i=0; i<nprocs; i++)
    fscanf( fp, "%x %x %x %x %x %x\n",
            shstore+i, shload + i, mpadd+i, mpmax+i, mpand+i, mpor+i );

 // initialize lastevent_color, lastevent_offs:
 lastevent_offs = (int *)malloc( nprocs * sizeof(int) );
 for (i=0; i<nprocs; i++)
    lastevent_offs[i] = Xoffset(t0);
 lastevent_color = (int *)malloc( nprocs * sizeof(int) );
 for (i=0; i<nprocs; i++)
    lastevent_color[i] = color( firstgps );
 nbarriers = (int *)malloc( nprocs * sizeof(int) );
 tbarriers = (int *)malloc( nprocs * sizeof(int) );
 lastbarriertime = (int *)malloc( nprocs * sizeof(int) );
 nlockups = (int *)malloc( nprocs * sizeof(int) );
 tlockups = (int *)malloc( nprocs * sizeof(int) );
 lastlockuptime = (int *)malloc( nprocs * sizeof(int) );
 for (i=0; i<nprocs; i++) {
    nbarriers[i] = tbarriers[i] = lastbarriertime[i] =
    nlockups[i] = tlockups[i] = lastlockuptime[i] = 0;
 }
 printf("reading events...\n");
 while (!feof(fp)) {
   int k = fscanf( fp, "%x %x %x\n", &evpid, &t, &gps );
   //char *evpidstr = malloc(10);
   //char *tstr = malloc(10);
   //char *gpsstr = malloc(10);
   //int k = fscanf( fp, "%s %s %s\n", evpidstr, tstr, gpsstr );
   if (k<3) break;
   //evpid = strtol( evpidstr, NULL, 16 );
   //t = strtol( tstr, NULL, 16 );
   //gps = strtol( gpsstr, NULL, 16 );
   ev = evpid >> 16;     // event type
   if (ev==TRACE_END) break;   // end marker found
   if (ev<0 || ev>7) {
      printf("Error: wrong event ID: %d (was: %d)\n", ev, evpid );
      exit(1);
   }
   pid = evpid & 0xffff; // processor ID
   if (pid >= nprocs || pid < 0 ) {
      printf("Error: wrong processor ID: %d\n", pid );
      exit(1);
   }
   xli = lastevent_offs[pid];      xre = Xoffset( t );
   yli = Yoffset(pid);             yre = yli + width;
   fprintf( out,
       "2 2 0 1 %d %d 0 0 20 0.000 0 0 -1 0 0 5\n\
         %d %d %d %d %d %d %d %d %d %d\n",
         lastevent_color[pid], lastevent_color[pid],
         xli, yli, xre, yli, xre, yre, xli, yre, xli, yli );
   lastevent_offs[pid] = xre;
   if (ev==TRACE_BARRIER_ENTRY) {
      lastbarriertime[pid] = t;
      nbarriers[pid] ++;
   }
   if (ev==TRACE_BARRIER_EXIT)
      if (lastbarriertime[pid] > 0) { // ignore the exitbarrier of the leader
                                      // processor that executed startTracing
         tbarriers[pid] += (t - lastbarriertime[pid]);
      }
   if (ev==TRACE_LOCKUP_WAIT) {
      lastlockuptime[pid] = t;
      nlockups[pid] ++;
   }
   if (ev==TRACE_LOCKUP_DONE)
      if (lastlockuptime[pid] > 0) { // ignore the exitbarrier of the leader
                                     // processor that executed startTracing
         tlockups[pid] += (t - lastlockuptime[pid]);
      }
   if (ev==TRACE_GROUP || ev==TRACE_BARRIER_EXIT || ev==TRACE_LOCKUP_DONE)
      lastevent_color[pid] = color( gps );
   else
   if (ev==TRACE_BARRIER_ENTRY)
      lastevent_color[pid] = 4;    // red
   else
   if (ev==TRACE_LOCKUP_WAIT)
      lastevent_color[pid] = 6;    // yellow
   else
   if (ev==TRACE_SPLIT)
      lastevent_color[pid] = 0;    // black
   else 
      lastevent_color[pid] = 7;    // white for unknown events
   
   //debug: printf("Event: %d %d %d %d => x=%d\n", ev, pid, t, gps, xre );
 }
 fclose(fp);
 // finalize for each processor:
 for (i=0; i<nprocs; i++) {
   xli = lastevent_offs[i];      xre = Xoffset( tf );
   yli = Yoffset(i);             yre = yli + width;
   fprintf( out, "2 2 0 1 %d %d 0 0 20 0.000 0 0 -1 0 0 5\n\
         %d %d %d %d %d %d %d %d %d %d\n",
         lastevent_color[i], lastevent_color[i],
         xli, yli, xre, yli, xre, yre, xli, yre, xli, yli );
 }
 // draw title
 fprintf(out,"4 0 0 0 0 18 20 0.0000 4 225 825 630 360 %s\\001\n", title );
 fprintf(out,"4 0 0 0 0 22 14 0.0000 4 225 825 6800 340 traced time period: %d msecs\\001\n", 
            (int)(ftdist / MHz / 1000.0 * vPs) );
 fprintf(out,"4 0 0 0 0 20 12 0.0000 4 225 825 9630 200 %d sh-loads,    %d sh-stores\\001\n", 
           shloads, shstores );
 fprintf(out,"4 0 0 0 0 20 12 0.0000 4 225 825 9630 370 %d mpadd,\
    %d mpmax,    %d mpand,     %d mpor\\001\n", 
           shmpadds, shmpmaxs, shmpands, shmpors );
 // draw processors:
 for (i=0; i<nprocs; i++)
    fprintf(out,"4 0 0 0 0 22 16 0.0000 4 100 440 75 %d P%d\\001\n",
                Yoffset(i) + width + dropP, i );
 for (i=0; i<nprocs; i++) {
    // draw barrier statistics:
    fprintf(out,"4 0 0 0 0 20 10 0.0000 4 100 440 %d %d %d barriers,\
   %6d msecs = %2.1f%% spent spinning on barriers\\001\n",
                xoff, Yoffset(i) + width + inbetw - ycorr,
                nbarriers[i], (int)( (float)(tbarriers[i])/MHz/1000.0*vPs),
                (float)(tbarriers[i]) / ftdist * 100.0 );
    // draw lockup statistics:
    fprintf(out,"4 0 0 0 0 20 10 0.0000 4 100 440 %d %d %4d lockups,\
   %6d msecs = %2.1f%% spent spinning on locks\\001\n",
                xoff+3900, Yoffset(i) + width + inbetw - ycorr,
                nlockups[i], (int)( (float)(tlockups[i])/MHz/1000.0*vPs),
                (float)(tlockups[i]) / ftdist * 100.0 );
    // draw shared memory access statistics:
    fprintf(out,"4 0 0 0 0 20 10 0.0000 4 100 440 %d %d %d sh loads,\
  %d sh stores,   %d mpadd,  %d mpmax,  %d mpand,  %d mpor\\001\n",
                xoff+7700, Yoffset(i) + width + inbetw - ycorr, 
                shload[i], shstore[i], mpadd[i], mpmax[i], mpand[i], mpor[i] );
 }
 fclose(out);
 return 0;
}
