#include <fork.h>
#include <io.h>
#include <assert.h>


sync int *weightedGroupVector( sh int p, sh int n, sh float *work )
{
  // work is an array of length n holding positive float values.
  // n < p.
  // Compute an int array ret[0:p-1] holding group indices ret[i]
  // in the range 0,...,n
  // with n_j := sum_{i, ret[i]==c_j} , j=0,...,n-1, such that
  // (1) n_j > 0 iff work[j] > 0
  // (2) n_j / \sum_{k} n_k approximately equal to work[j].

  // Note that this problem is related to the distribution
  // of p seats in a parliament among n parties with votes for
  // party i being stored in work[i], where each party with at
  // least one vote is guaranteed to have a seat in parliament.
 
  // We use a method adapted from that by Hare-Niemeyer.

  sh int *ret;
  int i, j;
  sh int nprocs = groupsize();
  sh float sum = 0.0;
#define MAXSEATS 100
  sh int seats[MAXSEATS]; 
  sh int pre[MAXSEATS+1]; 
  sh int sumseats = 0;

  seq ret = (int *)shmalloc( p * sizeof( int ));
  // first: compute the sum of all work[] entries
  seq
     for (i=0; i<n; i++) {
        assert( work[i] >= 0.0 );
        sum += work[i];
     }
  // seq printf("work sum: %f\n", sum );
  
  // now, each party gets at most one seat more than it
  // should have; each party with nonzero vote gets at least one:
 
  farm
   for (i=$; i<n; i+=nprocs)
     if (work[i] > 0.0) {
        seats[i] = (int) ( (float)p * work[i] / sum ) + 1;
        //pprintf("first seats[%d] = %d\n", i, seats[i]);
        syncadd( &sumseats, seats[i] );
     }
     else
        seats[i] = 0;
  
  // now sumseats is in the range p...p+N, where N = #nonzero parties
  // correct the result by taking away up to N seats:
  //seq assert( sumseats >= p );
  //seq pprintf("sumseats: %d\n", sumseats); 
  //seq assert( sumseats <= p+n );
  seq
    for (i=0; i<n; i++) {
      if (sumseats == p) break;
      if( seats[i]<=1 ) continue; // can't take away a seat from a nonzero party
      sumseats --;
      seats[i] --;
    }
  seq assert( sumseats == p );
  
  // now seats are allocated. Compute prefix vector in parallel:
  sumseats = 0;
  farm
    for (i=$; i<n; i+=nprocs)
      pre[i] = mpadd( &sumseats, seats[i] );
  pre[n] = sumseats;
  seq assert( sumseats == p );
  //seq for (i=0; i<n+1; i++)  printf("pre[%d] = %d\n", i, pre[i]);

  // From the prefix vector, compute the result vector 
  // (seat assignment) in parallel:
  for (i=$; i<n; i+=nprocs)  // each party "numbers" its seats:
     for (j=pre[i]; j<pre[i+1]; j++)
        ret[j] = i;

  return ret;
}




void main( void )
{
 int i;

 start {
   sh float a[80];
   sh int p = 9;
   sh int n = 8;   /* p > n */
   sh int *b;
   sh float sum = 0.0;
   seq for (i=0; i<n; i++) { a[n-i-1] = (double) i*i; }
   seq for (i=0; i<n; i++) printf(" a[%d] = %f\n", i, a[i]);

   seq printf("\nSeat assignment:\n");

   b = weightedGroupVector( 14, 8, a );

   seq for (i=0; i<14; i++) printf(" b[%d] = %d\n", i, b[i]);
 }
}

