/* Parallel Mergesort.    C.W. Kessler 10/97
 * sorts N elements using p processors, synchronous version.
 * Assumes that all elements are different;
 * otherwise the result will be wrong.
 */

#include <fork.h>
#include <assert.h>
#include <io.h>
#include <stdlib.h>
#include <string.h>

#define THRESHOLD 1

sh simple_lock screen = 0;  /*screen output is a critical section*/

/* print an array a of size n sequentially
 */
void print_array( int *a, int n )
{
 int i;
 simple_lockup( &screen );
  pprintf("Array %p of size %d:\n", a, n);
  for (i=0; i<n; i++) printf(" %d", a[i]);
  printf("\n");
 simple_unlock( &screen );
}


int compare( void *a, void *b) { 
 if (*(int*)a < *(int*)b) return -1;
 if (*(int*)a > *(int*)b) return 1;
 return 0;
}

void seq_sort( int *data, int n, int *presult)
{
  memcpy( presult, data, n );
  qsort( presult, n, sizeof(int), compare );
}

/* Alternative: sequential sorting, using bubble sort.
 * Assumes that outarray is allocated.
 */
void seqsort( int *array, int n, int *outarray )
{
 int i,j;
 int temp;
 for (i=0; i<n; i++) outarray[i] = array[i];
 for (i=0; i<n-1; i++)
   for (j=i+1; j<n; j++)
     if (outarray[i] > outarray[j]) {  /*swap:*/
       temp = outarray[i];
       outarray[i] = outarray[j];
       outarray[j] = temp;
     }
}


/* in sequential compute the rank of key within
 * array of size n, i.e. # array-elements < key
 */
int get_rank( int key, int *array, int n )
{
 int left = 0;
 int right = n-1;
 int mid;
 if (key > array[n-1]) return n;
 if (key == array[n-1]) return n-1;
 if (key <= array[0]) return 0;
 while (left < right-1) {   /*binary search*/
    /*always maintain array[left] <= key < array[right]*/
    mid = (right+left)/2;
    if (key < array[mid]) right = mid;
    else                  left = mid;
 }
 if (key==array[left]) return left;
 else                  return left+1;
}
 

/* merge array src1 of size n1 and src2 of size n2
 * into one array dest of size n1+n2. 
 * Assertions: p>1, n1*n2>=1, dest + temp array is allocated.
 */
sync void merge( sh int *src1, sh int n1, 
                 sh int *src2, sh int n2,
                 sh int *dest, sh int *temp)
{
 sh int p = 0;
 sh int iter;
 sh int *rank12 = temp,
        *rank21 = temp + n1;  /*temp. rank arrays*/
 int i;
 $ = mpadd( &p, 1 );    // renumber $
 farm assert(p>1);
 farm pprintf(" merge( src1=%p, n1=%d, src2=%p, n2=%d, dest=%p, n=%d, p=%d)\n",
           src1,n1,src2,n2,dest,n1+n2,p);
 iter = 0;
 farm
   /* self-scheduling par. loop over rank computations: */
   FORALL( i, &iter, 0, n1, 1 )
     rank12[i] = get_rank( src1[i], src2, n2 ); 
 iter = 0;
 farm
   FORALL( i, &iter, 0, n2, 1 )
     rank21[i] = get_rank( src2[i], src1, n1 ); 
 farm {
   /* copy elements to dest using the rank information */
   forall (i,0,n1,p)  dest[i+rank12[i]] = src1[i];
   forall (i,0,n2,p)  dest[i+rank21[i]] = src2[i];
 }
}


/* mergesort for an array of size n.
 * The sorted array is to be stored in
 * sortedarray which is assumed to be allocated.
 * temp is an allocated temporary buffer of length n.
 */
sync void mergesort( sh int *array, sh int n, sh int *sortedarray, sh int *temp )
{
 sh int p = groupsize();

 seq pprintf(" mergesort( array=%p, n=%d, p=%d)\n", array,n,p);

 if (n<=THRESHOLD) {
     seq seq_sort( array, n, sortedarray );
     return;
 }
 if (p==1) {
     farm seq_sort( array, n, sortedarray );
     return;
 }
 fork (2; @=$%2; $=$/2)
    mergesort( array + @*(n/2), (1-@)*(n/2) + @*(n-n/2), temp + @*(n/2), 
               sortedarray + @*(n/2) );
 merge( temp, n/2, temp+n/2, n-n/2, sortedarray, array );
}


void main( void )
{
 int j;
 start {
   sh int N;   /* number of array elements */
   sh int *a;  /* the array to be sorted */
   sh int *b;
   sh int *temp;
   initTracing(100000);
   seq {
     printf("Enter N = ");
     scanf("%d", &N);
   }
   a = (int *) shalloc( N * sizeof(int) );
   b = (int *) shalloc( N * sizeof(int) );
   temp = (int *) shalloc( N * sizeof(int) );
   seq {
      for (j=0; j<N; j++)
        a[j] = rand()%8192; /*set array*/
   }
   seq print_array( a, N );
   startTracing();
   mergesort( a, N, b, temp ); 
   stopTracing();
   seq print_array( b, N );
   writeTraceFile( "MERGESORT", "mergesort" );
 }
 exit(0);
}
